/* Copyright (C) 1999 Aaron Lehmann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include "gtkplot.h"
#include "gtksheet.h"

#include "app.h"
#include "curvefit.h"




static void linear_regression (gdouble *x, gdouble *y, const gint n, thing *in_thing)
{
    /* Algorhythm:
             n * (E xi yi) - (E xi) * (E yi)
       m =   _____________________________
	        n * (E xi^2) - (E xi)^2


	         E (yi) - m * (E xi)
       b =   _____________________________
	                  n

       mse =    avg(ei^2)
    */


    int i;
    gdouble curr_num=0, curr_den=0, curr_spare=0;
    gdouble m, b;
    long double sqsum=0;


    for (i=0; i != n; ++i)
    {
	curr_num += x[i] * y[i];
    }

    curr_num *= n;

    for (i=0; i != n; ++i)
    {
	curr_den += x[i];
    }
    for (i=0; i != n; ++i)
    {
	curr_spare += y[i];
    }

    curr_den = curr_den * curr_spare;

    curr_num = curr_num - curr_den;

    curr_den = 0;
    curr_spare = 0;

    for (i=0; i != n; ++i)
    {
	curr_den += x[i] * x[i];
    }

    curr_den *= n;

    for (i=0; i != n; ++i)
    {
	curr_spare += x[i];
    }

    curr_spare = curr_spare * curr_spare;
    curr_den = curr_den - curr_spare;

    m = curr_num / curr_den;


    /* Found m. Now get b */

    curr_num = 0;
    curr_spare = 0;

    for (i=0; i != n; ++i)
    {
	curr_num += y[i];
    }
    for (i=0; i != n; ++i)
    {
	curr_spare += x[i];
    }

    curr_spare *= m;

    curr_num = curr_num - curr_spare;

    b = curr_num / n;

    /* Now compute MSE (mean squared error) */
    
    for (i=0; i != n; ++i)
    {
	sqsum += (y[i] - ((x[i] * m) + b)) * (y[i] - ((x[i] * m) + b));
    }

    sqsum /= n;

    in_thing->m = m;
    in_thing->b = b;
    in_thing->mse = sqsum;
}

gboolean curve_fit (gdouble *x, gdouble *y, const gint n, thing *in_thing)
{
    switch (in_thing->regrtype)
    {
    case REGRESSION_LINEAR:
    {
	/* y = B + M * x */
	linear_regression (x, y, n, in_thing);
	return TRUE;
    }
    case REGRESSION_LOGARITHMIC:
    {
	/* y = B + M * log(x) */
	int i;
	gdouble *lnx = g_malloc (n * sizeof(gdouble));

	for (i = 0; i != n; ++i)
	{
	    if (x[i] <= 0.0)
	    {
		g_free (lnx);
		if (in_thing->error)
		{
		    free (in_thing->error);
		}
		in_thing->error = (char *) malloc (60*sizeof(char));
		strcpy (in_thing->error, _("X values must be positive for logarithmic curve fit"));
		return FALSE;
	    }
	    lnx[i] = log(x[i]);
	}
	linear_regression (lnx, y, n, in_thing);
	g_free (lnx);
	return TRUE;
    }
    case REGRESSION_EXPONENTIAL:
    {
	/* log(y) = log(B) + M * x */
	int i;
	gdouble *lny = g_malloc (n * sizeof(gdouble));

	for (i = 0; i != n; ++i)
	{
	    if (y[i] <= 0.0)
	    {
		g_free (lny);
		if (in_thing->error)
		{
		    free (in_thing->error);
		}
		in_thing->error = (char *) malloc (60*sizeof(char));
		strcpy (in_thing->error, _("Y values must be positive for exponential curve fit"));
		return FALSE;
	    }
	    lny[i] = log(y[i]);
	}
	linear_regression (x, lny, n, in_thing);
	in_thing->b = exp(in_thing->b);
	g_free (lny);
	return TRUE;
    }
    case REGRESSION_POWER:
    {
	/* log(y) = log(B) + M * log(x) */
	int i;
	gdouble *lnx = g_malloc (n * sizeof(gdouble));
	gdouble *lny = g_malloc (n * sizeof(gdouble));

	for (i = 0; i != n; ++i)
	{
	    if ((x[i] <= 0.0) || (y[i] <= 0.0))
	    {
		g_free (lnx);
		g_free (lny);
		if (in_thing->error)
		{
		    free (in_thing->error);
		}
		in_thing->error = (char *) malloc (60*sizeof(char));
		strcpy (in_thing->error, _("X and Y values must be positive for power curve fit"));
		return FALSE;
	    }
	    lnx[i] = log(x[i]);
	    lny[i] = log(y[i]);
	}
	linear_regression (lnx, lny, n, in_thing);
	in_thing->b = exp(in_thing->b);
	g_free (lnx);
	g_free (lny);
	return TRUE;
    }
    default:
    {
	return FALSE;
    }
    }
}
