/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/* 
 * load_procs.c
 */

/*
 * load_procs.c contains routines to load dstool configuration and memory objects from a data file 
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/param.h>
#include <sys/types.h>
#include <ctype.h>

#include "pm.h"
#include "saveload_def.h"
#include "memory.h"
#include "modellib.h"
#include "constants.h"
#include "utilities.h"
#include "math_utils.h"
#include "saveload_local.h"
#include "version.h"
#include <tcl.h>
#include <tk.h>

extern Tcl_Interp *interp;

void
load_go()
{
   int status;
   int force_varb, force_param;

/* We'll call load_go_force with force_varb == force_param == FALSE
 * so that if the varb dim differs from the current one, we'll query
 * the user as to what to do 
 */
   force_varb = FALSE;
   force_param = FALSE;
   status = load_go_force(&force_varb, &force_param);
   if (status != 0) {
       fprintf(stderr,"%d == status:  Illegal data present! Load partially completed.\n", status);
       }

}

/*
 * load_go_force()	reads data from file in format determined by option
 * Returns 0 = successful load; -1 = cannot open or find file;
 *        -2 = illegal data present; -3 = cannot initialize memory;
 *	  -4 = current varb dim differs from file varb dim;
 *	  -5 = current param dim differs from file param dim;
 * Arguments: if *force_varb = FALSE then ask permission in case file varb_dim differs from current.
 * 	      Similarly with force_param.
 */
int
load_go_force(force_varb, force_param)
int	*force_varb, *force_param;
{
  FILE          *fp;
  int		status=0;
  char          *fname =  (char *) calloc(SIZE_OF_FNAME,sizeof(char)); 
  char          *dirname =  (char *) calloc(SIZE_OF_DIR_PLUS_FNAME,sizeof(char));

  dirname = strcat(strcat((char *)pm(GET, "Load.Directory", dirname, NULL), "/"), 
		   (char *)pm(GET, "Load.Filename", fname, NULL) );
  if ( check_file_to_read(dirname) )	
    {
      fp = fopen(dirname,"r");
	/* Note: Format_Flag has been inverted, so that the value of 
	 * Load.Format_Flag corresponds to the version number of DsTool 
	 * whose data format is to be used (with 0 = unformatted data).  
	 * (The DsTool 3.0 data format will be implemented in Tcl)
	 * Note that this is the reverse of the original values, where
	 * 0 <--> DsTool 2.0 ;  1 <--> DsTool 1.x ;  other <--> unformatted.
	 * 		7/3/97  BAM 
  	 */
      switch ( *((int *)pm(GET, "Load.Format_Flag", NULL))) {
          case 2: status = load_form_data(fp, force_varb, force_param);
		  break;
 	  case 1: status = load_oldform_data(fp, force_varb, force_param);
		  break;
          case 0: status = load_unform_data(fp);
		  break;
	  default:    	/* Something's seriously wrong! */
	          status = -2;
		  break;
          }

      fclose(fp);
    }
  else				/* file does not exist */
    status = -1;

  free(fname);
  free(dirname);
  return( status );
}

/*
 * load_form_data()
 *
 * Loads data from a file.   If the line starts with 'pm', the command 
 * is interpreted as a postmaster command.  If it starts with 'set', it
 * is interpreted as setting a postmaster memory object.   If it starts
 * with some other word, it is ignored. 
 *
 */
int
  load_form_data(fp, force_varb, force_param)
FILE		*fp;
int		*force_varb, *force_param; 
{
  char word[256], c;
  int kw, keep_going = 0;
  int status = 0;  /* Flag for errors  7/4/97 BAM */

  while ( (fscanf(fp,"%s",word) != EOF) && (keep_going==0) )
    {
      switch(kw = keyword(word))
	{
	case COMMENT:
	  /* skip to end of line */
	  while (( (c=getc(fp)) != '\n') && (c != EOF));
	  break;
	case POSTMASTER:
	  /* interpret as postmaster command */
	  keep_going = load_pm(fp);
 	  if (keep_going != 0) { 	/* 7/4/97 BAM */
              status = keep_going;
	      keep_going = 0;
	      }
	  break;
        case SET:
	  keep_going = load_set(fp);
          if (keep_going != 0) {	/* 7/4/97 BAM */
              status = keep_going;
              keep_going = 0;
              }                
	  break;
	default:
	  /* print error message and skip to end of line */
/*	  system_mess_proc(0,"Illegal command in input file."); */
	  fprintf(stderr,"Illegal command %s in input file.\n",word);
	  status = -2;		/* 7/4/97 BAM */
	  while (( (c=getc(fp)) != '\n') && (c != EOF));
	  break;
	}
    }

  return(status);
}

int
keyword(word)
char *word;
{
  int i;
  
  for (i=0; i< N_KEYWORD; i++)
    if (!strcmp(word,key[i].word))
      return(key[i].index);
  return(NO_KEY);
}

int
  load_pm(fp)
FILE *fp;
{
  char cmd[MAX_LABEL_LEN], str[MAX_LABEL_LEN], sdata[MAX_LONG_STR], *sdata_p, c;
  int kw, status, pmtype, idata, i, n,lower, *ildata, len;
  double ddata, *dldata;
  int length, errorstatus;
    /* 7/7/97 BAM to handle DsTool 2 format configuration stuff with Tcl */ 

  errorstatus = 0;		/* 7/4/97  BAM */

  if (fscanf(fp, "%s %s", cmd, str) != 2) return 0;
/*  printf ("pm %s %s\n", cmd, str); */
  
  pmtype = pm_type(str, NULL, NULL);
  if ((pmtype==0)&&(strstr(str,"Win") == NULL)) {
      while (( (c=getc(fp)) != '\n') && (c != EOF));
      return 0;
      }
  switch(kw = pm_keyword(cmd))
    {
    case EXEC:
/*      fprintf(stderr, "pm(EXEC, %s, NULL)\n", str);*/
	/* Try to do it if it's not a regular window configuration command */
      if ((strstr(str,"Win") == NULL)||(strstr(str,"TwoD_Win")!= NULL)) {
          pm(EXEC, str, NULL);
          } 
      break;
    case PUT:
      switch (pmtype)
	{
	case INT:
	  status = fscanf(fp, "%d", &idata);
/*	  if (status != 1) {
              errorstatus = -2;
              fprintf(stderr, "%s: error loading %s\n", PROGRAM_TITLE, str);
	      }
	  fprintf(stderr, "pm(PUT, %s, %d, NULL)\n", str, idata);*/
	/* For compatibility with previous versions 8/4/97 BAM */
          if (!(strncmp("Flow.Stopping_Condition", str, 23)))
              stopcond_convert_2to3(&idata);
          if (!(strncmp("Defaults.Recording", str, 18)))
              if (idata == 0) 
                  idata = 1;
              else 
                  idata = 0;
	  pm(PUT, str, idata, NULL);
	  break;
	case DBL:
	  status = fscanf(fp, "%lf",&ddata);
/*          if (status != 1) {
		errorstatus = -2;
                fprintf(stderr, "%s: error loading %s\n", PROGRAM_TITLE, str);
		}
	  fprintf(stderr, "pm(PUT, %s, %lf, NULL)\n", str, ddata); */
	  pm(PUT, str, ddata, NULL);
	  break;
	case STRNG:
	  fgets((char *) sdata, MAX_LONG_STR, fp);
	  sdata_p = sdata;
/*	  status = fscanf(fp, "%s", sdata);*/
	 /*  fprintf(stderr, "pm(PUT, %s, %d, NULL)\n", str, idata);  */
/*	  pm(INIT, str, strlen(sdata)+1,*/

	  while (isspace((int)*sdata_p++)) /* strip off leading whitespace */
	      ;
	  sdata_p--;

	  /* strip off leading " if present */
	  if( sdata_p[0] == '"' )
	      sdata_p++;

	  /* clear trailing newline if present */
	  if (((len = strlen(sdata_p)) >0) && 
	      (sdata_p[len-1] == '\n'))
	      sdata_p[len-1] = '\0';

	  /* strip off trailing whitespace if present */
	  while (((len = strlen(sdata_p)) >0) && 
		 isspace(sdata_p[len -1]))
	      sdata_p[len-1] = '\0';

	  /* strip trailing " if present */
	  if (((len = strlen(sdata_p)) >0) && 
	      (sdata_p[len-1] == '"'))
	      sdata_p[len-1] = '\0';
	      
          pm(PUT, str, sdata_p, NULL);
	  break;
	case ADDRS:
	case MEMRY:
	case FNCT:
	  system_mess_proc(0,"load_pm: cannot transfer this data type.");
	  break;
	case 0:		/* Could be a Win command, or something else */
          length = strlen(str);
/*		8/20/97 BAM  -- not necessary anymore 
          if (strncmp(str,"TwoD_Win.Window_Number",length) == 0) {
              fscanf(fp, "%d", &idata);
              pm(PUT, "View.Win_Num", idata, NULL);
          } else  */
          if ((strncmp(str,"Win.Current", length) == 0)||(strstr(str,"Open_Status")!=NULL)) {
		/* Ignore it */
              while (( (c=getc(fp)) != '\n') && (c != EOF));
          } else {		/* I don't know what it is... */
            /*  system_mess_proc(0,"load_pm: received an unknown pm object"); */
              fprintf(stderr,"%s: load_pm: received an unknown pm object: %s\n", PROGRAM_TITLE, str);
            /*  printf ("    %s - not recognized \n", str); */
	      while (( (c=getc(fp)) != '\n') && (c != EOF));
              }
	  break;
	default: 
          system_mess_proc(0,"load_pm: this pm type is not recognized.");
          break;
	}
      break;
  case PUT_LIST:
      switch (pmtype)
	{
	case INT_LIST:
	  status = fscanf(fp,"%d %d", &lower, &n);
/*          if (status != 2) {
		errorstatus = -2;
                fprintf(stderr, "%s: error loading %s\n", PROGRAM_TITLE, str);
		}  */
	  pm(INIT, str, n+1, NULL);
	  for (i=0; i<=n; i++) {
	      status = fscanf(fp, "%d", &idata);
           /*   if (status != 1) {
                  errorstatus = -2;
                  fprintf(stderr, "%s: error loading %s\n", PROGRAM_TITLE, str);
                  } */
	      pm(PUT, str, i, idata, NULL);
	      }
	  /* fprintf(stderr, "pm(PUT_LIST, %s, %d, [ ",str,n);
	     for(i=0; i<n; i++) fprintf(stderr,"%d ", ildata[i]);
		 fprintf(stderr, "], NULL)\n"); */
	  break;
	case DBL_LIST:
	  status = fscanf(fp, "%d %d", &lower, &n);
      /*    if (status != 2) {
              errorstatus = -2;
              fprintf(stderr, "%s: error loading %s\n", PROGRAM_TITLE, str);
              }  */
	  pm(INIT, str, n+1, NULL);
	  for (i=0; i<=n; i++)
	    {
	      status = fscanf(fp, "%lf", &ddata);
          /*    if (status != 1) {
              	  errorstatus = -2;
                  fprintf(stderr, "%s: error loading %s\n", PROGRAM_TITLE, str);
                  }  */
	      pm(PUT, str, i, ddata, NULL);
	      /* fprintf(stderr, "pm(PUT_LIST, %s, %d,[ ",str,n);
		 for(i=0; i<n; i++) fprintf(stderr,"%lf ", dldata[i]);
		     fprintf(stderr, "], NULL)\n"); */
	    }
	  break;
	case STRNG_LIST:
	  system_mess_proc(0,"load_pm: this pm type not loadable yet.");
	  break;
	default:	/* Could be window geometry or something else... */
          if (strstr(str,"Win") == NULL) {
	      system_mess_proc(0,"load_pm: this pm type is not recognized.");
 	  } else {	/* It's window geometry info */
              load_config_wingeom(fp,str);
	      }
	  break;
	}
      break;
    case INIT:
      system_mess_proc(0, "Initializing not yet allowed in input files.");
      while (( (c=getc(fp)) != '\n') && (c != EOF));
      break;
    case CLEAR:
      system_mess_proc(0, "Clearing not yet allowed in input files.");
      while (( (c=getc(fp)) != '\n') && (c != EOF));
      break;
    case CREATE_OBJ:
    case CREATE_ELEM:
      /* not implemented - skip rest of line */
      system_mess_proc(0, "Creation not yet allowed in input files.");
      while (( (c=getc(fp)) != '\n') && (c != EOF));
      break;
    case GET:
    case GET_LIST:
    default:
      /* give error msg and skip rest of line */
      system_mess_proc(0,"Illegal pm command in input file");
      while (( (c=getc(fp)) != '\n') && (c != EOF));
      break;
    }
    return(errorstatus);
}

int
  load_set(fp)
FILE *fp;
{

    char
	*pm_result,
	word[MAX_SHORT_STR],
	cur_mem_str[MAX_SHORT_STR];

    int		
	keep_going = 0;		/* value 0 means continue !! */

    pm_result = (char *)pm(GET,"Cur_Memory.Mem_Type",cur_mem_str,NULL);
/*    printf ("%s ", cur_mem_str); */

    fscanf(fp,"%s",word); /* new_object or func_object */
   /* printf ("%s\n", word); */

    if 	(!strcmp(word,"func_object"))
	keep_going = skip_function_obj (fp);
    else if (!strcmp(word,"new_object")) {
	/* temporary */
	if (!strcmp(cur_mem_str,"Traj"))
	    keep_going = fill_memory_obj(fp, "Memory.Traj", 
					 DEFAULT_TRAJ_LENGTH, TRAJ_MEMORY);
	else if (!strcmp(cur_mem_str,"Mult"))
	    keep_going = fill_memory_obj(fp, "Memory.Mult", 
					 DEFAULT_MULT_LENGTH, MULT_MEMORY);
	else if (!strcmp(cur_mem_str,"Fixed"))
	    keep_going = fill_memory_obj(fp, "Memory.Fixed", 
					 DEFAULT_FP_LENGTH, FIXPT_MEMORY);
	else if (!strcmp(cur_mem_str,"Continuation"))
	    keep_going = fill_memory_obj(fp, "Memory.Continuation", 
					 DEFAULT_CONT_LENGTH, CONT_MEMORY);
	else if (!strcmp(cur_mem_str,"Param"))
	    keep_going = fill_memory_obj(fp, "Memory.Param", 
					 DEFAULT_PARAM_LENGTH, PARAM_MEMORY);
	else if (!strcmp(cur_mem_str,"Sel_Pt"))
	    keep_going = fill_memory_obj(fp, "Memory.Sel_Pt", 
					 DEFAULT_SEL_PT_LENGTH, SEL_PT_MEMORY);
    }
    return keep_going;
	
}

/* 
   read next two words ( both of which should be a "{" ) and
   continue reading until both these brackets are closed
*/   
int
  skip_function_obj(fp)

FILE
   *fp;
{
    int
	c,
	depth = 0;		/* depth of brackets */
    char 
	word[MAX_SHORT_STR];

    fscanf(fp,"%s",word);
    if (strcmp(word,"{"))
	return MINOR_ERROR;
    else
	depth = 1;
    
    fscanf(fp,"%s",word);
    if (strcmp(word,"{"))
	return MINOR_ERROR;
    else
	depth = 2;

    while ((depth > 0) && ((c = fgetc(fp)) != EOF)) 
	if (c == '{')
	    ++depth;
	else if (c == '}')
	    --depth;

    if (c == EOF)
	return MINOR_ERROR;
    else	    
	return NO_ERROR;
}


/* 
 * load_oldform_data()	reads data from file in dstool format.
 * Arguments: if force_varb = FALSE then ask permission in case file varb_dim differs from current
 * 	      Similarly with force_param.
 * Returns error code to load_go().
 */
int
  load_oldform_data(fp, force_varb, force_param)
FILE		*fp;
int		*force_varb, *force_param; 
{
  int		n;
  int		data_v_dim, data_p_dim, data_f_dim;
  char		word[40];
  int		cur_v_dim, cur_p_dim;
  int		cur_f_dim;
  int		keep_going = 0, kw, ivalue;
  double        dvalue;
  
      while ( (fscanf(fp,"%s",word) != EOF) && (keep_going==0) )
	{
	  switch(kw = old_keyword(word))
	    {
	    case System_Name:
	      if ( (n=read_sys_name(fp)) >= 0 )
		{
		  pm(PUT, "Model.Load_Number", n,
		     EXEC, "Model.Load", 
		     NULL);
		  cur_f_dim  = *((int *)pm(GET, "Model.Funct_Dim", NULL));
		}
	      break;
	    case Varb_Dim:
	      fscanf(fp,"%d", &data_v_dim); 
	      cur_v_dim = *((int *)pm(GET, "Model.Varb_Dim", NULL));
	      if ( (data_v_dim != cur_v_dim) && !(*force_varb) )
			keep_going = -4;
	      break;
	    case Param_Dim:
	      fscanf(fp,"%d", &data_p_dim);
	      cur_p_dim = *((int *)pm(GET, "Model.Param_Dim", NULL));
	      if ( (data_p_dim != cur_p_dim) && !(*force_param) )
		      keep_going = -5;
	      break;
	    case Function_Dim:
	      fscanf(fp,"%d", &data_f_dim);
	      break;

	    case Varb_Ic:	/*  double arrays of length v_dim for traj */
	      set_pm_dvalue(fp,data_v_dim, cur_v_dim, "Selected.Varb_Ic");
	      break;
	    case Varb_Fc:
	      set_pm_dvalue(fp,data_v_dim, cur_v_dim, "Selected.Varb_Fc");
	      break;
	    case Varb_Min:
	      set_pm_dvalue(fp,data_v_dim, cur_v_dim, "Defaults.Varb_Min");
	      break;
	    case Varb_Max:
	      set_pm_dvalue(fp,data_v_dim, cur_v_dim, "Defaults.Varb_Max");
	      break;

	    case Param:		/* double arrays of length p_dim for traj */
	      set_pm_dvalue(fp,data_p_dim, cur_p_dim, "Selected.Param_Ic");
	      break;
	    case Param_Min:
	      set_pm_dvalue(fp,data_p_dim, cur_p_dim, "Defaults.Param_Min");
	      break;
	    case Param_Max:
	      set_pm_dvalue(fp,data_p_dim, cur_p_dim, "Defaults.Param_Max");
	      break;

	    case Function_Min:	 /* double arrays of length f_dim for traj*/
	      set_pm_dvalue(fp,data_f_dim, cur_f_dim, "Defaults.Funct_Min");
	      break;
	    case Function_Max:
	      set_pm_dvalue(fp,data_f_dim, cur_f_dim, "Defaults.Funct_Max");
	      break;

	    case Dups:
	      fscanf(fp, "%lg", &dvalue );
	      pm(PUT, "Fixed.Dups", dvalue, NULL);
	      break;
	    case Var_Conv:
	      fscanf(fp, "%lg", &dvalue );
	      pm(PUT, "Fixed.Var_Conv", dvalue, NULL);
	      break;
	    case Funct_Conv:
	      fscanf(fp, "%lg", &dvalue );
	      pm(PUT, "Fixed.Funct_Conv", dvalue, NULL);
	      break;
	    case FD_Step:
	      fscanf(fp, "%lg", &dvalue );
	      pm(PUT, "Fixed.FD_Step", dvalue, NULL);
	      break;
	    case Eigen_Dist:
	      fscanf(fp, "%lg", &dvalue );
	      pm(PUT, "Fixed.Eigen_Dist", dvalue, NULL);
	      break;

	    case Map_Period:			  /* ints for periodic */
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Map_Period", ivalue, NULL);
	      break;
	    case Algorithm:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Algorithm", ivalue, NULL);
	      break;
	    case Guess:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Guess", ivalue, NULL);
	      break;
	    case Guess_Num:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Mc_Guesses", ivalue, NULL);
	      break;
	    case Setting:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Setting", ivalue, NULL);
	      break;
	    case Num_Iters:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Num_Iters", ivalue, NULL);
	      break;
	    case Stab_Points:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Stab_Points", ivalue, NULL);
	      break;
	    case Stab_Steps:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Stab_Steps", ivalue, NULL);
	      break;
	    case Unstab_Points:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Unstab_Points", ivalue, NULL);
	      break;
	    case Unstab_Steps:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Fixed.Unstab_Steps", ivalue, NULL);
	      break;

	    case Stepsize:	 /* double for Flow_Control (orbits) */
	      fscanf(fp, "%lg", &dvalue );
	      pm(PUT, "Flow.Stepsize", dvalue, NULL);
	      break;	     
	    case  Diverg_Cutoff:
	      fscanf(fp, "%lg", &dvalue );
	      pm(PUT, "Flow.Diverg_Cutoff", dvalue, NULL);
	      break;	     
	    case Final_Time:
	      fscanf(fp, "%lg", &dvalue );
	      pm(PUT, "Flow.Final_Time", dvalue, NULL);
	      break;	     
 
	    case Start_Save_Points:		  /* ints for Flow_Control */
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Flow.Start_Save_Points", ivalue, NULL);
	      break;	      
	    case Total_Iterates:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Flow.Total_Iterates", ivalue, NULL);
	      break;	      
	    case Skip_Size:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Flow.Skip_Size", ivalue, NULL);
	      break;	      
	    case Stopping_Condition:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Flow.Stopping_Condition", ivalue, NULL);
	      break;	      

	    case Varb_Events:			  /* arrays for Flow_Control */
	      set_pm_ivalue(fp, data_v_dim, cur_v_dim, "Flow.Varb_Events");
	      break;
	    case Funct_Events:
	      set_pm_ivalue(fp, data_f_dim, cur_f_dim, "Flow.Funct_Events");
	      break; 
	    case Varb_Event_Values:
	      set_pm_dvalue(fp, data_v_dim, cur_v_dim, 
			    "Flow.Varb_Event_Values");
	      break;
	    case Funct_Event_Values:
	      set_pm_dvalue(fp, data_f_dim, cur_f_dim, 
			    "Flow.Funct_Event_Values");
	      break;

	    case Disp_Points:			  /* ints for Defaults */
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Defaults.Disp_Points", ivalue, NULL);
	      break;	
	    case Clipping:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Defaults.Clipping", ivalue, NULL);
	      break;	
	    case Recording:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Defaults.Recording", ivalue, NULL);
	      break;	
	    case Def_Symbol_Index:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Defaults.Symbol_Index", ivalue, NULL);
	      break;	
	    case Def_Size_Index:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Defaults.Size_Index", ivalue, NULL);
	      break;	
	    case Precision:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Defaults.Precision", ivalue, NULL);
	      break;	

	    case Mult_Load_Choice:		  /* ints for Mult_Cntl */
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Mult.Load_Choice", ivalue, NULL);
	      break;
	    case Mult_Transformation:
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Mult.Transformation", ivalue, NULL);
	      break;
	    case Images:      
	      fscanf(fp, "%d", &ivalue );
	      pm(PUT, "Mult.Images", ivalue, NULL);
	      break;

	    case Mult_Trans_Param:		  /* double for Mult_Cntl */
	      fscanf(fp, "%lg", &dvalue );
	      pm(PUT, "Mult.Trans_Param", dvalue, NULL);
	      break;

	    case Mult_Points:		      /* int arrays for Mult_Cntl */
	      set_pm_ivalue(fp, data_v_dim+data_p_dim, cur_v_dim+cur_p_dim, 
			    "Mult.Points");
	      break;

	    case Mult_Radius:		    /* double arrays for Mult_Cntl */
	      set_pm_dvalue(fp, data_v_dim+data_p_dim, cur_v_dim+cur_p_dim, 
			    "Mult.Radius");
	      break;

	    case Traj_Mem_Ptr:
	      keep_going = fill_memory_obj_1p1(fp, "Memory.Traj", 
					   DEFAULT_TRAJ_LENGTH);
	      break;	    
	    case Mult_Mem_Ptr:
	      keep_going = fill_memory_obj_1p1(fp, "Memory.Mult", 
					   DEFAULT_MULT_LENGTH);
	      break;
	    case Fixed_Mem_Ptr:
	      keep_going = fill_memory_obj_1p1(fp, "Memory.Fixed", 
					   DEFAULT_FP_LENGTH);
	      break;
	    case Cont_Mem_Ptr:
	      keep_going = fill_memory_obj_1p1(fp, "Memory.Continuation", 
					   DEFAULT_CONT_LENGTH);
	      break;
	    case Param_Mem_Ptr:
	      keep_going = fill_memory_obj_1p1(fp, "Memory.Param", 
					   DEFAULT_PARAM_LENGTH);
	      break;
	    case Sel_Pt_Mem_Ptr:
	      keep_going = fill_memory_obj_1p1(fp, "Memory.Sel_Pt", 
					   DEFAULT_SEL_PT_LENGTH);
	      break;
	    case NEW_FUNC:
	      /* NOTE: function calls are not reloaded into memory.  
		 File CLOSED at this point */
	      /* Be sure that the functions are ALWAYS placed at the 
		 end of the data file so that we don't have to worry 
		 about filtering the function data. */
	      keep_going = 1;
	      break;
	    case CONFIGURATION:
	      /* load_config(fp); */
	      break;
	    default:
	      break;
	    }	
	}

  return(keep_going);
}


/*
  fill_memory_obj()  initializes a pointer to a memory object and reads data from a file into that object.
  Returns 0 upon sucessful completion; -3 if cannot initialize memory.
  */
int
  fill_memory_obj(fp,ptr_type,length, mtype)
FILE *fp;
char *ptr_type;
int length;
int mtype;
{
  memory		ptr;
  
  if (!(ptr = (memory) pm(GET, ptr_type, NULL)))
    {
      if ( *(int *)pm(INIT, ptr_type, NULL) ) 
	return( -3 );
      else
	ptr = (memory) pm(GET, ptr_type, NULL);
    }
  return( read_data_obj(fp,ptr,length,mtype) );
}


/*
  read_data_obj() reads from a file and loads data objects into memory.
  The objects handled are trajectories, fixed points, continuation data, parameter data, and selected points.
  */
int
read_data_obj(fp, ptr, length, mtype)
FILE            *fp;
memory          ptr;
int             length;
int 		mtype;
{
  char          word[SIZE_OF_GEN_INPUT];
  int           n_obj,n_pts,n_doubles,n_integers,n_dparams, n_iparams;
  int           *ilist,*iparams;
  double	*dlist, *dparams;
  int           keep_going, 
	        first_obj = TRUE;
  int 		colorloc;
  
  colorloc = where_are_colors(mtype);

  /*  fscanf(fp,"%s %s %d %s %s %d %s",word,word,&n_obj,word,word,&n_dparams,word);*/ /* read num objects, num doubles in header */
  /*  fscanf(fp,"%s %s %d %s",word,word,&n_iparams,word); */ /* read number of integers in header */
  /*  fscanf(fp,"%s %s %d %s %d",word,word,&n_doubles,word,&n_integers); */
  /*  get_double_list_1p1(fp,n_dparams,dparams);*/
  /*  get_integer_list_1p1(fp,n_iparams,iparams); */
  n_obj = * ((int *) pm(GET,"Cur_Memory.Num_Objects",NULL));

  n_dparams = * ((int *) pm(GET,"Cur_Memory.Num_Header_Doubles",NULL));
  n_iparams = * ((int *) pm(GET,"Cur_Memory.Num_Header_Integers",NULL));

  dparams = dvector(0,n_dparams-1);
  pm(GET_LIST,"Cur_Memory.Header_Doubles",0,n_dparams - 1,dparams,NULL);

  iparams = ivector(0,n_iparams-1);
  pm(GET_LIST,"Cur_Memory.Header_Integers",0,n_iparams - 1,iparams,NULL);
  if ((n_iparams >= 3)&&(colorloc == HEADER))
      symb_convert_2to3(iparams);

  n_doubles = * ((int *) pm(GET,"Cur_Memory.Num_Body_Doubles",NULL));
  n_integers = * ((int *) pm(GET,"Cur_Memory.Num_Body_Integers",NULL));
 
  dlist = dvector(0,n_doubles-1);
  ilist = ivector(0,n_integers-1);

  if (memory_vanilla_start_new_flow(ptr, n_obj, n_doubles, n_integers, length, n_dparams, n_iparams) == 0) {
      memory_vanilla_add_point(ptr, NULL, NULL, dparams, iparams); /* set header info */
      while ( n_obj-- )	{
	  keep_going = TRUE;
	  if (first_obj)
	      fscanf(fp,"%s %s",word,word);	/* { \n { */
	  else
	      fscanf(fp,"%s",word);	/* { */
	  n_pts = *((int *)pm(GET,"Cur_Memory.Object_Num_Points",NULL));
	  /*	  if (first_obj) 
		  fscanf(fp,"%s %s %d %s",word,word,&n_pts,word); *//* don't assume n_pts pts follow; read till next traj */
	  /* else
	     fscanf(fp,"%s %d %s",word,&n_pts,word); *//* read one less string */
	  while( keep_going ) {
	      if ( n_doubles )   
		  keep_going = get_double_list(fp,n_doubles,dlist);
	      if ( n_integers && keep_going )
		  keep_going =  get_integer_list(fp,n_integers,ilist);
              if (( n_integers >= 3)&&(colorloc == BODY)) 
                  symb_convert_2to3(ilist);
	      if ( keep_going )
		  memory_vanilla_add_point(ptr, dlist, ilist, NULL, NULL); /* do not change header; only body */
	    }
	  /*fscanf(fp,"%s",word);*/	/* trailing } read in point*/
	  memory_next_traj(ptr);
	  first_obj = FALSE;
      }
      memory_end_current_flow(ptr);
      fscanf(fp,"%s",word);	/* closing bracket "}" of set of objects */ 
  }
  free_dvector(dlist,0,n_doubles-1);
  free_ivector(ilist,0,n_integers-1);
  free_dvector(dparams,0,n_dparams-1);
  free_ivector(iparams,0,n_iparams-1);
  return( 0 );
}


/*
  fill_memory_obj()  initializes a pointer to a memory object and reads data from a file into that object.
  Returns 0 upon sucessful completion; -3 if cannot initialize memory.
  */
int
  fill_memory_obj_1p1(fp,ptr_type,length)
FILE *fp;
char *ptr_type;
int length;
{
  memory		ptr;
  
  if (!(ptr = (memory) pm(GET, ptr_type, NULL)))
    {
      if ( *(int *)pm(INIT, ptr_type, NULL) ) 
	return( -3 );
      else
	ptr = (memory) pm(GET, ptr_type, NULL);
    }
  return( read_data_obj_1p1(fp,ptr,length) );
}


/*
  read_data_obj() reads from a file and loads data objects into memory.
  The objects handled are trajectories, fixed points, continuation data, parameter data, and selected points.
  */
read_data_obj_1p1(fp, ptr, length)
FILE            *fp;
memory          ptr;
int             length;
{
  char          word[SIZE_OF_GEN_INPUT];
  int           n_obj,n_pts,n_doubles,n_integers,n_dparams, n_iparams;
  int           *ilist,*iparams;
  double	*dlist, *dparams;
  int           keep_going, first_obj = TRUE;
  
  fscanf(fp,"%s %s %d %s %s %d %s",word,word,&n_obj,word,word,&n_dparams,word); /* read num objects, num doubles in header */
  dparams = dvector(0,n_dparams-1);
  get_double_list_1p1(fp,n_dparams,dparams);
  fscanf(fp,"%s %s %d %s",word,word,&n_iparams,word); /* read number of integers in header */
  iparams = ivector(0,n_iparams-1);
  get_integer_list_1p1(fp,n_iparams,iparams);
  symb_convert_2to3(iparams);

  fscanf(fp,"%s %s %d %s %d",word,word,&n_doubles,word,&n_integers); 
  dlist = dvector(0,n_doubles-1);
  ilist = ivector(0,n_integers-1);

  if (memory_vanilla_start_new_flow(ptr, n_obj, n_doubles, n_integers, length, n_dparams, n_iparams) == 0)
    {
      memory_vanilla_add_point(ptr, NULL, NULL, dparams, iparams); /* set header info */
      while ( n_obj-- )
	{
	  keep_going = TRUE; 
	  if (first_obj) 
	    fscanf(fp,"%s %s %d %s",word,word,&n_pts,word); /* don't assume n_pts pts follow; read till next traj */
	  else
	    fscanf(fp,"%s %d %s",word,&n_pts,word); /* read one less string */
	  while( keep_going )
	    {
	      if ( n_doubles )   
		keep_going = get_double_list_1p1(fp,n_doubles,dlist);
	      if ( n_integers && keep_going )
		keep_going =  get_integer_list_1p1(fp,n_integers,ilist);
	      if ( keep_going )
		memory_vanilla_add_point(ptr, dlist, ilist, NULL, NULL); /* do not change header; only body */
	    }
	  memory_next_traj(ptr);
	  first_obj = FALSE;
	}
      memory_end_current_flow(ptr);
    }
  free_dvector(dlist,0,n_doubles-1);
  free_ivector(ilist,0,n_integers-1);
  free_dvector(dparams,0,n_dparams-1);
  free_ivector(iparams,0,n_iparams-1);
  return( 0 );
}


/*
  read_sys_name() reads the system name from a file.  If the system is currently loaded,
  -1 is returned.  If the system is not currently
  installed, the number of the system (in DS_Sel) is returned.  
  If the system name cannot be found, -2 is returned.
*/
int
read_sys_name(fp)
FILE     *fp;
{
  char      name[MAX_LEN_DS_TITLE], *modelname, *pname;
  int       i;

  fgets( name, MAX_LEN_DS_TITLE, fp );		  /* read to end of line */
  pname = name;					  /* strip off leading white space */
  while ( isspace( (int)*pname++ ))
    ;
  pname--;
  modelname = get_ds_name();

  if ( !strncmp( pname, modelname, strlen(modelname) )) /* current system matches name in file */
    return -1;
  for( i=0; i<N_DS; i++)
    {
      modelname = DS_Sel[i].DS_Name;
      if ( !strncmp( pname, modelname, strlen(modelname) )) /* compare at most strlen(modelname) chars. */
	return( i );				  /* This eliminate mismatch because of junk or whitespace */
    }						  /*  at the end of name */
  return( -2 );
}

/*     
  get_integer_list_1p1() fills up an integer array with integers read from a file
  Function returns TRUE if it properly fills array.
  Function returns FALSE if next line is a text line (eg, a new trajectory)
  */
int
  get_integer_list(fp,data_dim,p)
FILE 		*fp;
int 		data_dim;
int 		*p;
{
  int		i;
  char		word[SIZE_OF_GEN_INPUT];
  
  
  if ( (data_dim == 0) || (fscanf(fp,"%s",word)==EOF)  || (strcmp(word,"{")) ) /* data strats with { */
    return(FALSE);

  for (i=0; i<data_dim; i++)
    fscanf(fp,"%d", (p + i) );
  fscanf(fp,"%s",word);		/* read trailing } at end of point */
  return(TRUE);  
}

/*     
  get_double_list() fills up a double array with double read from a file
  Function returns TRUE if it properly fills array.
  Function returns FALSE if next line is a text line (eg, a new trajectory)
  */
int
  get_double_list(fp,data_dim,p)
FILE 		*fp;
int 		data_dim;
double 		*p;
{
  int		i;
  char		word[SIZE_OF_GEN_INPUT];
  
  if ( (data_dim==0) || (fscanf(fp,"%s",word)==EOF)  || strcmp(word,"{") ) /* data strats with { */
    return(FALSE);
  for (i=0; i<data_dim; i++)
    fscanf(fp,"%lg", (p + i) );
  fscanf(fp,"%s",word);		/* read trailing } at end of point */
  return(TRUE);
}



/*     
  get_integer_list_1p1() fills up an integer array with integers read from a file
  Function returns TRUE if it properly fills array.
  Function returns FALSE if next line is a text line (eg, a new trajectory)
  */
int
  get_integer_list_1p1(fp,data_dim,p)
FILE 		*fp;
int 		data_dim;
int 		*p;
{
  int		i;
  char		word[SIZE_OF_GEN_INPUT];
  
  if ( (data_dim == 0) || (fscanf(fp,"%s",word)==EOF)  || (!strcmp(word,"#")) ) /* new text line begins with pound sign*/
    return(FALSE);

  *p = atoi(word); 
  for (i=1; i<data_dim; i++)
    fscanf(fp,"%d", (p + i) );
  return(TRUE);  
}

/*     
  get_double_list_1p1() fills up a double array with double read from a file
  Function returns TRUE if it properly fills array.
  Function returns FALSE if next line is a text line (eg, a new trajectory)
  */
int
  get_double_list_1p1(fp,data_dim,p)
FILE 		*fp;
int 		data_dim;
double 		*p;
{
  int		i;
  char		word[SIZE_OF_GEN_INPUT];
  
  if ( (data_dim==0) || (fscanf(fp,"%s",word)==EOF)  || (!strcmp(word,"#")) ) /* new text line begins with pound sign */
    return(FALSE);
  *p = atof(word); 
  for (i=1; i<data_dim; i++)
    fscanf(fp,"%lg", (p + i) );
  return(TRUE);
}


/*	
  set_pm_dvalue() sets the appropriate postmaster double precision variable.
  This is not intended to be a general purpose function, just a convenient way to 
  read in many different variable types with minimal effort.
  */
void
set_pm_dvalue(fp, data_dim, cur_dim, pm_label)
FILE 		*fp;
int 		data_dim, cur_dim;
char *pm_label;
{
  double		value;
  int 		i;
  
  for (i=0;i< min(data_dim,cur_dim);i++)
    {						  /* allow for data varbs to be different dim than current system */
      if (fscanf(fp,"%lg",&value)==1)
	pm(PUT, pm_label, i, value, NULL);
    }
}

/*	
  set_pm_ivalue() sets the appropriate postmaster integer variable.
  This is not intended to be a general purpose function, just a convenient way to 
  read in many different variable types with minimal effort.
  */
void 
set_pm_ivalue(fp, data_dim, cur_dim, pm_label)
FILE 		*fp;
int 		data_dim, cur_dim;
char *pm_label;
{
  int 		i, value;
  
  for (i=0;i< min(data_dim,cur_dim);i++)
    {						  /* allow for data varbs to be different dim than current system */
      if (fscanf(fp,"%d",&value)==1)
	pm(PUT, pm_label, i, value, NULL);
    }
}

/*     
  old_keyword() returns NO_KEY if the given string is not in a list of
  key words.  If the word is in the list, it returns the integer position of the word on the list
  */
int
  old_keyword(w)
char    *w;
{
  int		i;
  
  allupper(w);
  for (i=0; i< N_OLDKEYWORD; i++)
    if (!strcmp(w,old_key[i].word))
      return(old_key[i].index);
  return(NO_KEY);
}

/*
  allupper() converts a string to all uppercase characters.  The original string is
  lost in the process.  Ex: if word points to "HelLo WorLd!#1" then allupper(word)
  returns a pointer to "HELLO WORLD!#1".
  */
char *
  allupper(w)
char *w;
{
  char *p;
  
  p = w;
  while ( (*w++ = toupper(*w)) != '\0')
    ;
  return( p );
}

/*
  alllower() converts a string to all lowercase characters.  The original string is
  lost in the process.  Ex: if word points to "HelLo WorLd!#1" then alllower(word)
  returns a pointer to "hello world!#1".
  */
char *
  alllower(w)
char *w;
{
  char *p;
 
  p = w;
  while ( (*w++ = tolower(*w)) != '\0')
    ;
  return( p );
}

int
    load_model_if_new()
{
  char      	load_name[MAX_LEN_DS_TITLE], *model_name;
  char 		tclcmd[MAX_LONG_STR];

  int		i;

  pm(GET,"Load.Model_Name",load_name,NULL);
  model_name = get_ds_name();

  if (! strncmp(load_name,model_name,strlen(model_name)))
      return NO_ERROR;
  for( i=0; i<N_DS; i++) {
      model_name = DS_Sel[i].DS_Name;
      if ( !strncmp( load_name, model_name, strlen(model_name) )) /* compare at most strlen(modelname) chars. */
	  pm(PUT, "Model.Load_Number", i,
	     EXEC, "Model.Load", 
	     NULL);
/*	  load_model();*/	
  }

/* Update the windows so that the defaults are loaded in correctly when the other panels
 *	are opened    8/3/97 BAM 
 */
  sprintf (tclcmd, "pm_to_tcl");
/*  printf ("%s\n", tclcmd); */
  Tcl_Eval(interp, tclcmd);

  return -2;
      
}



int 
load_config_wingeom(fp,str)
FILE *fp;
char *str;
{
    int status, i, n, lower, len;
    char tclcmd[MAX_LONG_STR], win_name[MAX_LONG_STR], c;  
    int idata1, idata2, idata3, idata4;
    int errorstatus;

    status = fscanf(fp,"%d %d", &lower, &n);
    if (status != 2) {
        errorstatus = -2;
        fprintf(stderr, "%s: error loading %s\n", PROGRAM_TITLE, str);
        }
    if (n == 3) {     /* It should...  */
	/* Get window geometry info */
        status = fscanf(fp, "%d %d %d %d", &idata1, &idata2, &idata3, &idata4);
        if (status != 4) {
            errorstatus = -2;
            fprintf(stderr, "%s: error loading %s\n", PROGRAM_TITLE, str);
            }
        len = strlen(str);
        for (i=0;i<(len-9);i++) {  /* Strip off "Win.Locn." */
            win_name[i] = str[i+9];
            }
        win_name[i] = '\0';
        
	len = strlen(win_name);
        if (strstr(win_name, "One") != NULL) { 
	  /* It's a one-D window.  NOTE: in version 3, they are multiple windows, 
	   * whereas in version 2, they weren't.  This patch is needed to remedy 
	   * that discrepancy.		8/3/97 BAM 
           */
            sprintf (tclcmd, "window(mult_open) oneD");
        } else if (strstr(win_name, "Two") != NULL) { /* It's a TwoD window from v3 */
            sprintf (tclcmd, "window(mult_open) twoD");
        } else if (strstr(win_name, "Three") != NULL) { /* It's a ThreeD window from v3 */
            sprintf (tclcmd, "window(open) threeD");
        } else if (strncmp(win_name, "Snap", len) == 0) {/* It's a Snapshot window */
	    sprintf (tclcmd, "twoD(snap)");
        } else {		/* It's none of the above */
            alllower(win_name);
            sprintf (tclcmd, "window(open) %s", win_name);
            }
  /*      printf ("%s\n", tclcmd);  */
        Tcl_Eval(interp, tclcmd);
/*        printf ("%s <---  %s \n", win_name, interp->result);  */
        strcpy(win_name, interp->result);

        sprintf (tclcmd, "window(geometry) %s +%d+%d", win_name, idata1, idata2); 
   /*     printf ("%s\n", tclcmd);  */
        Tcl_Eval(interp, tclcmd);
        } else {		/* Skip to end of line */
            while (( (c=getc(fp)) != '\n') && (c != EOF));
            }
  /*  printf ("\n"); */
    return(errorstatus);
}


/* 8/4/97 BAM */
int
stopcond_convert_2to3(int_data)
int *int_data;
{
    switch (*int_data) {
        case OLD_PROP_NSTEP:
            *int_data = PROP_NSTEP;
            break;
        case OLD_PROP_FSTOP:
            *int_data = PROP_FSTOP;
            break;
        case OLD_PROP_TF:
            *int_data = PROP_TF;
            break;
        case OLD_PROP_POINCARE:
            *int_data = PROP_POINCARE;
            break;
        default:
            break;
        }
 
}

