/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves
 *      shawn@talula.demon.co.uk
 *      http://www.talula.demon.co.uk/allegro/
 *
 *      The 2d polygon rasteriser.
 *
 *      See readme.txt for copyright information.
 */
 /*
===================================
============ Copyright ============
===================================

   Note: these license terms have recently been modified! See the Allegro
   website for an explanation of why a new license was needed.

   Allegro is gift-ware. It was created by a number of people working in
   cooperation, and is given to you freely as a gift. You may use, modify,
   redistribute, and generally hack it about in any way you like, and you do
   not have to give us anything in return. However, if you like this product
   you are encouraged to thank us by making a return gift to the Allegro
   community. This could be by writing an add-on package, providing a useful
   bug report, making an improvement to the library, or perhaps just
   releasing the sources of your program so that other people can learn from
   them. If you redistribute parts of this code or make a game using it, it
   would be nice if you mentioned Allegro somewhere in the credits, but you
   are not required to do this. We trust you not to abuse our generosity.

   http://www.talula.demon.co.uk/allegro/
*/

/*
modified by Caolan.McNamara@ul.ie http://www.csn.ul.ie/~caolan for use
as a replacement of the gdImageFilledPolygon in use by gd1.3 which didnt
work the way i wanted it to, this is basically a straight cut and paste job.
*/

#include "xgd.h"

void myline(int x1,int y, int x2);

#include <limits.h>
#include <stdlib.h>

#ifndef MIN
#define MIN(x,y)     (((x) < (y)) ? (x) : (y))
#define MAX(x,y)     (((x) > (y)) ? (x) : (y))
#define MID(x,y,z)   MAX((x), MIN((y), (z)))
#endif

#ifndef ABS
#define ABS(x)       (((x) >= 0) ? (x) : (-(x)))
#endif



/* number of fractional bits used by the polygon rasteriser */
#define POLYGON_FIX_SHIFT     18

/* information for polygon scanline fillers */
typedef struct POLYGON_SEGMENT
{
   long u, v, du, dv;              /* long point u/v coordinates */
   long c, dc;                     /* single color gouraud shade values */
   long r, g, b, dr, dg, db;       /* RGB gouraud shade values */
   float z, dz;                     /* polygon depth (1/z) */
   float fu, fv, dfu, dfv;          /* floating point u/v coordinates */
   unsigned char *texture;          /* the texture map */
   int umask, vmask, vshift;        /* texture map size information */
   int seg;                         /* destination bitmap selector */
} POLYGON_SEGMENT;



/* an active polygon edge */
typedef struct POLYGON_EDGE
{
   int top;                         /* top y position */
   int bottom;                      /* bottom y position */
   long x, dx;                     /* long point x position and gradient */
   long w;                         /* width of line segment */
   POLYGON_SEGMENT dat;             /* texture/gouraud information */
   struct POLYGON_EDGE *prev;       /* doubly linked list */
   struct POLYGON_EDGE *next;
} POLYGON_EDGE;



/* fill_edge_structure:
 *  Polygon helper function: initialises an edge structure for the 2d
 *  rasteriser.
 */
void fill_edge_structure(POLYGON_EDGE *edge, gdPoint *i1, gdPoint *i2)
{
   gdPoint *it;
   if (i2->y < i1->y) {

      it = i1;
      i1 = i2;
      i2 = it;
   }

   edge->top = i1->y;
   edge->bottom = i2->y -1;
   edge->dx = ((i2->x - i1->x) << POLYGON_FIX_SHIFT) / (i2->y - i1->y);
   edge->x = (i1->x << POLYGON_FIX_SHIFT) + (1<<(POLYGON_FIX_SHIFT-1)) - 1;
   edge->prev = NULL;
   edge->next = NULL;

   if (edge->dx < 0)
      edge->x += MIN(edge->dx+(1<<POLYGON_FIX_SHIFT), 0);

   edge->w = MAX(ABS(edge->dx)-(1<<POLYGON_FIX_SHIFT), 0);
}



/* _add_edge:
 *  Adds an edge structure to a linked list, returning the new head pointer.
 */
POLYGON_EDGE *_add_edge(POLYGON_EDGE *list, POLYGON_EDGE *edge, int sort_by_x)
{
   POLYGON_EDGE *pos = list;
   POLYGON_EDGE *prev = NULL;

   if (sort_by_x) {
      while ((pos) && ((pos->x + (pos->w + pos->dx) / 2) < 
		       (edge->x + (edge->w + edge->dx) / 2))) {
	 prev = pos;
	 pos = pos->next;
      }
   }
   else {
      while ((pos) && (pos->top < edge->top)) {
	 prev = pos;
	 pos = pos->next;
      }
   }

   edge->next = pos;
   edge->prev = prev;

   if (pos)
      pos->prev = edge;

   if (prev) {
      prev->next = edge;
      return list;
   }
   else
      return edge;
}



/* _remove_edge:
 *  Removes an edge structure from a list, returning the new head pointer.
 */
POLYGON_EDGE *_remove_edge(POLYGON_EDGE *list, POLYGON_EDGE *edge)
{
   if (edge->next) 
      edge->next->prev = edge->prev;

   if (edge->prev) {
      edge->prev->next = edge->next;
      return list;
   }
   else
      return edge->next;
}



/* polygon:
 *  Draws a filled polygon with an arbitrary number of corners. Pass the 
 *  number of vertices, then an array containing a series of x, y points 
 *  (a total of vertices*2 values).
 */
void NEW_polygon(gdImagePtr im,gdPointPtr points,int vertices, int color)
{
   int c;
   int top = INT_MAX;
   int bottom = INT_MIN;
   gdPoint *i1, *i2;
   POLYGON_EDGE *start;
   POLYGON_EDGE *edge, *next_edge;
   POLYGON_EDGE *active_edges = NULL;
   POLYGON_EDGE *inactive_edges = NULL;

   /* allocate some space and fill the edge table */
   edge = (POLYGON_EDGE *)malloc(sizeof(POLYGON_EDGE) * vertices);
   start = edge;
   i1 = points;
   i2 = points + (vertices-1);

   for (c=0; c<vertices; c++) {
      if (i1->y != i2->y) {
	 fill_edge_structure(edge, i1, i2);

	 if (edge->bottom >= edge->top) {

	    if (edge->top < top)
	       top = edge->top;

	    if (edge->bottom > bottom)
	       bottom = edge->bottom;

	    inactive_edges = _add_edge(inactive_edges, edge, 0);
	    edge++;
	 }
      }
      i2 = i1;
      i1++;
   }

   /* for each scanline in the polygon... */
   for (c=top; c<=bottom; c++) 
   	{

      /* check for newly active edges */
      edge = inactive_edges;
      while ((edge) && (edge->top == c)) {
	 next_edge = edge->next;
	 inactive_edges = _remove_edge(inactive_edges, edge);
	 active_edges = _add_edge(active_edges, edge, 1);
	 edge = next_edge;
      }

      /* draw horizontal line segments */
      edge = active_edges;
      while ((edge) && (edge->next)) {
	 gdImageLine(im,edge->x>>POLYGON_FIX_SHIFT, c, (edge->next->x+edge->next->w)>>POLYGON_FIX_SHIFT,c,color);
	 edge = edge->next->next;
      }

      /* update edges, sorting and removing dead ones */
      edge = active_edges;
      while (edge) {
	 next_edge = edge->next;
	 if (c >= edge->bottom) {
	    active_edges = _remove_edge(active_edges, edge);
	 }
	 else {
	    edge->x += edge->dx;
	    while ((edge->prev) && 
		   (edge->x+edge->w/2 < edge->prev->x+edge->prev->w/2)) {
	       if (edge->next)
		  edge->next->prev = edge->prev;
	       edge->prev->next = edge->next;
	       edge->next = edge->prev;
	       edge->prev = edge->prev->prev;
	       edge->next->prev = edge;
	       if (edge->prev)
		  edge->prev->next = edge;
	       else
		  active_edges = edge;
	    }
	 }
	 edge = next_edge;
      }
   }
   free(start);
}
