/*	MikMod sound library
	(c) 1998, 1999 Miodrag Vallat and others - see file AUTHORS for
	complete list.

	This library is free software; you can redistribute it and/or modify
	it under the terms of the GNU Library General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.
 
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Library General Public License for more details.
 
	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
	02111-1307, USA.
*/

/*==============================================================================

  $Id: drv_oss.c,v 1.25 1999/06/28 03:00:05 miod Exp $

  Driver for output on Linux and FreeBSD Open Sound System (OSS) (/dev/dsp) 

==============================================================================*/

/*
	Written by Chris Conn <cconn@tohs.abacom.com>

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <errno.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <string.h>
#include <stdlib.h>
#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif
#ifdef __FreeBSD__
#include <machine/soundcard.h>
#else 
#include <sys/soundcard.h>
#endif

#include <mikmod_internals.h>

/* compatibility with very, very old versions of OSS */
#ifndef AFMT_S16_LE
#define AFMT_S16_LE (16)
#endif
#ifndef AFMT_U8
#define AFMT_U8 (8)
#endif

#define DEFAULT_FRAGSIZE 14
#define DEFAULT_NUMFRAGS 16

static	int sndfd=-1;
static	SBYTE *audiobuffer=NULL;
static	int fragsize=DEFAULT_FRAGSIZE,numfrags=DEFAULT_NUMFRAGS;

static void OSS_CommandLine(CHAR *cmdline)
{
	CHAR *ptr;

	if((ptr=MD_GetAtom("buffer",cmdline,0))) {
		fragsize=atoi(ptr);
		if((fragsize<7)||(fragsize>17)) fragsize=DEFAULT_FRAGSIZE;
		free(ptr);
	}
	if((ptr=MD_GetAtom("count",cmdline,0))) {
		numfrags=atoi(ptr);
		if((numfrags<2)||(numfrags>255)) numfrags=DEFAULT_NUMFRAGS;
		free(ptr);
	}
}

static BOOL OSS_IsThere(void)
{
	/* under Linux, and perhaps other Unixes, access()ing the device is not
	   enough since it won't fail if the machine doesn't have sound support
	   in the kernel or sound hardware                                      */
	int fd;

	if((fd=open("/dev/dsp",O_WRONLY))>0) {
		close(fd);
		return 1;
	}
	return (errno==EACCES?1:0);
}

static BOOL OSS_Init_internal(void)
{
	int play_precision,play_stereo,play_rate;
	int orig_precision,orig_stereo,orig_rate;
	audio_buf_info buffinf;

	orig_precision=play_precision=(md_mode&DMODE_16BITS)?AFMT_S16_LE:AFMT_U8;
	orig_stereo=play_stereo=(md_mode&DMODE_STEREO)?1:0;
	orig_rate=play_rate=md_mixfreq;

	if((ioctl(sndfd,SNDCTL_DSP_SAMPLESIZE,&play_precision)<0)||
	   (orig_precision!=play_precision)) {
		_mm_errno=MMERR_OSS_SETSAMPLESIZE;
		return 1;
	}
	if((ioctl(sndfd,SNDCTL_DSP_STEREO,&play_stereo)<0)||
	   (orig_stereo!=play_stereo)) {
		_mm_errno=MMERR_OSS_SETSTEREO;
		return 1;
	}
	if((ioctl(sndfd,SNDCTL_DSP_SPEED,&play_rate)<0)) {
		_mm_errno=MMERR_OSS_SETSPEED;
		return 1;
	}
	/* if(orig_rate!=play_rate) */
		md_mixfreq=play_rate;

	ioctl(sndfd,SNDCTL_DSP_GETOSPACE,&buffinf);
	if(!(audiobuffer=(SBYTE*)_mm_malloc(buffinf.fragsize)))
		return 1;

	return VC_Init();
}

static BOOL OSS_Init(void)
{
	int fragmentsize;

	if((sndfd=open("/dev/dsp",O_WRONLY))<0) {
		_mm_errno=MMERR_OPENING_AUDIO;
		return 1;
	}

	if((fragsize==DEFAULT_FRAGSIZE)&&(getenv("MM_FRAGSIZE"))) {
		fragsize=atoi(getenv("MM_FRAGSIZE"));
		if((fragsize<7)||(fragsize>17)) fragsize=DEFAULT_FRAGSIZE;
	}
	if((numfrags==DEFAULT_NUMFRAGS)&&(getenv("MM_NUMFRAGS"))) {
		numfrags=atoi(getenv("MM_NUMFRAGS"));
		if((numfrags<2)||(numfrags>255)) numfrags=DEFAULT_NUMFRAGS;
	}

	fragmentsize=(numfrags<<16)|fragsize;
	
	if(ioctl(sndfd,SNDCTL_DSP_SETFRAGMENT,&fragmentsize)<0) {
		_mm_errno=MMERR_OSS_SETFRAGMENT;
		return 1;
	}

	return OSS_Init_internal();
}

static void OSS_Exit_internal(void)
{
	VC_Exit();
	if (audiobuffer) {
		free(audiobuffer);
		audiobuffer=NULL;
	}
}

static void OSS_Exit(void)
{
	OSS_Exit_internal();

	if (sndfd>=0) {
		close(sndfd);
		sndfd=-1;
	}
}

static void OSS_PlayStop(void)
{
	VC_PlayStop();

	ioctl(sndfd,SNDCTL_DSP_POST);
}

static void OSS_Update(void)
{
	audio_buf_info buffinf;

	ioctl(sndfd,SNDCTL_DSP_GETOSPACE,&buffinf);
	if((buffinf.bytes)||(buffinf.fragstotal<=2)) {
		int cnt=(buffinf.fragments<1)?1:buffinf.fragments;

		while(cnt--)
			write(sndfd,audiobuffer,
			      VC_WriteBytes(audiobuffer,buffinf.fragsize));
	}
}

static BOOL OSS_Reset(void)
{
	OSS_Exit_internal();
	ioctl(sndfd,SNDCTL_DSP_RESET);
	return OSS_Init_internal();
}

MDRIVER drv_oss={
	NULL,
	"Open Sound System",
	"Open Sound System driver v1.5",
	0,255,
	"oss",

	OSS_CommandLine,
	OSS_IsThere,
	VC_SampleLoad,
	VC_SampleUnload,
	VC_SampleSpace,
	VC_SampleLength,
	OSS_Init,
	OSS_Exit,
	OSS_Reset,
	VC_SetNumVoices,
	VC_PlayStart,
	OSS_PlayStop,
	OSS_Update,
	NULL,
	VC_VoiceSetVolume,
	VC_VoiceGetVolume,
	VC_VoiceSetFrequency,
	VC_VoiceGetFrequency,
	VC_VoiceSetPanning,
	VC_VoiceGetPanning,
	VC_VoicePlay,
	VC_VoiceStop,
	VC_VoiceStopped,
	VC_VoiceGetPosition,
	VC_VoiceRealVolume
};

/* ex:set ts=4: */
