# File "result.tcl":
# Main module to display Malaga results.

# This file is part of Malaga, a system for Left Associative Grammars.
# Copyright (C) 1995-1998 Bjoern Beutel
#
# Bjoern Beutel
# Universitaet Erlangen-Nuernberg
# Abteilung fuer Computerlinguistik
# Bismarckstrasse 12
# D-91054 Erlangen
# e-mail: malaga@linguistik.uni-erlangen.de
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Global variables:
# $result_numbers -- the list of result numbers
# $result_value($number) -- the value of result $number
# $result_input -- the input string that has been analysed

#------------------------------------------------------------------------------

# Display malaga results.

proc result_display {} {

  global font font_size result_numbers char_set result_value result_input
  
  .result.frame.canvas delete matrix_tag
  
  set canvas_border 10
  set char_height $font_size(result)
  set current_y $canvas_border
  set start_x $canvas_border
  set max_x $start_x
  
  set input_id [.result.frame.canvas create matrix \
		$start_x $current_y $result_input \
		-font $font(result) \
		-char_set $char_set \
		-tags matrix_tag]
  set max_x [maximum $max_x [widget_right .result.frame.canvas $input_id]]
  set current_y [expr ([widget_bottom .result.frame.canvas $input_id] \
		       + 2 * $font_size(result))]
  
  # Run through the result list:
  set first_result 1
  foreach number $result_numbers {
    
    # Space between result
    if {! $first_result} {
      incr current_y $char_height
    }

    # Display result number
    set number_id [.result.frame.canvas create matrix \
		   $start_x $current_y \
		   "! \"$number: \"" \
		   -font $font(result) \
		   -char_set $char_set \
		   -tags matrix_tag]
    
    # Display result value
    set matrix_id [.result.frame.canvas create matrix \
		   [widget_right .result.frame.canvas $number_id ] $current_y \
		   $result_value($number) \
		   -font $font(result) \
		   -char_set $char_set \
		   -tags matrix_tag]
    
    set max_x [maximum $max_x [widget_right .result.frame.canvas $matrix_id]]
    set current_y [widget_bottom .result.frame.canvas $matrix_id]
    
    # Center result number vertically
    .result.frame.canvas move $number_id \
    0 [expr ($current_y \
	     - [widget_bottom .result.frame.canvas $number_id])/2]
    
    set first_result 0
  }
  
  # Reconfigure scroll bars:
  .result.frame.canvas configure \
  -scrollregion \
  "0 0 [expr $max_x + $canvas_border] [expr $current_y + $canvas_border]"
}

#------------------------------------------------------------------------------

# Read the malaga results.

proc result_read {} {

  global result_numbers result_value result_input
  
  set result_input [read_line stdin]
  set result_numbers {}
  
  # Read all results
  while {[set line [read_line stdin]] != "end" } {
    set result_number [lindex $line 0]
    lappend result_numbers $result_number
    set result_value($result_number) [lindex $line 1]
  }
}

#------------------------------------------------------------------------------

# Create window where to display the results

proc result_open_window {} {

  global result_geometry

  if [winfo exists .result] {return}

  if {! [info exists result_geometry]} {set result_geometry 600x400}
  
  # Define start window.
  toplevel .result
  wm title .result "Malaga Results"
  wm iconname .result "Results"
  wm minsize .result 100 100
  wm geometry .result $result_geometry
  wm focusmodel .result active

  # Define the different menus.
  frame .result.menu -relief raised -borderwidth 1
  pack .result.menu -side top -fill x
  create_window_menu result
  create_font_menu result
  tk_menuBar .result.menu .result.menu.window .result.menu.fonts
  
  create_scroll_frame result
  focus .result.frame.canvas
}

#------------------------------------------------------------------------------

# Close the result window

proc result_close_window {} {

  global result_geometry
  
  if [winfo exists .result] {
    set result_geometry [winfo geometry .result]
    destroy .result
  }
}

#------------------------------------------------------------------------------

proc result_redraw {mode} {

  result_open_window

  if {$mode == "reload"} {result_read}
  result_display
  if {[wm state .result] == "normal"} {raise .result}
}

#------------------------------------------------------------------------------

# Initialize the variables of "result".

proc result_init {} {

  global font font_size

  set font_size(result) 14
  set font(result) $font($font_size(result))
}

#------------------------------------------------------------------------------
