/* File "transmit_process.c":
 * Invoking the transmit process from Malaga. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* includes =================================================================*/

#define _XOPEN_SOURCE
#define _XOPEN_SOURCE_EXTENDED 1
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/wait.h>
#include <signal.h>
#include <pwd.h>
#include <fcntl.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "files.h"
#include "input.h"
#include "scanner.h"
#include "symbols.h"

#undef GLOBAL
#define GLOBAL

#include "transmit_process.h"

/* constants ================================================================*/

#define MAX_ARGS 10       /* maximum number of arguments */
#define BUFFER_SIZE 10000 /* maximum size of input and output line */
#define STACK_SIZE 100    /* maximum complexity of values */

/* variables ================================================================*/

LOCAL FILE *transmit_input_stream;   /* to read data from transmit process */
LOCAL FILE *transmit_output_stream;  /* to write data to transmit process */
LOCAL pid_t process_id;              /* ID of transmit process */
LOCAL value_t stack[STACK_SIZE];     /* size of stack to build values */
LOCAL long_t top;                    /* first10f free element in <stack> */

/* functions ================================================================*/

LOCAL void push_value (value_t value)
/* Push <value> on stack. */
{
  if (top + 1 >= STACK_SIZE)
    error ("value too complex");
  
  stack[top] = value;
  top++;
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_symbol (void)
/* Parse a symbol and push it on the value stack. */
{
  symbol_t symbol;

  test_token (TOK_IDENT);
  symbol = find_symbol (token_name);
  push_value (symbol_to_value (symbol));
  read_next_token ();
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_simple_value (void)
/* Parse a value and leave it on the value_stack. */
{
  long_t n; /* number of values in list or record */
  
  switch (next_token) 
  {
  case '<': /* Parse a list. */
    read_next_token ();
    n = 0;
    if (next_token != '>') 
    {
      parse_simple_value ();
      n++;
      while (next_token == ',') 
      {
	read_next_token ();
	parse_simple_value ();
	n++;
      }
    }
    parse_token ('>');
    stack[top - n] = build_list (n, stack + top - n);
    top -= (n-1);
    break;
    
  case '[': /* Parse a record. */
    read_next_token ();
    n = 0;
    if (next_token != ']') 
    {
      parse_symbol ();
      parse_token (':');
      parse_simple_value ();
      n++;
      while (next_token == ',') 
      {
	read_next_token ();
	parse_symbol ();
	parse_token (':');
	parse_simple_value ();
	n++;
      }
    }
    parse_token (']');
    stack [top - 2*n] = build_record (n, stack + top - 2*n);
    top -= (2*n-1);
    break;
    
  case TOK_IDENT: /* Parse a symbol. */
    parse_symbol ();
    break;
    
  case TOK_STRING: /* Parse a string. */
    delete_escapes (token_name);
#ifdef HANGUL
    copy_string (token_name, ENCODED_STRING (token_name), 
		 token_name + TOKEN_NAME_MAX);
#endif
    push_value (string_to_value (token_name, NULL));
    read_next_token ();
    break;
      
  case TOK_NUMBER: /* Parse a number value. */
    push_value (double_to_value (token_number));
    read_next_token ();
    break;
    
  default:
    error ("value expected, not %s", token_as_text (next_token));
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void start_transmit_process (void)
/* Start the Malaga transmit process whose name is in the environment variable
 * MALAGA_TRANSMIT if it is not already running. */
{
  int to_transmit_fd[2];
  int from_transmit_fd[2];
  string_t malaga_transmit;

  if (process_id != 0)
  {
    if (waitpid (process_id, 0, WNOHANG) == 0)
      return;

    process_id = 0;
    fclose (transmit_input_stream);
    fclose (transmit_output_stream);
    transmit_input_stream = transmit_output_stream = NULL;
  }

  if (pipe (to_transmit_fd) == -1 || pipe (from_transmit_fd) == -1)
    error ("can't create pipe to transmit process: %s", strerror (errno));
    
  malaga_transmit = getenv ("MALAGA_TRANSMIT");
  if (malaga_transmit == NULL) 
    error ("missing environment variable \"MALAGA_TRANSMIT\"");

  switch (process_id = fork ())
  {
  case -1:
    error ("can't create transmit process: %s", strerror (errno));
    break;
    
  case 0:
  {
    string_t args[MAX_ARGS + 1];
    long_t num_args;

    dup2 (to_transmit_fd[0], STDIN_FILENO);
    close (to_transmit_fd[0]);
    close (to_transmit_fd[1]);
    dup2 (from_transmit_fd[1], STDOUT_FILENO);
    close (from_transmit_fd[0]);
    close (from_transmit_fd[1]);

    num_args = 0;
    parse_whitespace (&malaga_transmit);
    while (*malaga_transmit != EOS)
    {
      if (num_args == MAX_ARGS)
      {
	fprintf (stderr, "too many arguments in \"MALAGA_TRANSMIT\"\n");
	exit (1);
      }
      args[num_args] = parse_word (&malaga_transmit);
      num_args++;
    }
    args[num_args] = NULL;
    
    execvp (args[0], args);
    fprintf (stderr, "can't start transmit process \"%s\": %s\n", 
	     args[0], strerror (errno));
    exit (1);
  }
  default:
    close (to_transmit_fd[0]);
    transmit_output_stream = fdopen (to_transmit_fd[1], "w");
    close (from_transmit_fd[1]);
    transmit_input_stream = fdopen (from_transmit_fd[0], "r");
    if (transmit_input_stream == NULL || transmit_output_stream == NULL)
      error ("can't open data stream: %s", strerror (errno));
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void stop_transmit_process (void)
/* Stop the Malaga transmit process. */
{
  if (transmit_input_stream != NULL)
    fclose (transmit_input_stream);

  if (transmit_output_stream != NULL)
    fclose (transmit_output_stream);

  if (process_id != 0)
    kill (process_id, SIGTERM);

  transmit_input_stream = transmit_output_stream = NULL;
  process_id = 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t transmit (value_t argument)
/* Start the Malaga transmit process whose name is in the environment variable
 * MALAGA_TRANSMIT if it is not already running.
 * Convert <argument> to text, send it to the transmit process and get answer.
 * Convert the answer into internal value format and return it. */
{
  char buffer[BUFFER_SIZE];

  start_transmit_process ();

  /* Send <argument> to transmit process. */
  value_as_text (buffer, argument, buffer + BUFFER_SIZE, TRUE);
  fprintf (transmit_output_stream, "%s\n", buffer);
  fflush (transmit_output_stream);

  /* Read result from transmit process. */
  read_line (transmit_input_stream, buffer, BUFFER_SIZE);

  /* Convert result to value. */
  set_scanner_input (buffer);
  top = 0;
  parse_simple_value ();
  test_token (EOF);
  return stack[0];
}

/*---------------------------------------------------------------------------*/
