/* File "input.c":
 * Support reading and parsing input. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <stdlib.h>
#include "basic.h"

#undef GLOBAL
#define GLOBAL

#include "input.h"

/* constants ================================================================*/

#define QUERY_LENGTH 10

/* functions ================================================================*/

GLOBAL void parse_whitespace (string_t *input)
/* Read whitespace in *<input> and update *<input>. */
{
  while (IS_SPACE (**input))
    (*input)++;
}

/*---------------------------------------------------------------------------*/

GLOBAL long_t parse_integer (string_t *input)
/* Parse the next integer number from *<input> and update *<input>.
 * If there is no integer, an error is reported. */
{
  long_t number;
  bool_t negative;

  negative = (**input == '-');
  if (**input == '-')
    (*input)++;
  
  if (! IS_DIGIT (**input))
    error ("number expected as argument");
  
  number = 0;
  while (IS_DIGIT (**input))
  {
    if (number >= 214748364L)
      error ("number too big");
    
    number = 10 * number + (**input - '0');
    (*input)++;
  }
  
  if (**input != EOS && ! IS_SPACE (**input))
    error ("illegal char in number");
  
  parse_whitespace (input);
  return (negative ? -number : number);
}

/*---------------------------------------------------------------------------*/
  
GLOBAL double parse_double (string_t *input)
/* Parse the next double from *<input> and update *<input>.
 * If there is no double, an error is reported. */
{
  bool_t negate;
  double number;

  if (**input == '-')
  {
    negate = TRUE;
    (*input)++;
  }
  else
    negate = FALSE;

  if (! IS_DIGIT (**input))
    error ("number expected as argument");
  
  number = 0.0;
  while (IS_DIGIT (**input))
  {
    number = 10.0 * number + (**input - '0');
    (*input)++;
  }
  
  if (**input == '.')
  {
    double factor;

    (*input)++;
    if (! IS_DIGIT (**input))
      error ("missing digit after \".\"");

    factor = 1.0;
    while (IS_DIGIT (**input))
    {
      factor *= 0.1;
      number += factor * (**input - '0');
      (*input)++;
    }
  }

  if (**input == 'E' || **input == 'e')
  {
    double multiplier;
    short_t exponent;

    (*input)++;
    multiplier = (**input == '-') ? 0.1 : 10.0;

    if (**input == '-' || **input == '+')
      (*input)++;

    if (! IS_DIGIT (**input))
      error ("missing exponent");

    /* Read exponent number. */
    exponent = 0;
    while (IS_DIGIT (**input)) 
    {
      exponent = 10 * exponent + (**input - '0');
      if (exponent >= 300)
	error ("exponent too big");
      (*input)++;
    }
	  
    /* Multiply <number> by <multiplier> ^ <exponent>. */
    while (exponent > 0)
    {
      if (exponent & 1)
	number *= multiplier;
      multiplier = multiplier * multiplier;
      exponent = exponent / 2;
    }
  }

  if (**input != EOS && ! IS_SPACE (**input))
    error ("illegal char in number");

  if (negate) 
    number = -number;

  parse_whitespace (input);
  return number;
}
  
/*---------------------------------------------------------------------------*/

GLOBAL string_t parse_word (string_t *input)
/* If there is a word in *<input>, parse it up to the next space
 * and update *<input>. Return the word. It must be freed with "free".
 * If there's no word, report an error. */
{
  string_t word_start;
  string_t result;

  if (**input == EOS)
    error ("argument expected");

  if (**input == '\"') /* Read quoted word that contains spaces and quotes. */
  {
    long_t length, i;
    string_t p;
    
    /* Calculate the length of the word. */
    length = 0;
    (*input)++;
    word_start = *input;
    while (**input != EOS && **input != '\"')
    {
      if ((*input)[0] == '\\' && (*input)[1] != EOS) 
	(*input) += 2;
      else
	(*input)++;
      length++;
    }

      if (**input != '\"')
	error ("missing closing '\"'");
      
      (*input)++;
      
      /* Copy the word into a new string. */
      result = new_vector (sizeof (char), length + 1);
      p = word_start;
      for (i = 0; i < length; i++)
      {
	if (p[0] == '\\' && p[1] != EOS)
	{
	  result[i] = p[1];
	  p += 2;
	}
	else
	{
	  result[i] = p[0];
	  p++;
	}
      }
      result[length] = 0;
  } 
  else
  {
    word_start = *input;
    while (**input != EOS && ! IS_SPACE (**input))
      (*input)++;
    
    result = new_string_section (word_start, *input);
  }
  
  parse_whitespace (input);
  return result;
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t parse_yes_no (string_t *input)
/* Parse next word in <input>. It must be "yes", "no", "on" or "off".
 * Return TRUE iff next word is "yes" or "on". */
{
  string_t argument = parse_word (input);
  bool_t return_value;

  if (strcmp_no_case (argument, "yes") == 0 
      || strcmp_no_case (argument, "on") == 0)
    return_value = TRUE;
  else if (strcmp_no_case (argument, "no") == 0
	   || strcmp_no_case (argument, "off") == 0)
    return_value = FALSE;
  else
    error ("\"yes\"/\"on\" or \"no\"/\"off\" expected, not \"%s\"", argument);

  free (argument);
  return return_value;
}

/*---------------------------------------------------------------------------*/

GLOBAL void parse_end (string_t input)
/* Test if there are no more arguments in <input>. */
{
  if (*input != EOS)
    error ("unexpected argument: \"%s\"", input);
}

/*---------------------------------------------------------------------------*/

GLOBAL void read_line (FILE *stream, string_t buffer, long_t buffer_size)
/* Read user input from <stream> into <buffer> until eof or newline is met.
 * If input is longer than <buffer_size> chars, signal an error. */
{ 
  string_t end_of_line;

  end_of_line = buffer;
  while (TRUE) 
  {
    short_t c;
    
    c = fgetc (stream);
    if (c == '\n' || c == EOF)
      break;
    else if (end_of_line >= buffer + (buffer_size-1)) 
    {
      while (c != '\n' && c != EOF) 
	c = fgetc (stream);
      
      error ("input line too long");
    }
    else
      *end_of_line++ = c;
  }
  
  *end_of_line = EOS;
}

/*---------------------------------------------------------------------------*/
