/* $Id: visual.c,v 1.8 1999/05/05 20:26:07 marcus Exp $
******************************************************************************

   VT switch handling for Linux console

   Copyright (C) 1999 Marcus Sundber	[marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

/* We want SA_NOMASK from glibc 2.1 */
#define _GNU_SOURCE

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/linvtsw.h>

#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#ifdef HAVE_SYS_VT_H
#include <sys/vt.h>
#else
#include <linux/vt.h>
#endif
#ifdef HAVE_SYS_KD_H
#include <sys/kd.h>
#else
#include <linux/kd.h>
#endif
#ifdef HAVE_LINUX_TASKS_H
#include <linux/tasks.h>
#endif

#ifndef PID_MAX
#define PID_MAX 0x8000
#else
#if PID_MAX > 0x10000
/* Make sure we don't get stuck in a huge loop... */
#define PID_MAX 0x10000
#endif
#endif


static int refcount = 0;
static int prividx = -1;

typedef struct {
	ggi_linvtsw_arg vthandling;
	int 	vtfd;
	int	origvtnum;
	int	vt_switched_away;
	int	queue_expose_event;
} linvtsw_priv;

#define LINVTSW_PRIV(vis) ((linvtsw_priv*)LIBGGI_DRVPRIV((vis),prividx))

/*
******************************************************************************
 VT switch handling
******************************************************************************
*/

#ifdef SIGUNUSED
#define SWITCHSIG	SIGUNUSED
#else
/* Just a random signal which is rarely used */
#define SWITCHSIG	SIGXCPU
#endif


static ggi_visual *vtvisual;


static inline void
setsig(int sig, void (*func)(int))
{
	struct sigaction sa;

	sa.sa_handler = func;
	sa.sa_flags = 0;
#ifdef SA_RESTART
	/* Restart syscalls if possible */
	sa.sa_flags |= SA_RESTART;
#endif
#ifdef SA_NOMASK
	/* Don't block the signal in our handler */
	sa.sa_flags |= SA_NOMASK;
#endif
	sigemptyset(&sa.sa_mask);
	sigaction(sig, &sa, NULL);
	
	/* Make sure things like blocking select() are interrupted by the
	   switch signal */
	siginterrupt(sig, 1);
}


/* Release the VT */
static void
release_vt(void *arg)
{
	ggi_visual *vis = arg;
	linvtsw_priv *priv = LINVTSW_PRIV(vis);

	GGIDPRINT_MISC("acknowledging VT-switch\n");
	ioctl(priv->vtfd, VT_RELDISP, 1);
	priv->vt_switched_away = 1;
}


static void
switching_handler(int signo)
{
	volatile linvtsw_priv *priv;
	sigset_t newset, oldset;
	
	/* Block all signals while executing handler */
	sigfillset(&newset);
	sigprocmask(SIG_BLOCK, &newset, &oldset);
	
	priv = LINVTSW_PRIV(vtvisual);
	
	if (priv->vt_switched_away) {
		GGIDPRINT_MISC("acquire_vt START\n");

		/* Acknowledge the VT */
		ioctl(priv->vtfd, VT_RELDISP, VT_ACKACQ);

		if (priv->vthandling.onconsole) {
			/* An expose event will be queued in
			   GII_keyboard_poll */
			priv->queue_expose_event = 1;
		}

		if (priv->vthandling.switchback) {
			priv->vthandling.switchback(priv->vthandling.funcarg);
		}

		priv->vt_switched_away = 0;

		GGIDPRINT_MISC("acquire_vt DONE\n");
	} else {
		GGIDPRINT_MISC("release_vt START\n");

		if (priv->vthandling.switchaway) {
			priv->vthandling.switchaway(priv->vthandling.funcarg);
		}

		if (*priv->vthandling.autoswitch) {
			release_vt(vtvisual);
			if (*priv->vthandling.dohalt) {
				/* Suspend program until switch back */
				sigset_t tmpset = oldset;

				GGIDPRINT_MISC("release_vt SUSPEND\n");

				sigdelset(&tmpset, SWITCHSIG);
				sigprocmask(SIG_SETMASK, &tmpset, NULL);

				while (priv->vt_switched_away) {
					/* Note: we rely on the acquire signal
					   interrupting us  */
					pause();
				GGIDPRINT_MISC("release_vt INTERRUPTED\n");
				}
			}
		}
		GGIDPRINT_MISC("release_vt DONE\n");
	}
	sigprocmask(SIG_SETMASK, &oldset, NULL);
}


static int
restore_vt(linvtsw_priv *priv)
{
	if (priv->vthandling.vtnum != priv->origvtnum) {
		ioctl(priv->vtfd, VT_ACTIVATE, priv->origvtnum);
	}

	return 0;
}

static int
get_newcons(int fd)
{
	int vtnum;

	if (ioctl(fd, VT_OPENQRY, &vtnum) < 0 ||
	    vtnum <= 0) {
		fprintf(stderr,
"L/vtswitch: Unable to get a new virtual console\n");
		return -1;
	}
	
	return vtnum;
}


static char nopermstring[] = 
"L/vtswitch: You are not running on a virtual console and do not have\n\tpermission to open a new one\n";

static int
vtswitch_open(ggi_visual *vis)
{
	linvtsw_priv *priv = LINVTSW_PRIV(vis);
	char filename[80];
	int fd, neednewvt = 0;
	struct vt_stat vt_state;
	struct vt_mode qry_mode;

	/* Query free VT and current VT */
	fd = open("/dev/tty", O_WRONLY);
	if (fd < 0) {
		perror("L/vtswitch: open /dev/tty");
		return -1;
	}
	
	if (priv->vthandling.forcenew) {
		if ((priv->vthandling.vtnum = get_newcons(fd)) < 0) {
			close(fd);
			return -1;
		}
		neednewvt = 1;
	} else if (ioctl(fd, VT_GETSTATE, &vt_state) != 0) {
		GGIDPRINT_MISC("L/vtswitch: VT_GETSTATE failed\n");
		close(fd);
		fd = open("/dev/console", O_WRONLY);
		if (fd < 0) {
			fprintf(stderr, nopermstring);
			return -1;
		}
		if ((priv->vthandling.vtnum = get_newcons(fd)) < 0) {
			close(fd);
			return -1;
		}
		neednewvt = 1;
	} else {
		priv->vthandling.vtnum = vt_state.v_active;
	}
	if (neednewvt) {
		/* Detach from our old controlling tty. */
		int res = ioctl(fd, TIOCNOTTY);	
		GGIDPRINT_MISC("L/vtswitch: TIOCNOTTY = %d\n", res);
	}
	close(fd);

	if (neednewvt && geteuid() && setsid() < 0) {
		int i;
		int pid = getpid();
		int ppgid = getpgid(getppid());

		/* We're not running as root and setsid() failed, so we try
		   to set our process group ID that of our parrent. */
		setpgid(pid, ppgid);
		if (setsid() < 0) {
			/* We propably have a child process... */
			for (i = 1; i < 5; i++) {
				if (getpgid(pid + i) == pid) {
					setpgid(pid + i, ppgid);
				}
			}
			if (setsid() < 0) {
				/* That failed too. Now we scan the entire
				   process space after possible childs.
				   (This takes 32ms on a P225, so we can live
				   with doing this once at startup time...) */
				for (i = 2; i < PID_MAX; i++) {
					if (getpgid(i) == pid) {
						setpgid(i, ppgid);
					}
				}
				/* Last chance, if this fails we probably
				   can't switch VTs below. */
				setsid();
			}
		}
	}

	/* Open VT */
	sprintf(filename, "/dev/tty%d", priv->vthandling.vtnum);

	priv->vtfd = open(filename, O_WRONLY);
	if (priv->vtfd < 0) {
		/* Try devfs style device */
		sprintf(filename, "/dev/vc/%d", priv->vthandling.vtnum);
		priv->vtfd = open(filename, O_WRONLY);
		if (priv->vtfd < 0) {
			fprintf(stderr, "L/vtswitch: open %s: %s\n",
				filename, strerror(errno));
			priv->vtfd = -1;
			return -1;
		}
	}

        if (ioctl(priv->vtfd, VT_GETSTATE, &vt_state) < 0) {
		fprintf(stderr, "L/vtswitch: unable to get current console\n");
		close(priv->vtfd);
		return -1;
	}
	priv->origvtnum = vt_state.v_active;
        if (priv->vthandling.vtnum != vt_state.v_active) {
		if (ioctl(priv->vtfd, VT_ACTIVATE, priv->vthandling.vtnum)) {
			fprintf(stderr, nopermstring);
			close(priv->vtfd);
			return -1;
		}
		while (ioctl(priv->vtfd, VT_WAITACTIVE,
			     priv->vthandling.vtnum) < 0) {
			ggUSleep(150000);
		}
	}

	GGIDPRINT_MISC("L/vtswitch: Using VT %d.\n", priv->vthandling.vtnum);
	
	if (priv->vthandling.onconsole) {
		/* Disable normal text on the console */
		ioctl(priv->vtfd, KDSETMODE, KD_GRAPHICS);
	}
	
	/* Unfortunately, we need to take control over VT switching like
	 * Xfree and SVGAlib does.  Sigh.
	 */
	ioctl(priv->vtfd, VT_GETMODE, &qry_mode);

	qry_mode.mode = VT_PROCESS;
	qry_mode.relsig = SWITCHSIG;
	qry_mode.acqsig = SWITCHSIG;
	
	priv->vt_switched_away = 0;
	vtvisual = vis;

	setsig(SWITCHSIG, switching_handler);

	if (ioctl(priv->vtfd, VT_SETMODE, &qry_mode) < 0) {
		perror("L/vtswitch: Setting VT mode failed");
		restore_vt(priv);
		close(priv->vtfd);
		priv->vtfd = -1;
		return -1;
	}
	
	GGIDPRINT_MISC("L/vtswitch: open OK.\n");

	return 0;
}


static void
vtswitch_close(ggi_visual *vis)
{
	linvtsw_priv *priv = LINVTSW_PRIV(vis);
	struct vt_mode qry_mode;

	if (priv->vtfd >= 0) {
		if (priv->vthandling.onconsole) {
			ioctl(priv->vtfd, KDSETMODE, KD_TEXT);
		}
		if (ioctl(priv->vtfd, VT_GETMODE, &qry_mode) == 0) {
			qry_mode.mode = VT_AUTO;
			ioctl(priv->vtfd, VT_SETMODE, &qry_mode);
		}
		signal(SWITCHSIG, SIG_DFL);
		restore_vt(priv);

		close(priv->vtfd);
		priv->vtfd = -1;
	}

	GGIDPRINT_MISC("L/vtswitch: close OK.\n");
}

int GGIdlinit(ggi_visual *vis, const char *args, void *argptr)
{
	int idx = -1;
	ggi_linvtsw_arg *myargs = (ggi_linvtsw_arg *) argptr;
	linvtsw_priv *priv;
	
	if (myargs == NULL) {
		ggiPanic("Target tried to use linvtsw helper in a wrong way!\n");
	}
	
	myargs->vtnum = -1;

	ggLock(_ggi_global_lock);
	if (refcount == 0) {
		idx = _ggi_alloc_drvpriv();
		if (idx < 0) {
			ggUnlock(_ggi_global_lock);
			return GGI_DL_ERROR;
		}
		refcount++;
		prividx = idx;
	}
	ggUnlock(_ggi_global_lock);
	
	priv = malloc(sizeof(linvtsw_priv));
	if (priv == NULL) {
		refcount--;
		if (!refcount) {
			_ggi_free_drvpriv(prividx);
		}
		return GGI_DL_ERROR;
	}
	priv->vthandling = *myargs;
	LINVTSW_PRIV(vis) = priv;

	if (vtswitch_open(vis) != 0) {
		free(priv);
		refcount--;
		if (!refcount) {
			_ggi_free_drvpriv(prividx);
		}
		return GGI_DL_ERROR;
	}
	myargs->vtnum = priv->vthandling.vtnum;
	myargs->doswitch = release_vt;

	return 0;
}

int GGIdlcleanup(ggi_visual *vis)
{
	/* Make sure we're only called once */
	if (refcount == 0 || LINVTSW_PRIV(vis) == NULL) return 0;
	
	vtswitch_close(vis);

	free(LINVTSW_PRIV(vis));
	LINVTSW_PRIV(vis) = NULL;

	ggLock(_ggi_global_lock);
	refcount--;
	if (refcount == 0) {
		_ggi_free_drvpriv(prividx);
		prividx = -1;
	}
	ggUnlock(_ggi_global_lock);
                
	return 0;
}
		
#include <ggi/internal/ggidlinit.h>
