/*
This is part of the audio CD player library
Copyright (C)1998-99 Tony Arcieri

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.
*/

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <pwd.h>
#include <errno.h>
#include <unistd.h>
#include <time.h>

#define __LIBCDAUDIO_INTERNAL

#ifndef INADDR_NONE
#define INADDR_NONE 0xFFFFFFFF
#endif

#include <cdaudio.h>
#include <cddb.h>

/* Static function prototypes */
static int cddb_sum(long val);
static int cddb_serverlist_process_line(char *line, struct cddb_conf *conf, struct cddb_serverlist *list, struct cddb_server *proxy);
static int cddb_process_line(char *line, struct __unprocessed_disc_data *data);
static int cddb_sites_process_line(char *line, struct cddb_host *host);

/* CDDB message for reporting errors */
char cddb_message[256];
int parse_disc_artist = 1;
int cddb_submit_method = CDDB_SUBMIT_EMAIL;
char *cddb_submit_email_address = CDDB_EMAIL_SUBMIT_ADDRESS;

/* CDDB sum function */
static int
#if __STDC__
cddb_sum(long val)
#else
cddb_sum(val)
   long val;
#endif
{
   char *bufptr, buf[16];
   int ret = 0;
   
   snprintf(buf, 16, "%lu", val);
   for(bufptr = buf; *bufptr != '\0'; bufptr++)
     ret += (*bufptr - '0');
   
   return ret;
}

/* Produce CDDB ID for the CD in the device referred to by cd_desc */
unsigned long 
#if __STDC__
__internal_cddb_discid(struct disc_info disc)
#else
__internal_cddb_discid(disc)
   struct disc_info disc;
#endif
{
   int index, tracksum = 0, discid;
   
   for(index = 0; index < disc.disc_total_tracks; index++)
     tracksum += cddb_sum(disc.disc_track[index].track_pos.minutes * 60 + disc.disc_track[index].track_pos.seconds);
   
   discid = (disc.disc_length.minutes * 60 + disc.disc_length.seconds) - (disc.disc_track[0].track_pos.minutes * 60 + disc.disc_track[0].track_pos.seconds);
      
   return ((tracksum % 0xFF) << 24 | discid << 8 | disc.disc_total_tracks) & 0xFFFFFFFF;
}

unsigned long
#if __STDC__
cddb_discid(int cd_desc)
#else
cddb_discid(cd_desc)
   int cd_desc;
#endif
{
   struct disc_info disc;
   
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
   
   if(!disc.disc_present)
     return -1;
   
   return __internal_cddb_discid(disc);
}

/* Transform a URL in the CDDB configuration file into a valid hostname */
int
#if __STDC__
cddb_process_url(struct cddb_host *host, const char *url)
#else
cddb_process_url(host, url)
   struct cddb_host *host;
   const char *url;
#endif
{
   int index = 0;
   char urltmp[512];
   char hostbuffer[256];
   char portbuffer[256], *portptr;
   
   if(strchr(url, ':') == NULL) {
      strncpy(cddb_message, "Invalid URL", 256);
      return -1;
   }
   strncpy(urltmp, url, 512);
   strtok(urltmp, ":");
   if(strcmp(urltmp, "cddbp") == 0)
     host->host_protocol = CDDB_MODE_CDDBP;
   else if(strcmp(urltmp, "http") == 0)
     host->host_protocol = CDDB_MODE_HTTP;
   else {
      snprintf(cddb_message, 256, "%s: invalid protocol", urltmp);
      return -1;
   }
   
   strncpy(hostbuffer, (char *)(strtok(NULL, ":") + 2), 256);
   if((portptr = strtok(NULL, ":")) != NULL) {
      strncpy(host->host_server.server_name, hostbuffer, 256);
      index = 0;
      do {
	portbuffer[index++] = portptr[0];
	portptr++;
      } while(portptr[0] != '/' && portptr[0] != '\n' && portptr[0] != '\0');
      portbuffer[index] = '\0';
      host->host_server.server_port = strtol(portbuffer, NULL, 10);
      if(portptr[0] == '/') portptr++;
      if(portptr[0] != '\n' && portptr[0] != '\0')
	strncpy(host->host_addressing, portptr, 256);
      else
	strncpy(host->host_addressing, "", 256);
   } else {
      while(hostbuffer[index] != '/' && hostbuffer[index] != '\n' && hostbuffer[index] != '\0')
	index++;
      if(hostbuffer[index] == '/')
	strncpy(host->host_addressing, (char *)hostbuffer + index + 1, 256);
      else
        strncpy(host->host_addressing, "", 256);
      hostbuffer[index] = '\0';
      strncpy(host->host_server.server_name, hostbuffer, 256);
      if(host->host_protocol == CDDB_MODE_HTTP)
	host->host_server.server_port = HTTP_DEFAULT_PORT;
      else
	host->host_server.server_port = CDDBP_DEFAULT_PORT;
   }
	
   return 0;
}

/* Process a line in a CDDB configuration file */
static int
#if __STDC__
cddb_serverlist_process_line(char *line, struct cddb_conf *conf, struct cddb_serverlist *list, struct cddb_server *proxy)
#else
cddb_serverlist_process_line(line, conf, list, proxy)
   char *line;
   struct cddb_conf *conf;
   struct cddb_serverlist *list;
   struct cddb_server *proxy;
#endif
{
   struct cddb_host proxy_host;
   char *var, *value, procbuffer[256];

   if(strchr(line, '=') == NULL)
     return 0;
   
   line[strlen(line) - 1] = '\0';

   var = strtok(line, "=");
   if(var == NULL)
     return 0;
   value = strtok(NULL, "=");
   
   if(value == NULL)
     value = "";

   if(strcasecmp(var, "ACCESS") == 0) {
      if(strncasecmp(value, "LOCAL", 2) == 0)
        conf->conf_access = CDDB_ACCESS_LOCAL;
      else
        conf->conf_access = CDDB_ACCESS_REMOTE;
   } else if(strcasecmp(var, "PROXY") == 0) {
      if(cddb_process_url(&proxy_host, value) < 0)
        return -1;
      conf->conf_proxy = CDDB_PROXY_ENABLED;
      memcpy(proxy, &proxy_host.host_server, sizeof(struct cddb_server));
   } else if(strcasecmp(var, "SERVER") == 0) {
      if(list->list_len >= CDDB_MAX_SERVERS)
        return 0;
      strncpy(procbuffer, value, 256);
      if(strchr(procbuffer, ' ') != NULL) {
         strtok(procbuffer, " ");
         value = strtok(NULL, " ");
      } else
        value = NULL;
      if(cddb_process_url(&list->list_host[list->list_len], procbuffer) != -1) {
	 if(value != NULL && strcmp(value, "CDI") == 0)
           list->list_host[list->list_len].host_protocol = CDINDEX_MODE_HTTP;
	 else if(value != NULL && strcmp(value, "COVR") == 0)
	   list->list_host[list->list_len].host_protocol = COVERART_MODE_HTTP;
         list->list_len++;
      }
   }

   return 0;
}

/* Read ~/.cddbrc */
int
#if __STDC__
cddb_read_serverlist(struct cddb_conf *conf, struct cddb_serverlist *list, struct cddb_server *proxy)
#else
cddb_read_serverlist(conf, list, proxy)
   struct cddb_conf *conf;
   struct cddb_serverlist *list;
   struct cddb_server *proxy;
#endif
{
   FILE *cddbconf;
   int index;
   char inbuffer[256];
   char localconfpath[256];
   struct stat st;
   
   if(getenv("HOME") == NULL) {
      strncpy(cddb_message, "$HOME is not set!", 256);
      return -1;
   }
   
   list->list_len = 0;
   conf->conf_access = CDDB_ACCESS_REMOTE;
   conf->conf_proxy = CDDB_PROXY_DISABLED;
      
   snprintf(localconfpath, 256, "%s/.cdserverrc", getenv("HOME"));
   if(stat(localconfpath, &st) < 0)
     return 0;
	
   cddbconf = fopen(localconfpath, "r");
   
   while(!feof(cddbconf)) {
      fgets(inbuffer, 256, cddbconf);
      inbuffer[255] = '\0';
         
      for(index = 0; index < strlen(inbuffer); index++)
	if(inbuffer[index] == '#') {
           inbuffer[index] = '\0';
           break;
        }
      
      if(cddb_serverlist_process_line(inbuffer, conf, list, proxy) < 0)
	return -1;
   }
	
   fclose(cddbconf);
	
   return 0;
}

/* Write ~/.cddbrc */
int
#if __STDC__
cddb_write_serverlist(struct cddb_conf conf, struct cddb_serverlist list, struct cddb_server proxy)
#else
cddb_write_serverlist(conf, list, proxy)
   struct cddb_conf *conf;
   struct cddb_serverlist *list;
   struct cddb_server *proxy;
#endif
{
   FILE *cddbconf;
   int index;
   time_t timeval;
   char localconfpath[256];
   
   if(getenv("HOME") == NULL) {
      strncpy(cddb_message, "$HOME is not set!", 256);
      return -1;
   }
   
   snprintf(localconfpath, 256, "%s/.cdserverrc", getenv("HOME"));
   cddbconf = fopen(localconfpath, "w");
    
   timeval = time(NULL);
   fprintf(cddbconf, "# CD Server configuration file generated by %s %s.\n", PACKAGE, VERSION);
   fprintf(cddbconf, "# Created %s\n", ctime(&timeval));
   if(conf.conf_access == CDDB_ACCESS_REMOTE)
     fputs("ACCESS=REMOTE\n", cddbconf);
   else
     fputs("ACCESS=LOCAL\n", cddbconf);
  
   if(conf.conf_proxy == CDDB_PROXY_ENABLED)
     fprintf(cddbconf, "PROXY=http://%s:%d/\n", proxy.server_name, proxy.server_port);
   for(index = 0; index < list.list_len; index++) {
      switch(list.list_host[index].host_protocol) {
       case CDDB_MODE_HTTP:
	 fprintf(cddbconf, "SERVER=http://%s:%d/%s CDDB\n", list.list_host[index].host_server.server_name, list.list_host[index].host_server.server_port, list.list_host[index].host_addressing);
	 break;
       case CDDB_MODE_CDDBP:
	 fprintf(cddbconf, "SERVER=cddbp://%s:%d/ CDDB\n", list.list_host[index].host_server.server_name, list.list_host[index].host_server.server_port);
	 break;
       case CDINDEX_MODE_HTTP:
	 fprintf(cddbconf, "SERVER=http://%s:%d/%s CDI\n", list.list_host[index].host_server.server_name, list.list_host[index].host_server.server_port, list.list_host[index].host_addressing);
	 break;
       case COVERART_MODE_HTTP:
	 fprintf(cddbconf, "SERVER=http://%s:%s/%s COVR\n", list.list_host[index].host_server.server_name, list.list_host[index].host_server.server_port, list.list_host[index].host_addressing);
      }
   }
   
   fclose(cddbconf);
	
   return 0;
}

/* Convert numerical genre to text */
char
#if __STDC__
*cddb_genre(int genre)
#else
*cddb_genre(genre)
   int genre;
#endif
{
   switch(genre) {
    case CDDB_BLUES:
      return "blues";
    case CDDB_CLASSICAL:
      return "classical";
    case CDDB_COUNTRY:
      return "country";

   case CDDB_DATA:
      return "data";
    case CDDB_FOLK:
      return "folk";
    case CDDB_JAZZ:
      return "jazz";
    case CDDB_MISC:
      return "misc";
    case CDDB_NEWAGE:
      return "newage";
    case CDDB_REGGAE:
      return "reggae";
    case CDDB_ROCK:
      return "rock";
    case CDDB_SOUNDTRACK:
      return "soundtrack";
   }
   
   return "unknown";
}

/* Convert genre from text form into an integer value */
int
#if __STDC__
cddb_genre_value(char *genre)
#else
cddb_genre_value(genre)
   char *genre;
#endif
{
   if(strcmp(genre, "blues") == 0)
     return CDDB_BLUES;
   else if(strcmp(genre, "classical") == 0)
     return CDDB_CLASSICAL;
   else if(strcmp(genre, "country") == 0)
     return CDDB_COUNTRY;
   else if(strcmp(genre, "data") == 0)
     return CDDB_DATA;
   else if(strcmp(genre, "folk") == 0)
     return CDDB_FOLK;
   else if(strcmp(genre, "jazz") == 0)
     return CDDB_JAZZ;
   else if(strcmp(genre, "misc") == 0)
     return CDDB_MISC;
   else if(strcmp(genre, "newage") == 0)
     return CDDB_NEWAGE;
   else if(strcmp(genre, "reggae") == 0)
     return CDDB_REGGAE;
   else if(strcmp(genre, "rock") == 0)
     return CDDB_ROCK;
   else if(strcmp(genre, "soundtrack") == 0)
     return CDDB_SOUNDTRACK;
   else
     return CDDB_UNKNOWN;
}

int
#if __STDC__
cddb_connect(struct cddb_server *server)
#else
cddb_connect(server)
   struct cddb_server *server;
#endif
{
   int sock;
   struct sockaddr_in sin;
   struct hostent *host;
   
   sin.sin_family = AF_INET;
   sin.sin_port = htons(server->server_port);
     
   if((sin.sin_addr.s_addr = inet_addr(server->server_name)) == INADDR_NONE) {		  
      if((host = gethostbyname(server->server_name)) == NULL) {
	 strncpy(cddb_message, strerror(errno), 256);
	 return -1;
      }
      
      memcpy(&sin.sin_addr, host->h_addr, host->h_length);
   }
   
   if((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
      strncpy(cddb_message, strerror(errno), 256);
      return -1;
   }
   
   if(connect(sock, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
      strncpy(cddb_message, strerror(errno), 256);
      return -1;
   }
   
   return sock;
}

		  
/* Connect to a CDDB server and say hello */
int
#if __STDC__
cddb_connect_server(struct cddb_host host, struct cddb_server *proxy, struct cddb_hello hello, ...)
#else
cddb_connect_server(mode, host, proxy, hello, ...)
   int mode;
   struct cddb_host host;
   struct cddb_server proxy;
   struct cddb_hello hello
#endif
{
   int sock, token[3], http_string_len;
   char *http_string, syshostname[256], sysdomainname[256], outbuffer[256];
   struct passwd *pw;
   va_list arglist;
   
   va_start(arglist, hello);
   
   if(proxy != NULL) {
      if((sock = cddb_connect(proxy)) < 0)
        return -1;
   } else {	
      if((sock = cddb_connect(&host.host_server)) < 0)
        return -1;
   } 
	
   pw = getpwuid(getuid());
   gethostname(syshostname, 256);
   getdomainname(sysdomainname, 256);
   
   if(host.host_protocol == CDDB_MODE_HTTP) {
      http_string = va_arg(arglist, char *);
      http_string_len = va_arg(arglist, int);
      if(proxy != NULL)
	snprintf(http_string, http_string_len, "GET http://%s:%d/%s?hello=%s+%s.%s+%s+%s&proto=%d HTTP/1.0\n\n", host.host_server.server_name, host.host_server.server_port, host.host_addressing, pw->pw_name, syshostname, sysdomainname, hello.hello_program, hello.hello_version, CDDB_PROTOCOL_LEVEL);
      else
        snprintf(http_string, http_string_len, "GET /%s?hello=%s+%s.%s+%s+%s&proto=%d HTTP/1.0\n\n", host.host_addressing, pw->pw_name, syshostname, sysdomainname, hello.hello_program, hello.hello_version, CDDB_PROTOCOL_LEVEL);
   } else {
      if(cddb_read_token(sock, token) < 0) {
	 va_end(arglist);
         return -1;
      }
	   
      if(token[0] != 2) {
	 va_end(arglist);  
         return -1;
      }
	   
      snprintf(outbuffer, 256, "cddb hello %s %s.%s %s %s\n", pw->pw_name, syshostname, sysdomainname, hello.hello_program, hello.hello_version);
      write(sock, outbuffer, strlen(outbuffer));
	   
      if(cddb_read_token(sock, token) < 0) {
	 va_end(arglist);
	 return -1;
      }
	   
      if(token[0] != 2) {
	 va_end(arglist);
	 return -1;
      }
	   
      snprintf(outbuffer, 256, "proto %d\n", CDDB_PROTOCOL_LEVEL);
      write(sock, outbuffer, strlen(outbuffer));
	      
      if(cddb_read_token(sock, token) < 0) {
	 va_end(arglist);
	 return -1;
      }
   }
	
   va_end(arglist);
   return sock;
}

/* Generate the CDDB request string */
static int
#if __STDC__
cddb_generate_http_request(char *outbuffer, const char *cmd, char *http_string, int outbuffer_len)
#else
cddb_generate_http_request(outbuffer, cmd, http_string, outbuffer_len)
   char *outbuffer;
   const char *cmd;
   const char *http_string;
   int outbuffer_len;
#endif
{
   char getstring[512];
   char reqstring[512];
	
   if(strchr(http_string, '?') == NULL)
     return -1;
    
   strncpy(getstring, http_string, 512);
   strtok(getstring, "?");
   strncpy(reqstring, strtok(NULL, "?"), 512);
   
   snprintf(outbuffer, outbuffer_len, "%s?cmd=%s&%s\n", getstring, cmd, reqstring);
	
   return 0;
}

/* Skip HTTP header */
int
#if __STDC__
cddb_skip_http_header(int sock)
#else
cddb_skip_http_header(sock)
   int sock;
#endif
{
   char inchar;
   int len;
	
   do {
     len = 0;
     
     do {
       if(read(sock, &inchar, 1) < 1) {
	  strncpy(cddb_message, "Unexpected socket closure", 256);
	  return -1;
       }
       len++;
     } while(inchar != '\n');
   } while(len > 2);		  
	
   return 0;
}

/* Read a single line */
static int
#if __STDC__
cddb_read_line(int sock, char *inbuffer, int len)
#else
cddb_read_line(sock, inbuffer, len)
   int sock;
   char *inbuffer;
   int len;
#endif
{
   int index;
   char inchar;
   
   for(index = 0; index < len; index++) {
      read(sock, &inchar, 1);
      if(inchar == '\n') {
	 inbuffer[index] = '\0';
	 if(inbuffer[0] == '.')
	   return 1;
	 return 0;
      }
      inbuffer[index] = inchar;
   }
   
   return index;
}

/* Query the CDDB for the CD currently in the CD-ROM, and find the ID of the
   CD (which may or may not be the one generated) and what section it is
   under (genre) */
int
#if __STDC__
cddb_query(int cd_desc, int sock, int mode, struct cddb_query *query, ...)
#else
cddb_query(cd_desc, sock, mode, query)
   int cd_desc;
   int sock;
   int mode;
   struct cddb_query *query;
#endif
{
   unsigned long discid;
   int index, slashed = 0, token[3];
   struct disc_info disc;
   char offsetbuffer[1024], offsettmp[1024], outbuffer[1024], outtemp[1024], inbuffer[256], tmpbuffer[256], procbuffer[256], proctemp[256], artistbuffer[256];
   char *tmptr, *http_string, idtext[16];
   va_list arglist;
	
   va_start(arglist, query);
   query->query_matches = 0;
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
   
   if((discid = __internal_cddb_discid(disc)) < 0)
     return -1;
   
   if(mode == CDDB_MODE_HTTP) {
      http_string = va_arg(arglist, char *);
      
      snprintf(offsetbuffer, 1024, "%d", disc.disc_total_tracks);
      for(index = 0; index < disc.disc_total_tracks; index++) {
	 snprintf(offsettmp, 1024, "%s+%d", offsetbuffer, (disc.disc_track[index].track_pos.minutes * 60 + disc.disc_track[index].track_pos.seconds) * 75 + disc.disc_track[index].track_pos.frames);
         strncpy(offsetbuffer, offsettmp, 1024);
      }
      snprintf(outtemp, 1024, "cddb+query+%08lx+%s+%d", discid, offsetbuffer, disc.disc_length.minutes * 60 + disc.disc_length.seconds);
      cddb_generate_http_request(outbuffer, outtemp, http_string, 1024);
   } else {	  
      snprintf(offsetbuffer, 1024, "%d", disc.disc_total_tracks);
      for(index = 0; index < disc.disc_total_tracks; index++) {
         snprintf(offsettmp, 1024, "%s %d", offsetbuffer, (disc.disc_track[index].track_pos.minutes * 60 + disc.disc_track[index].track_pos.seconds) * 75 + disc.disc_track[index].track_pos.frames);
         strncpy(offsetbuffer, offsettmp, 1024);
      }
      snprintf(outbuffer, 1024, "cddb query %08lx %s %d\n", discid, offsetbuffer, disc.disc_length.minutes * 60 + disc.disc_length.seconds);
   }
	
   va_end(arglist);
   write(sock, outbuffer, strlen(outbuffer));

   if(mode == CDDB_MODE_HTTP)
     cddb_skip_http_header(sock);
	
   if(cddb_read_line(sock, inbuffer, 256) < 0)
     return -1;
   strncpy(tmpbuffer, inbuffer, 256);
   token[0] = inbuffer[0] - 48;
   token[1] = inbuffer[1] - 48;
   token[2] = inbuffer[2] - 48;
	
   strncpy(cddb_message, (char *)inbuffer + 4, 256);
   
   if(token[0] != 2)
     return -1;

   if(token[1] == 0) {
      if(token[2] != 0) {
	 query->query_match = QUERY_NOMATCH;
	 return 0;
      }
	   
      query->query_match = QUERY_EXACT;
      query->query_matches = 1;
      slashed = 0;
      if(strchr(tmpbuffer, '/') != NULL && parse_disc_artist) {
	      strtok(tmpbuffer, "/");
	      strncpy(query->query_list[0].list_title, strtok(NULL, "/") + 1, 64);
	      slashed = 1;
      }
      
      strtok(tmpbuffer, " ");
      query->query_list[0].list_genre = cddb_genre_value(strtok(NULL, " "));
      strncpy(idtext, strtok(NULL, " "), 16);
      sscanf(idtext, "%xd", &query->query_list[0].list_id);
      strncpy(artistbuffer, strtok(NULL, " "), 256);
      while((tmptr = strtok(NULL, " ")) != NULL) {
	 snprintf(procbuffer, 256, "%s %s", artistbuffer, tmptr);
	 strncpy(artistbuffer, procbuffer, 256);
      }
      if(slashed)
	 strncpy(query->query_list[0].list_artist, procbuffer, 64);
      else {
	 strncpy(query->query_list[0].list_title, procbuffer, 64);
	 strncpy(query->query_list[0].list_artist, "", 64);
      }
   } else if(token[1] == 1) {
      if(token[2] == 0)
	query->query_match = QUERY_EXACT;
      else if(token[2] == 1)
        query->query_match = QUERY_INEXACT;
      else {	     
	query->query_match = QUERY_NOMATCH;
        return 0;
      }
	   
      query->query_matches = 0;
      while(!cddb_read_line(sock, inbuffer, 256)) {
	 slashed = 0;
	 if(strchr(inbuffer, '/') != NULL) {		
	    strtok(inbuffer, "/");
	    strncpy(query->query_list[query->query_matches].list_title, strtok(NULL, "/") + 1, 64);
	    slashed = 1;
	 }
	 query->query_list[query->query_matches].list_genre = cddb_genre_value(strtok(inbuffer, " "));
	 strncpy(idtext, strtok(NULL, " "), 16);
	 sscanf(idtext, "%xd", &query->query_list[query->query_matches].list_id);
	 strncpy(procbuffer, strtok(NULL, " "), 256);
	 while((tmptr = strtok(NULL, " ")) != NULL) {
	    snprintf(proctemp, 256, "%s %s", procbuffer, tmptr);
	    strncpy(procbuffer, proctemp, 256);
	 }
	 if(slashed)
	    strncpy(query->query_list[query->query_matches++].list_artist, procbuffer, 64);
	 else {
	    strncpy(query->query_list[query->query_matches].list_title, procbuffer, 64);
	    strncpy(query->query_list[query->query_matches++].list_artist, "", 64);
	 }		
      }
   } else {
      query->query_match = QUERY_NOMATCH;
      return 0;
   }
   
   return 0;
}

static int
#if __STDC__
cddb_process_line(char *line, struct __unprocessed_disc_data *data)
#else
cddb_process_line(line, data)
   char *line;
   struct __unprocessed_disc_data *data;
#endif
{
   char *var, *value;
   
   line[strlen(line) - 1] = '\0';
   if(strstr(line, "Revision") != NULL) {
      strtok(line, ":");
      data->data_revision = strtol(strtok(NULL, ":") + 1, NULL, 10);
      return 0;
   }	  
	
   if(strchr(line, '=') == NULL)
     return 0;
	
   var = strtok(line, "=");
   if(var == NULL)
     return 0;
   value = strtok(NULL, "=");

   if(value == NULL)
     value = "";

   if(strcmp(var, "DTITLE") == 0) {
      if(data->data_title_index >= MAX_EXTEMPORANEOUS_LINES)
        return 0;
      strncpy(data->data_title[data->data_title_index++], value, 80);
   } else if(strncmp(var, "TTITLE", 6) == 0) {
      if(data->data_track[strtol((char *)var + 6, NULL, 10)].track_name_index >= MAX_EXTEMPORANEOUS_LINES)
        return 0;
      strncpy(data->data_track[strtol((char *)var + 6, NULL, 10)].track_name[data->data_track[strtol((char *)var + 6, NULL, 10)].track_name_index++], value, 80);
   } else if(strcmp(var, "EXTD") == 0) {
      if(data->data_extended_index >= MAX_EXTENDED_LINES)
        return 0;
      strncpy(data->data_extended[data->data_extended_index++], value, 80);
   } else if(strncmp(var, "EXTT", 4) == 0) {
      if(data->data_track[strtol((char *)var + 4, NULL, 10)].track_extended_index >= MAX_EXTENDED_LINES)
        return 0;
      strncpy(data->data_track[strtol((char *)var + 4, NULL, 10)].track_extended[data->data_track[strtol((char *)var + 4, NULL, 10)].track_extended_index++], value, 80);
   }

   return 0;
}

/* Read the actual CDDB entry */
int
#if __STDC__
cddb_read(int cd_desc, int sock, int mode, struct cddb_entry entry, struct disc_data *data, ...)
#else
   int cd_desc;
   int sock;
   struct cddb_entry entry;
   struct disc_data *data;
#endif
{
   int index, token[3];
   char outbuffer[512], outtemp[512], inbuffer[512], *http_string;
   struct disc_info disc;
   struct __unprocessed_disc_data indata;
   va_list arglist;
   
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
 
   if((indata.data_id = __internal_cddb_discid(disc)) < 0)
     return -1;
      
   va_start(arglist, data);
   indata.data_genre = entry.entry_genre;
   indata.data_title_index = 0;
   indata.data_extended_index = 0;
   for(index = 0; index < disc.disc_total_tracks; index++) {
      indata.data_track[index].track_name_index = 0;
      indata.data_track[index].track_extended_index = 0;
   }
  
   if(mode == CDDB_MODE_HTTP) {
      http_string = va_arg(arglist, char *);
      snprintf(outtemp, 512, "cddb+read+%s+%08lx", cddb_genre(entry.entry_genre), entry.entry_id);
      cddb_generate_http_request(outbuffer, outtemp, http_string, 512);
   } else 
      snprintf(outbuffer, 512, "cddb read %s %08lx\n", cddb_genre(entry.entry_genre), entry.entry_id);
   
   va_end(arglist);
	
   write(sock, outbuffer, strlen(outbuffer));
   
   if(mode == CDDB_MODE_HTTP)
     cddb_skip_http_header(sock);
	
   if(cddb_read_token(sock, token) < 0)
     return -1;
   
   if(token[0] != 2 && token[1] != 1)
     return -1;
	
   while(!cddb_read_line(sock, inbuffer, 512))
     cddb_process_line(inbuffer, &indata);

   data_format_input(data, indata, disc.disc_total_tracks);
   data->data_revision++;
 
   return 0;
}

/* Process a single line in the sites list */
static int
#if __STDC__
cddb_sites_process_line(char *line, struct cddb_host *host)
#else
cddb_sites_process_line(line, host)
   char *line;
   struct cddb_hostelement host;
#endif
{
   char *ptr;
   
   strncpy(host->host_server.server_name, strtok(line, " "), 256);
   
   ptr = strtok(NULL, " ");
   if(strcasecmp(ptr, "cddbp") == 0)
     host->host_protocol = CDDB_MODE_CDDBP;
   else if(strcasecmp(ptr, "http") == 0)
     host->host_protocol = CDDB_MODE_HTTP;
   else return -1;
	
   host->host_server.server_port = strtol(strtok(NULL, " "), NULL, 10);
   
   ptr = strtok(NULL, " ");
   if(strcmp(ptr, "-") != 0)
     strncpy(host->host_addressing, ptr + 1, 256);
   else
     strncpy(host->host_addressing, "", 256);
	
   return 0;
}

/* Read the CDDB sites list */
int
#if __STDC__
cddb_sites(int sock, int mode, struct cddb_serverlist *list, ...)
#else
cddb_sites(sock, mode, list, ..)
   int sock;
   int mode;
   struct cddb_serverlist *list;
#endif
{
   int token[3], http_string_len;
   char outbuffer[512], inbuffer[256], *http_string;
   va_list arglist;
   
   va_start(arglist, list);
   if(mode == CDDB_MODE_HTTP) {
      http_string = va_arg(arglist, char *);
      http_string_len = va_arg(arglist, int);
      cddb_generate_http_request(outbuffer, "sites", http_string, 512);
   } else
      strcpy(outbuffer, "sites\n");
	
   write(sock, outbuffer, strlen(outbuffer));
   
   if(mode == CDDB_MODE_HTTP)
     cddb_skip_http_header(sock);
	
   if(cddb_read_token(sock, token) < 0)
     return -1;
   
   if(token[0] != 2)
     return -1;
	
   list->list_len = 0;
   
   while(!cddb_read_line(sock, inbuffer, 256))
     if(cddb_sites_process_line(inbuffer, &list->list_host[list->list_len]) != -1)
       list->list_len++;
   
   return 0;
}

/* Terminate the connection */
int
#if __STDC__
cddb_quit(int sock)
#else
cddb_quit(sock)
   int sock;
#endif
{
   char outbuffer[8];
   
   strcpy(outbuffer, "quit\n");
   write(sock, outbuffer, strlen(outbuffer));
   
   shutdown(sock, 2);
   close(sock);
   
   return 0;
}

/* Return the numerical value of a reply, allowing us to quickly check if
   anything went wrong */
int
#if __STDC__
cddb_read_token(int sock, int token[3])
#else
cddb_read_token(sock, token)
   int sock;
   token[3];
#endif
{
   char inbuffer[512];
   
   if(cddb_read_line(sock, inbuffer, 512) < 0)
      return -1;
      
   if(strncmp(inbuffer, "<!DOC", 5) == 0) {
      strncpy(cddb_message, "404 CDDB CGI not found", 256);
      return -1;
   }
   
   token[0] = inbuffer[0] - 48;
   token[1] = inbuffer[1] - 48;
   token[2] = inbuffer[2] - 48;
   
   strncpy(cddb_message, (char *)inbuffer + 4, 256);
   
   return 0;
}

/* This is the function for completely automated CDDB operation */
int
#if __STDC__
cddb_read_data(int cd_desc, struct disc_data *data)
#else
cddb_read_data(cd_desc, data)
   int cd_desc;
   struct disc_data *data;
#endif
{
   int sock = -1, index;
   char http_string[512];
   struct disc_info disc;
   struct cddb_entry entry;
   struct cddb_hello hello;
   struct cddb_query query;
   struct cddb_conf conf;
   struct cddb_server *proxy_ptr;
   struct cddb_serverlist list;
   
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
   
   if(!disc.disc_present)
     return -1;

   proxy_ptr = (struct cddb_server *)malloc(sizeof(struct cddb_server));
   cddb_read_serverlist(&conf, &list, proxy_ptr);
   if(!conf.conf_access) {
     free(proxy_ptr);	  
     return -1;
   }
   if(!conf.conf_proxy) {
     free(proxy_ptr);
     proxy_ptr = NULL;
   }
   
   if(list.list_len < 1)
     return -1;
   
   strncpy(hello.hello_program, PACKAGE, 256);
   strncpy(hello.hello_version, VERSION, 256);
   
   index = 0;
	
   /* Connect to a server */
   do {
      switch(list.list_host[index].host_protocol) { 
       case CDDB_MODE_CDDBP: 
	 sock = cddb_connect_server(list.list_host[index++], proxy_ptr, hello);
	 break;
        case CDDB_MODE_HTTP:
	 sock = cddb_connect_server(list.list_host[index++], proxy_ptr, hello, http_string, 512);
	 break;
       case CDINDEX_MODE_HTTP:
	 sock = cdindex_connect_server(list.list_host[index++], proxy_ptr, http_string, 512);
      }
   } while(index < list.list_len && sock == -1);
   
   if(sock < 0) {
     if(conf.conf_proxy) free(proxy_ptr);
     return -1;
   }
	
   index--;
	
   /* CDDB Query, not nessecary for CD Index operations */
   switch(list.list_host[index].host_protocol) {
     case CDDB_MODE_CDDBP:
       if(cddb_query(cd_desc, sock, CDDB_MODE_CDDBP, &query) < 0) {
	 if(conf.conf_proxy) free(proxy_ptr);
         return -1;
       }
       break;
     case CDDB_MODE_HTTP:
       if(cddb_query(cd_desc, sock, CDDB_MODE_HTTP, &query, http_string) < 0) {
	 if(conf.conf_proxy) free(proxy_ptr);
	 return -1;
       }
       shutdown(sock, 2);
       close(sock);
       
       /* We must now reconnect to execute the next command */
       if((sock = cddb_connect_server(list.list_host[index], proxy_ptr, hello, http_string, 512)) < 0)
	 return -1;
      
      break;
   }
   
   if(conf.conf_proxy) free(proxy_ptr);
	
   /* Since this is an automated operation, we'll assume inexact matches are
      correct. */
   
   entry.entry_id = query.query_list[0].list_id;
   entry.entry_genre = query.query_list[0].list_genre;
   
   /* Read operation */
   switch(list.list_host[index].host_protocol) {
     case CDDB_MODE_CDDBP:
       if(cddb_read(cd_desc, sock, CDDB_MODE_CDDBP, entry, data) < 0)
         return -1;
	   
       cddb_quit(sock);
       break;
     case CDDB_MODE_HTTP:
       if(cddb_read(cd_desc, sock, CDDB_MODE_HTTP, entry, data, http_string) < 0)
	 return -1;
       
       shutdown(sock, 2);
       close(sock);
       break;
    case CDINDEX_MODE_HTTP:
      if(cdindex_read(cd_desc, sock, data, http_string) < 0)
	return -1;
      
      shutdown(sock, 2);
      close(sock);
   }
   
   return 0;
}

/* Generate an entry for when CDDB is disabled/not working */
int
#if __STDC__
cddb_generate_unknown_entry(int cd_desc, struct disc_data *data)
#else
cddb_generate_unknown_entry(cd_desc, data)
   int cd_desc;
   struct disc_data *data;
#endif
{
   int index;
   struct disc_info disc;

   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
     
   if((data->data_id = __internal_cddb_discid(disc)) < 0)
     return -1;
   
   if(__internal_cdindex_discid(disc, data->data_cdindex_id, CDINDEX_ID_SIZE) < 0)
     return -1;
   
   strcpy(data->data_title, "Unknown");
   strcpy(data->data_artist, "Unknown");
   data->data_genre = CDDB_UNKNOWN;
   for(index = 0; index < disc.disc_total_tracks; index++) {
      strcpy(data->data_track[index].track_name, "Unknown");
   }
	
   return 0;
}

/* Generate an entry using CDDB if it's available, or Unknowns if it's not */
int
#if __STDC__
cddb_generate_new_entry(int cd_desc, struct disc_data *data)
#else
cddb_generate_new_entry(cd_desc, data)
   int cd_desc;
   struct disc_data *data;
#endif
{
   struct disc_data outdata;
   
   if(cddb_read_data(cd_desc, data) < 0)
     cddb_generate_unknown_entry(cd_desc, data);
   
   memcpy(&outdata, data, sizeof(struct disc_data));
   cddb_write_disc_data(cd_desc, outdata);
	
   return 0;
}

/* Read from the local database, using CDDB if there isn't an entry cached */
int
#if __STDC__
cddb_read_disc_data(int cd_desc, struct disc_data *outdata)
#else
cddb_read_disc_data(cd_desc, outdata)
   int cd_desc;
   struct disc_data *outdata;
#endif
{
   FILE *cddb_data;
   int index;
   char root_dir[256], file[256], inbuffer[512];
   struct disc_info disc;
   struct stat st;
   struct __unprocessed_disc_data data;
   
   if(getenv("HOME") == NULL) {
      strncpy(cddb_message, "$HOME is not set!", 256);
      return -1;
   }
   
   snprintf(root_dir, 256, "%s/.cddb", getenv("HOME"));
   
   if(stat(root_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else {
         cddb_generate_new_entry(cd_desc, outdata);
	 
	 return 0;
      }
   } else {
      if(!S_ISDIR(st.st_mode)) {
	 errno = ENOTDIR;
	 return -1;
      }
   }
   
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
      
   if((data.data_id = __internal_cddb_discid(disc)) < 0)
     return -1;
   
   if(cdindex_discid(cd_desc, data.data_cdindex_id, CDINDEX_ID_SIZE) < 0)
     return -1;
   
   data.data_art.art_present = 0;
   data.data_title_index = 0;
   data.data_extended_index = 0;
   for(index = 0; index < disc.disc_total_tracks; index++) {
      data.data_track[index].track_name_index = 0;
      data.data_track[index].track_extended_index = 0;
   }
   
   for(index = 0; index < 12; index++) {
      snprintf(file, 256, "%s/%s/%08lx", root_dir, cddb_genre(index), data.data_id);
      if(stat(file, &st) == 0) {
	 cddb_data = fopen(file, "r");
	 while(!feof(cddb_data)) {
	    fgets(inbuffer, 512, cddb_data);			   
	    cddb_process_line(inbuffer, &data);
	 }
	 
	 data.data_genre = index;
	 fclose(cddb_data);
	 
	 data_format_input(outdata, data, disc.disc_total_tracks);
	 
	 return 0;
      }
   }
   
   if(cddb_read_data(cd_desc, outdata) < 0)
     cddb_generate_new_entry(cd_desc, outdata);

   return 0;
}

/* Write to the local cache */
int
#if __STDC__
cddb_write_disc_data(int cd_desc, struct disc_data indata)
#else
cddb_write_disc_data(cd_desc, indata)
   int cd_desc;
   struct disc_data indata;
#endif
{
   FILE *cddb_data;
   int index, tracks;
   char root_dir[256], genre_dir[256], file[256];
   struct stat st;
   struct disc_info disc;
   struct __unprocessed_disc_data data;
   
   if(getenv("HOME") == NULL) {
      strncpy(cddb_message, "$HOME is not set!", 256);
      return -1;
   }
   
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
   
   data_format_output(&data, indata, disc.disc_total_tracks);
   
   snprintf(root_dir, 256, "%s/.cddb", getenv("HOME"));
   snprintf(genre_dir, 256, "%s/%s", root_dir, cddb_genre(data.data_genre));
   snprintf(file, 256, "%s/%08lx", genre_dir, data.data_id);
   
   if(stat(root_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else
	mkdir(root_dir, 0755);
   } else {
      if(!S_ISDIR(st.st_mode)) {
	 errno = ENOTDIR;
	 return -1;
      }   
   }
   
   if(stat(genre_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else
	mkdir(genre_dir, 0755);
   } else {
      if(!S_ISDIR(st.st_mode)) {
	 errno = ENOTDIR;
	 return -1;
      }
   }
   
   if((cddb_data = fopen(file, "w")) == NULL)
     return -1;
   
   fprintf(cddb_data, "# xmcd CD database file generated by %s %s\n", PACKAGE, VERSION);
   fputs("# \n", cddb_data);
   fputs("# Track frame offsets:\n", cddb_data);
   for(index = 0; index < disc.disc_total_tracks; index++)
     fprintf(cddb_data, "#       %d\n", (disc.disc_track[index].track_pos.minutes * 60 + disc.disc_track[index].track_pos.seconds) * 75 + disc.disc_track[index].track_pos.frames);
   fputs("# \n", cddb_data);
   fprintf(cddb_data, "# Disc length: %d seconds\n", disc.disc_length.minutes * 60 + disc.disc_length.seconds);
   fputs("# \n", cddb_data);
   fprintf(cddb_data, "# Revision: %d\n", data.data_revision);
   fprintf(cddb_data, "# Submitted via: %s %s\n", PACKAGE, VERSION);
   fputs("# \n", cddb_data);
   fprintf(cddb_data, "DISCID=%08lx\n", data.data_id);
   for(index = 0; index < data.data_title_index; index++)
     fprintf(cddb_data, "DTITLE=%s\n", data.data_title[index]);
   for(tracks = 0; tracks < disc.disc_total_tracks; tracks++) {
      for(index = 0; index < data.data_track[tracks].track_name_index; index++)
	fprintf(cddb_data, "TTITLE%d=%s\n", tracks, data.data_track[tracks].track_name[index]);
   }
   if(data.data_extended_index == 0)
     fputs("EXTD=\n", cddb_data);
   else {
      for(index = 0; index < data.data_extended_index; index++)
	fprintf(cddb_data, "EXTD=%s\n", data.data_extended[index]);
   }
   
   for(tracks = 0; tracks < disc.disc_total_tracks; tracks++) {
      if(data.data_track[tracks].track_extended_index == 0)
	fprintf(cddb_data, "EXTT%d=\n", tracks);
      else {
	 for(index = 0; index < data.data_track[tracks].track_extended_index; index++)
	   fprintf(cddb_data, "EXTT%d=%s\n", tracks, data.data_track[tracks].track_extended[index]);
      }
   }	 
	 
   fputs("PLAYORDER=", cddb_data);
   
   fclose(cddb_data);
   
   return 0;
}

/* Delete an entry from the local cache based upon a data structure */
int
#if __STDC__
cddb_erase_entry(struct disc_data data)
#else
cddb_erase_entry(data)
   struct disc_data data;
#endif
{
   char root_dir[256], genre_dir[256], file[256];
   struct stat st;
   
   if(getenv("HOME") == NULL) {
      strncpy(cddb_message, "$HOME is not set!", 256);
      return -1;
   }
   
   snprintf(root_dir, 256, "%s/.cddb", getenv("HOME"));
   snprintf(genre_dir, 256, "%s/%s", root_dir, cddb_genre(data.data_genre));
   snprintf(file, 256, "%s/%08lx", genre_dir, data.data_id);
   
   if(stat(root_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else
	return 0;
   } else {
      if(!S_ISDIR(st.st_mode))
	return 0;  
   }
   
   if(stat(genre_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else
	return 0;
   } else {
      if(!S_ISDIR(st.st_mode))
	return 0;
   }
   
   if(unlink(file) < 0) {
      if(errno != ENOENT)
	return -1;
   }
   
   return 0;
}

/* Return the status of a CDDB entry */
int
#if __STDC__
cddb_stat_disc_data(int cd_desc, struct cddb_entry *entry)
#else
cddb_stat_disc_data(cd_desc, entry)
   int cd_desc;
   struct cddb_entry *entry;
#endif
{
   int index;
   struct disc_info disc;
   struct stat st;
   char root_dir[256], file[256];

   if(getenv("HOME") == NULL) {
      strncpy(cddb_message, "$HOME is not set!", 256);
      return -1;
   }
   
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
   
   if((entry->entry_id = __internal_cddb_discid(disc)) < 0)
     return -1;
   
   if(cdindex_discid(cd_desc, entry->entry_cdindex_id, CDINDEX_ID_SIZE) < 0)
     return -1;
   
   snprintf(root_dir, 256, "%s/.cddb", getenv("HOME"));
   
   if(stat(root_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else {
	 entry->entry_present = 0;
	 return 0;
      }
   } else {
      if(!S_ISDIR(st.st_mode)) {
	 errno = ENOTDIR;
	 return -1;
      }
   }
   
   for(index = 0; index < 12; index++) {
      snprintf(file, 256, "%s/%s/%08lx", root_dir, cddb_genre(index), entry->entry_id);
      if(stat(file, &st) == 0) {
	 entry->entry_timestamp = st.st_mtime;
	 entry->entry_present = 1;
	 entry->entry_genre = index;
	 
	 return 0;
      }
   }

   entry->entry_present = 0;
   
   return 0;
}

/* Wrapper for HTTP CDDB query */
int
#if __STDC__
cddb_http_query(int cd_desc, struct cddb_host host, struct cddb_hello hello, struct cddb_query *query)
#else
cddb_http_query(cd_desc, host, hello, query)
   int cd_desc;
   struct cddb_host host;
   struct cddb_hello hello;
   struct cddb_query *query;
#endif
{
   int sock;
   char http_string[512];
	
   if((sock = cddb_connect_server(host, NULL, hello, http_string, 512)) < 0)
     return -1;
	
   if(cddb_query(cd_desc, sock, CDDB_MODE_HTTP, query, http_string) < 0)
     return -1;
	
   shutdown(sock, 2);
   close(sock);
	
   return 0;
}

/* Wrapper for HTTP CDDB query using a proxy */
int
#if __STDC__
cddb_http_proxy_query(int cd_desc, struct cddb_host host, struct cddb_server proxy, struct cddb_hello hello, struct cddb_query *query)
#else
cddb_http_proxy_query(cd_desc, host, proxy, hello, query)
   int cd_desc;
   struct cddb_host host;
   struct cddb_server proxy;
   struct cddb_hello hello;
   struct cddb_query *query;
#endif
{
   int sock;
   char http_string[512];

   if((sock = cddb_connect_server(host, &proxy, hello, http_string, 512)) < 0)
     return -1;

   if(cddb_query(cd_desc, sock, CDDB_MODE_HTTP, query, http_string) < 0)
     return -1;

   shutdown(sock, 2);
   close(sock);

   return 0;
}

/* Wrapper for HTTP CDDB read */
int
#if __STDC__
cddb_http_read(int cd_desc, struct cddb_host host, struct cddb_hello hello, struct cddb_entry entry, struct disc_data *data)
#else
cddb_http_read(cd_desc, host, hello, entry, data)
   int cd_desc;
   struct cddb_host host;
   struct cddb_hello hello;
   struct cddb_entry entry;
   struct cddb_disc_data *data;
#endif
{
   int sock;
   char http_string[512];
   
   if((sock = cddb_connect_server(host, NULL, hello, http_string, 512)) < 0)
     return -1;
   
   if(cddb_read(cd_desc, sock, CDDB_MODE_HTTP, entry, data, http_string) < 0)
     return -1;
   
   shutdown(sock, 2);
   close(sock);
   
   return 0;
}

/* Wrapper for HTTP CDDB read using a proxy */
int
#if _STDC__
cddb_http_proxy_read(int cd_desc, struct cddb_host host, struct cddb_server proxy, struct cddb_hello hello, struct cddb_entry entry, struct disc_data *data)
#else
cddb_http_proxy_read(cd_desc, host, proxy, hello, entry, data)
   int cd_desc;
   struct cddb_host host;
   struct cddb_server proxy;
   struct cddb_hello hello;
   struct cddb_entry entry;
   struct disc_data *data;
#endif
{
   int sock;
   char http_string[512];

   if((sock = cddb_connect_server(host, &proxy, hello, http_string, 512)) < 0)
     return -1;

   if(cddb_read(cd_desc, sock, CDDB_MODE_HTTP, entry, data, http_string) < 0)
     return -1;

   shutdown(sock, 2);
   close(sock);

   return 0;
}

int
#if __STDC__
cddb_http_sites(int cd_desc, struct cddb_host host, struct cddb_hello hello, struct cddb_serverlist *list)
#else
cddb_http_sites(cd_desc, host, hello, list)
   int cd_desc;
   struct cddb_host host;
   struct cddb_hello hello;
   struct cddb_serverlist *list;
#endif
{
   int sock;
   char http_string[512];

   if((sock = cddb_connect_server(host, NULL, hello, http_string, 512)) < 0)
     return -1;

   if(cddb_sites(cd_desc, CDDB_MODE_HTTP, list, http_string) < 0)
     return -1;

   shutdown(sock, 2);
   close(sock);

   return 0;
}

int
#if __STDC__
cddb_http_proxy_sites(int cd_desc, struct cddb_host host, struct cddb_server proxy, struct cddb_hello hello, struct cddb_serverlist *list)
#else
cddb_http_proxy_sites(cd_desc, host, proxy, hello, list)
   int cd_desc;
   struct cddb_host host;
   struct cddb_server proxy;
   struct cddb_hello hello;
   struct cddb_serverlist *list;
#endif
{
   int sock;
   char http_string[512];
   
   if((sock = cddb_connect_server(host, &proxy, hello, http_string, 512)) < 0)
     return -1;
    
   if(cddb_sites(cd_desc, CDDB_MODE_HTTP, list, http_string) < 0)
     return -1;

   shutdown(sock, 2);
   close(sock);
	
   return 0;
}

int
#if __STDC__
cddb_http_submit(int cd_desc, struct cddb_host host, struct cddb_server *proxy, char *email_address)
#else
cddb_http_submit(cd_desc, host, proxy, email_address)
   int cd_desc;
   struct cddb_host host;
   struct cddb_server proxy;
   char *email_address;
#endif
{
   FILE *cddb_entry;
   int sock, index, changed_artist = 0, changed_track[MAX_TRACKS], token[3], error = 0;
   char inbuffer[512], outbuffer[512], cddb_file[512], *home;
   struct stat st;
   struct cddb_entry entry;
   struct disc_info disc;
   struct disc_data data;
   
   if((home = getenv("HOME")) == NULL) {
      strncpy(cddb_message, "$HOME is not set!", 256);
      return -1;
   }
   
   if(cd_stat(cd_desc, &disc) < 0)
     return -1;
   
   if(!disc.disc_present)
     return -1;
   
   if(cddb_stat_disc_data(cd_desc, &entry) < 0)
     return -1;
    
   if(entry.entry_present) {
     if(cddb_read_disc_data(cd_desc, &data) < 0)
       return -1;
   } else {
      strncpy(cddb_message, "No CDDB entry present in cache", 256);
      return -1;
   }
   
   if(proxy != NULL) {
      if((sock = cddb_connect(proxy)) < 0) {
	 strncpy(cddb_message, strerror(errno), 256);
	 return -1;
      }
   } else {	  
      if((sock = cddb_connect(&host.host_server)) < 0) {
	 strncpy(cddb_message, strerror(errno), 256);
         return -1;
      }
   }
	
   if(strlen(data.data_title) < 1 || strcmp(data.data_title, "Unknown") == 0) {
      strncpy(cddb_message, "Edit the disc title before submission.", 256);
      return -1;
   }
      	
   if(strcmp(data.data_artist, "Unknown") == 0) {
      strncpy(data.data_artist, "", 256);
      changed_artist = 1;
   }
	
   for(index = 0; index < disc.disc_total_tracks; index++) {
      changed_track[index] = 0;
      if(strcmp(data.data_track[index].track_name, "Unknown") == 0) { 
	 snprintf(data.data_track[index].track_name, 256, "Track %d", index);
	 changed_track[index] = 1;
      }
   }
   
   cddb_write_disc_data(cd_desc, data);
   
   if(cddb_submit_method == CDDB_SUBMIT_EMAIL) {
      snprintf(outbuffer, 512, "cat %s/.cddb/%s/%08lx | mail -s \"cddb %s %08lx\" %s", home, cddb_genre(data.data_genre), data.data_id, cddb_genre(data.data_genre), data.data_id, cddb_submit_email_address);
      if(system(outbuffer) != 0)
	return -1;
      return 0;
   }
   
   if(proxy != NULL)
     snprintf(outbuffer, 512, "POST http://%s:%d%s HTTP/1.0\n", host.host_server.server_name, host.host_server.server_port, HTTP_SUBMIT_CGI);
   else
     snprintf(outbuffer, 512, "POST %s HTTP/1.0\n", HTTP_SUBMIT_CGI);
   write(sock, outbuffer, strlen(outbuffer));
   
   snprintf(outbuffer, 512, "Category: %s\n", cddb_genre(data.data_genre));
   write(sock, outbuffer, strlen(outbuffer));
	
   snprintf(outbuffer, 512, "Discid: %08lx\n", data.data_id);
   write(sock, outbuffer, strlen(outbuffer));
	
   snprintf(outbuffer, 512, "User-Email: %s\n", email_address);
   write(sock, outbuffer, strlen(outbuffer));
	
   snprintf(outbuffer, 512, "Submit-Mode: %s\n", CDDB_SUBMIT_MODE ? "submit" : "test");
   write(sock, outbuffer, strlen(outbuffer));
	
   strncpy(outbuffer, "X-Cddbd-Note: Submission problems?  E-mail libcdaudio@gjhsnews.mesa.k12.co.us\n", 512);
   write(sock, outbuffer, strlen(outbuffer));
	
   snprintf(cddb_file, 512, "%s/.cddb/%s/%08lx", getenv("HOME"), cddb_genre(data.data_genre), data.data_id);
   stat(cddb_file, &st);
	
   snprintf(outbuffer, 512, "Content-Length: %d\n\n", (int) st.st_size);
   write(sock, outbuffer, strlen(outbuffer));
	
   cddb_entry = fopen(cddb_file, "r");
   while(!feof(cddb_entry)) {
       fgets(outbuffer, 512, cddb_entry);
       write(sock, outbuffer, strlen(outbuffer));
   }
   
   cddb_read_line(sock, inbuffer, 512);
   if(strncmp(inbuffer + 9, "200", 3) != 0) {
      strncpy(cddb_message, inbuffer, 256);
      return -1;
   }
	
   cddb_skip_http_header(sock);
   
   if(cddb_read_token(sock, token) < 0)
     error = 1;
   
   if(token[0] != 2)
     error = 1;
   
   shutdown(sock, 2);
   close(sock);
   
   if(changed_artist)
      strncpy(data.data_artist, "Unknown", 256);
	
   for(index = 0; index < disc.disc_total_tracks; index++)
      if(changed_track[index])
	strncpy(data.data_track[index].track_name, "Unknown", 256);
   
   data.data_revision++;
   cddb_write_disc_data(cd_desc, data);
	
   if(error)
     return -1;
	
   return 0;
}
