// ---------------------------------------------------------------------------
// - Object.hpp                                                              -
// - standard object library - base object declaration                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef ALEPH_OBJECT_HPP
#define ALEPH_OBJECT_HPP

namespace aleph {
  // forward class
  class Cons;
  class Vector;
  class String;
  class Interp;
  class Nameset;

  // declare the fundamental types of the aleph system
  typedef unsigned char      t_byte;
  typedef unsigned short     t_word;
  typedef unsigned int       t_quad;
  typedef unsigned long long t_octa;
  typedef long long          t_long;
  typedef double             t_real;

  // declare the fundamental constants
  const long nilp = 0;
  const char nilc = '\0';
  const char eofc = '\004';
  const char eolc = '\n';
  const char tabc = '\t';

  /// The Object class is the foundation of the standard object library .
  /// The object class defines only a reference count field which is used
  /// to control the life of a particular object. When an object is created,
  /// the reference count is set to 0. Such object is said to be transient.
  /// The "iref" static method increment the reference count. The "dref"
  /// method decrement and eventually destroy the object. The "cref" method
  /// eventually destroy an object if its reference count is nulll. The object
  /// class is an abstract class. For each derived object, the repr method
  /// is defined to return the class name. Additionally, the object class
  /// defines a set of methods which are used by the interpreter to virtually
  /// modify or evaluate an object. There are two sets of methods. The first
  /// set operates directly on the object. The second set operates by name
  /// on the object. Working by name is equivalent to access a member of a
  /// a particular object. The "cdef" method create or set a constant object
  /// to the calling object. The "vdef" method create or set an object to the
  /// calling object. The "eval" method evaluates an object in the current
  /// interpreter nameset. The "apply" method evaluates a set of arguments
  /// and apply them to the calling object. It is somehow equivalent to a 
  /// function call. When called by name, it is equivalent to a method call.
  /// @author amaury darsch

  class Object {
  public:
    enum t_oper {ADD, SUB, MUL, DIV, MINUS, EQL, NEQ, GEQ, LEQ, GTH, LTH};

  private:
    /// the pointer reference count
    long d_rcount;

  public:
    /// Create a new default object with zero reference count
    Object (void);

    /// Destroy unconditionally this object.
    virtual ~Object (void) {};

    /// @return the class name
    virtual String repr (void) const =0;

    /// @return an object class name or nil
    static const String repr (Object* object);
      
    /// increment the object reference count
    /// @param object the object to process
    static inline Object* iref (Object* object) {
      if (object == nilp) return nilp;
      object->d_rcount++;
      return object;
    }
    
    /// Decrement the reference count and destroy the object if null
    /// @param object the object to process
    static inline void dref (Object* object) {
      if (object == nilp) return;
      if (--object->d_rcount <= 0) delete object;
    }

    /// clean this object if the reference count is null
    /// @param object the object to process
    static inline void cref (Object* object) {
      if (object == nilp) return;
      if (object->d_rcount <= 0) delete object;
    }

    /// decrement the object reference count but do not detroy if null
    /// @param object the object to process
    static inline void tref (Object* object) {
      if (object == nilp) return;
      if (object->d_rcount > 0) object->d_rcount--;
    }

    /// operate this object with another one
    /// @param type the operator type
    /// @param object the operand object
    virtual Object* oper (t_oper type, Object* object);

    /// set an object as a const object
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    virtual Object* cdef (Interp* interp, Nameset* nset, Object* object);

    /// set an object as a const object by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the name to define as const
    /// @param object the object to set
    virtual Object* cdef (Interp* interp, Nameset* nset, const String& name, 
			  Object* object);

    /// set an object to this object
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    virtual Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// set an object to this object by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the name to set this object
    /// @param object the object to set
    virtual Object* vdef (Interp* interp, Nameset* nset, const String& name, 
			  Object* object);

    /// evaluate an object in the current nameset
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    virtual Object* eval (Interp* interp, Nameset* nset);

    /// evaluate an object in the current nameset by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the member name to evaluate in this object
    virtual Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply an object with a set of arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset 
    /// @param args   the arguments to apply
    virtual Object* apply (Interp* interp, Nameset* nset, Cons* args);

    /// apply an object by name with a set of arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    virtual Object* apply (Interp* interp, Nameset* nset, const String& name,
			   Cons* args);

    /// apply an object by quark with a set of arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply this arguments
    /// @param args   the arguments to apply
    virtual Object* apply (Interp* interp, Nameset* nset, const long quark,
			   Cons* args);

    /// apply an object by object with a set of arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset    
    /// @param object the object to apply this arguments
    /// @param args   the arguments to apply
    virtual Object* apply (Interp* interp, Nameset* nset, Object* object,
			   Cons* args);

    /// apply an object by name with a vector of arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply these arguments
    /// @param argv   the vector arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name, 
		   Vector* argv);
  };
}

#endif
