// ---------------------------------------------------------------------------
// - Throw.cpp                                                               -
// - aleph engine - builtin exception functions implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Vector.hpp"
#include "Builtin.hpp"
#include "Globalset.hpp"
#include "Exception.hpp"

namespace aleph {

  // throw builtin function implementation
  Object* builtin_throw (Interp* interp, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : (argv->length ());

    // check for 0 argument
    if (argc == 0) {
      delete argv;
      throw Exception ("user-exception");
    }

    // check for one argument
    if (argc == 1) {
      String eid = argv->getstring (0);
      delete argv;
      throw Exception (eid);
    }

    // check for two arguments
    if (argc == 2) {
      String eid = argv->getstring (0);
      String rsn = argv->getstring (1);
      delete argv;
      throw Exception (eid, rsn);
    }

    // check for 3 arguments
    if (argc == 3) {
      String eid   = argv->getstring (0);
      String rsn   = argv->getstring (1);
      Object*  obj = Object::iref (argv->get (2));
      delete argv;
      Object::tref (obj);
      throw Exception (eid, rsn, obj);
    }
    // invalid call
    throw Exception ("argument-error", "invalid argument with throw");
  }

  // try builtin function implementation

  Object* builtin_try (Interp* interp, Nameset* nset, Cons* args) {
    // check for arguments
    long argc = (args == nilp) ? 0 : args->length ();
    if ((argc == 0) || (argc > 2))
      throw Exception ("argument-error", "invalid arguments with try");
    Object* form = args->getcar ();

    // with 1 arguments we just run the form and catch any exception
    // the result is either the form result, the return/exception/any
    Object* result = nilp;
    if (argc == 1) {
      try {
	result = (form == nilp) ? nilp : form->eval (interp, nset);
	Object::iref (result);
      } catch (const Exception& e) {
	result = Object::iref (e.getobj ());
      } catch (...) {
	result = nilp;
      }
    }

    // with two arguments, we have a form to execute when an exception is
    // caught
    if (argc == 2) {
      Object* eform = args->getcadr ();
      try {
	result = (form == nilp) ? nilp : form->eval (interp, nset);
	Object::iref (result);
      } catch (const Exception& e) {
	Globalset* gset = new Globalset (nset);
	Object::iref (gset);
	gset->symcst ("what", new Exception (e));
	try {
	  result = (eform == nilp) ? nilp : eform->eval (interp, gset);
	  Object::iref (result);
	} catch (...) {
	  result = nilp;
	  Object::dref (gset);
	  throw;
	}
	Object::dref (gset);
      } catch (...) {
	result = (eform == nilp) ? nilp : eform->eval (interp, nset);
	Object::iref (result);
      }
    }
    Object::tref (result);
    return result;
  }
}
