// ---------------------------------------------------------------------------
// - Qualified.cpp                                                           -
// - aleph engine - qualified name class implementation                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Lexical.hpp"
#include "Nameset.hpp"
#include "Qualified.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a default qualified

  Qualified::Qualified (void) {
    d_lnum = 0;
  }

  // create a qualified with a name and a line number

  Qualified::Qualified (const String& name, const long lnum) {
    d_name = name;
    d_lnum = lnum;
    d_path = Strvec::split (d_name, ":");
    // check for good name
    long size = d_path.length ();
    if (size < 2)
      throw Exception ("syntax-error", "invalid qualified name", d_name);
    for (long i = 0; i < size; i++) {
      if (Lexical::valid (d_path.get (i)) == false)
	throw Exception ("syntax-error", "invalid qualified name", d_name);
    }
  }

  // copy constructor for this qualified

  Qualified::Qualified (const Qualified& that) {
    d_name = that.d_name;
    d_lnum = that.d_lnum;
    d_path = that.d_path;
  }

  // return the class name

  String Qualified::repr (void) const {
    return "Qualified";
  }

  // this procedure performs a partial evaluation and return the object
  static inline Object* eval_partial (Interp* interp, Nameset* nset, 
				      const Strvec& path) {
    Object* obj = nilp;
    long    len = 0;
    obj = nset->eval (interp,nset,path.first ());
    len = path.length () - 1;
    for (long i = 1; i < len; i++) {
      if (obj == nilp) return nilp;
      obj = obj->eval (interp, nset, path.get (i)); 
    }
    return obj;
  }

  // set this qualified name as a const symbol

  Object* Qualified::cdef (Interp* interp, Nameset* nset, Object* object) {
    Object* obj = eval_partial (interp, nset, d_path);
    if (obj == nilp) 
      throw Exception ("eval-error", "nil object with qualified name", d_name);
    return obj->cdef (interp, nset, d_path.last (), object);
  }

  // set this qualified name with an object

  Object* Qualified::vdef (Interp* interp, Nameset* nset, Object* object) {
    Object* obj = eval_partial (interp, nset, d_path);
    if (obj == nilp) 
      throw Exception ("eval-error", "nil object with qualified name", d_name);
    return obj->vdef (interp, nset, d_path.last (), object);
  }

  // evaluate this qualified name

  Object* Qualified::eval (Interp* interp, Nameset* nset) {
    Object* obj = eval_partial (interp, nset, d_path);
    if (obj == nilp) 
      throw Exception ("eval-error", "nil object with qualified name", d_name);
    return obj->eval (interp, nset, d_path.last ());
  }
}
