// ---------------------------------------------------------------------------
// - Localset.cpp                                                            -
// - aleph engine - local set class implementation                           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Symbol.hpp"
#include "Localset.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a new local set

  Localset::Localset (void) {
    p_ptbl = new NameTable;
    p_stbl = nilp;
    Object::iref (p_ptbl);
    symcst (".", this);
  }

  // create a new local set but use only the primary table

  Localset::Localset (Localset* lset) {
    if (lset == nilp) {
      p_ptbl = new NameTable;
      p_stbl = nilp;
    } else {
      p_ptbl = lset->p_ptbl;
      p_stbl = new NameTable;
    }
    Object::iref (p_ptbl);
    Object::iref (p_stbl);
    symcst (".", this); 
  }

  // destroy this local set

  Localset::~Localset (void) {
    Object::dref (p_ptbl);
    Object::dref (p_stbl);
  }

  // return the class name

  String Localset::repr (void) const {
    return "Localset";
  }

  // bind a new object by name

  void Localset::bind (const String& name, Object* object) {
    if (p_stbl != nilp)
      p_stbl->add (name, object);
    else
      p_ptbl->add (name, object);
  }

  // return true if the name exists in the local set

  bool Localset::exists (const String& name) const {
    if (p_stbl != nilp) return p_stbl->exists (name);
    return p_ptbl->exists (name);
  }

  // find an object in this local set - return nil if not found

  Object* Localset::find (const String& name) const {
    Object* obj = nilp;
    if (p_stbl != nilp) {
      obj = p_stbl->get (name);
      if (obj == nilp) obj = p_ptbl->get (name);
    } else {
      obj = p_ptbl->get (name);
    }
    if (obj != nilp) return obj;
    if (p_parent != nilp) return p_parent->find (name);
    return obj;
  }

  // remove an object by name in this localset

  void Localset::remove (const String& name) {
    if ((p_stbl != nilp) && (p_stbl->exists (name) == true)) {
      p_stbl->remove (name);
      return;
    }
    p_ptbl->remove (name);
  }

  // set an object as a const object by name

  Object* Localset::cdef (Interp* interp, Nameset* nset, const String& name, 
			   Object* object) {
    // try first to find the object
    Object* obj = nilp;
    if (p_stbl != nilp) {
      obj = p_stbl->get (name);
      if (obj == nilp) obj = p_ptbl->get (name);
    } else {
      obj = p_ptbl->get (name);
    }
    if (obj != nilp) {
      obj->cdef (interp, nset, object);
      return object;
    }

    // the object is not found - create a symbol and bind it
    Symbol* sym = new Symbol (name, object);
    sym->setconst (true);
    if (p_stbl != nilp)
      p_stbl->add (name, sym);
    else
      p_ptbl->add (name, sym);
    return object;
  }

  // set an object to this object by name

  Object* Localset::vdef (Interp* interp, Nameset* nset, const String& name,
			   Object* object) {
    // try first to find the object
    Object* obj = nilp;
    if (p_stbl != nilp) {
      obj = p_stbl->get (name);
      if (obj == nilp) obj = p_ptbl->get (name);
    } else {
      obj = p_ptbl->get (name);
    }
    if (obj != nilp) {
      obj->vdef (interp, nset, object);
      return object;
    }

    // the object is not found - create a symbol and bind it
    Symbol* sym = new Symbol (name, object);
    if (p_stbl != nilp)
      p_stbl->add (name, sym);
    else
      p_ptbl->add (name, sym);
    return object;
  }

  // evaluate an object in the current nameset by name

  Object* Localset::eval (Interp* interp, Nameset* nset, const String& name) {
    // try first to find the object
    Object* obj = nilp;
    if (p_stbl != nilp) {
      obj = p_stbl->get (name);
      if (obj == nilp) obj = p_ptbl->get (name);
    } else {
      obj = p_ptbl->get (name);
    }
    if (obj != nilp) return obj->eval (interp, nset);
    // try in the parent
    if (p_parent != nilp) return p_parent->eval (interp, nset, name);
    // not found
    throw Exception ("eval-error", "unbound symbol", name);
  }
}
