// ---------------------------------------------------------------------------
// - Lexical.hpp                                                             -
// - aleph engine - lexical name class definition                            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_LEXICAL_HPP
#define  ALEPH_LEXICAL_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The Lexical class is a simple class which holds a lexical name. Such
  /// object is generated by the aleph lexical analyzer. The object is
  /// constructed by name. Static methods are available to check for the
  /// name character syntax.
  /// @author amaury darsch

  class Lexical : public Literal {
  private:
    /// the lexical name
    String d_name;
    /// the line number
    long d_lnum;

  public:
    /// create an empty lexical
    Lexical (void);

    /// create a new lexical with a name
    /// @param name the lexical name
    Lexical (const String& name);

    /// create a new lexical with a name and a line number
    /// @param name the lexical name
    /// @param line the line number
    Lexical (const String& name, const long lnum);

    /// copy constructor for this lexical
    /// @param that the lexical to copy
    Lexical (const Lexical& that);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this lexical
    inline String toLiteral (void) const {
      return d_name;
    }

    /// @return a string representation of this lexical
    inline String toString (void) const {
      return d_name;
    }

    /// @return the lexical line number
    inline long getlnum (void) const {
      return d_lnum;
    }

    /// @return true if the character is a valid lexical one
    static bool valid (const char c);

    /// @return true if the lexical name is a valid one
    static bool valid (const String& name);

    /// set this lexical as a const symbol
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* cdef (Interp* interp, Nameset* nset, Object* object);

    /// set this lexical with an object
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// evaluate this lexical name in the current nameset
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    Object* eval (Interp* interp, Nameset* nset);

  private:
    // make the assignment operator private
    Lexical& operator = (const Lexical&);
  };
}

#endif
