// ---------------------------------------------------------------------------
// - Closure.hpp                                                             -
// - aleph engine - closure class definition                                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_CLOSURE_HPP
#define  ALEPH_CLOSURE_HPP

#ifndef  ALEPH_LOCALSET_HPP
#include "Localset.hpp"
#endif

namespace aleph {

  /// The Closure class is the class used to model lambda and gamma 
  /// expressions. When the closure is created, a local set of arguments and
  /// closed variables is generated. Note that such local set is dynamic.
  /// There is no restriction to reconfigure a particular lambda at run-time.
  /// The difference between a lambda and a gamma expression resides in the
  /// nameset binding when the closure is called. With a lambda, the closure
  /// nameset parent is the calling nameset. With a gamma expression, the
  /// parent nameset is always the top-level interpreter nameset. Note also,
  /// that the symbol "self" is automatically binded for this closure.
  /// @author amaury darsch

  class Closure : public Object {
  public:
    /// the type of closure
    enum t_ctype {LAMBDA, GAMMA};

  private:
    /// the closure type
    t_ctype d_type;
    /// the closure nameset
    Localset* p_lset;
    /// the number of arguments
    long d_argc;
    /// the runtime form
    Object* p_form;

  public:
    /// Create a new closure 
    /// @param type the closure type
    /// @param argl the argument list
    /// @param form the execution object 
    Closure (t_ctype type, Cons* argl, Object* form);

    /// Destroy this closure
    ~Closure (void);

    /// @return the class name
    String repr (void) const;

    /// add a new argument to this closure
    /// @param name the argument name
    /// @param flag the const flag for this argument
    void addarg (const String& name, const bool flag);

    /// add a new argument to this closure
    /// @param object the argument object
    void addarg (Object* object);

    /// add a closed variable to this closure
    /// @param name the closed variable name
    /// @param object the object to bind
    void addclv (const String& name, Object* object);

    /// apply this closure with a set of arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param args the arguments to apply with
    Object* apply (Interp* interp, Nameset* nset, Cons* args);

  private:
    // make the copy constructor private
    Closure (const Closure&);
    // make the assignment operator private
    Closure& operator = (const Closure&);
  };
}

#endif
