#   include	"config.h"

#   include	"bmintern.h"
#   include	<string.h>
#   include	<debugon.h>

/************************************************************************/
/*									*/
/*  Color reduction.							*/
/*									*/
/************************************************************************/

typedef struct HistogramEntry
    {
    int			heCount;
    unsigned char	heRed;
    unsigned char	heGreen;
    unsigned char	heBlue;
    } HistogramEntry;

typedef struct	HashBucket
    {
    HistogramEntry	hbHistogramEntry;
    struct HashBucket *	hbNext;
    } HashBucket;

#   define	hbCount	hbHistogramEntry.heCount
#   define	hbRed	hbHistogramEntry.heRed
#   define	hbGreen	hbHistogramEntry.heGreen
#   define	hbBlue	hbHistogramEntry.heBlue

typedef struct ColorBox
    {
    int		cbFirst;
    int		cbColorCount;
    int		cbPixelCount;
    int		cbCutNumber;
    } ColorBox;

typedef struct CutNode
    {
    unsigned int	cnLeft;
    unsigned int	cnRight;
    unsigned char	cnValue;
    unsigned char	cnComponent;
    } CutNode;

#   define	CN_LEAF		0
#   define	CN_RED		1
#   define	CN_GREEN	2
#   define	CN_BLUE		3

/************************************************************************/
/*  Color hash varia.							*/
/************************************************************************/

#define HASH_SIZE 6553
#define ppm_hash(r,g,b)	((( (int) (r) * 33023 +    \
			    (int) (g) * 30013 +    \
			    (int) (b) * 27011 ) & 0x7fffffff) % HASH_SIZE )

static void bmFreeColorHash(	HashBucket **	hashTable	)
    {
    int			i;

    for ( i= 0; i < HASH_SIZE; i++ )
	{
	HashBucket *	hashBucket= hashTable[i];

	while( hashBucket )
	    {
	    HashBucket *	nx= hashBucket->hbNext;
	    free( (char *)hashBucket );
	    hashBucket= nx;
	    }
	}

    return;
    }

/************************************************************************/
/*  Insert a new hash item: Convenience routine.			*/
/*  NOTE that the cleanup on failure is done here, not in the caller.	*/
/************************************************************************/
static int bmInsertHash(	HashBucket **	hashTable,
				int		hash,
				int		r,
				int		g,
				int		b		)
    {
    HashBucket *	hashBucket= (HashBucket *)malloc( sizeof(HashBucket) );

    if  ( ! hashBucket )
	{ XDEB(hashBucket); bmFreeColorHash( (void *)hashTable ); return -1; }

    hashBucket->hbNext= hashTable[hash];
    hashTable[hash]= hashBucket;

    hashBucket->hbCount= 1;

    hashBucket->hbRed= r;
    hashBucket->hbGreen= g;
    hashBucket->hbBlue= b;

    return 0;
    }

/************************************************************************/
/*  Compare colors, used with qsort()					*/
/************************************************************************/
static int bmHistRedCompare( const void * v1, const void * v2 )
    { return ((HistogramEntry *)v1)->heRed- ((HistogramEntry *)v2)->heRed; }
static int bmHistGreenCompare( const void * v1, const void * v2 )
    { return ((HistogramEntry *)v1)->heGreen- ((HistogramEntry *)v2)->heGreen; }
static int bmHistBlueCompare( const void * v1, const void * v2 )
    { return ((HistogramEntry *)v1)->heBlue- ((HistogramEntry *)v2)->heBlue; }

/************************************************************************/
/*  Find the median for splitting a box.				*/
/************************************************************************/
static void bmSplitBox(	HistogramEntry *	histogram,
			int			colorCount,
			int			pixelCount,
			int *			pColorsLeft,
			int *			pPixelsLeft,
			int *			pComponent,
			int *			pValue		)
    {
    unsigned char	rMin, rMax;
    unsigned char	gMin, gMax;
    unsigned char	bMin, bMax;

    int			i;
    int			count;

    rMin= rMax= histogram[0].heRed;
    gMin= gMax= histogram[0].heGreen;
    bMin= bMax= histogram[0].heBlue;

    for ( i= 1; i < colorCount; i++ )
	{
	if  ( rMin > histogram[i].heRed )
	    { rMin = histogram[i].heRed;	}
	if  ( rMax < histogram[i].heRed )
	    { rMax = histogram[i].heRed;	}

	if  ( gMin > histogram[i].heGreen )
	    { gMin = histogram[i].heGreen;	}
	if  ( gMax < histogram[i].heGreen )
	    { gMax = histogram[i].heGreen;	}

	if  ( bMin > histogram[i].heBlue )
	    { bMin = histogram[i].heBlue;	}
	if  ( bMax < histogram[i].heBlue )
	    { bMax = histogram[i].heBlue;	}
	}

    if  ( 77* ( rMax- rMin ) > 150* ( gMax- gMin ) &&
	  77* ( rMax- rMin ) >  29* ( bMax- bMin ) )
	{
	qsort( histogram, colorCount, sizeof(HistogramEntry),
							bmHistRedCompare );
	*pComponent= CN_RED;
	}
    else{
	if  ( 150* ( gMax- gMin ) > 29* ( bMax- bMin ) )
	    {
	    qsort( histogram, colorCount, sizeof(HistogramEntry),
							bmHistGreenCompare );
	    *pComponent= CN_GREEN;
	    }
	else{
	    qsort( histogram, colorCount, sizeof(HistogramEntry),
							bmHistBlueCompare );
	    *pComponent= CN_BLUE;
	    }
	}

    count= histogram[0].heCount;
    for ( i= 1; i < colorCount- 1; i++ )
	{
	if  ( count >= pixelCount/ 2 )
	    { break;	}
	count += histogram[i].heCount;
	}

    switch( *pComponent )
	{
	case CN_RED:	*pValue= histogram[i].heRed;	break;
	case CN_GREEN:	*pValue= histogram[i].heGreen;	break;
	case CN_BLUE:	*pValue= histogram[i].heBlue;	break;
	default:	LDEB(*pComponent);		break; /* shame */
	}

    *pColorsLeft= i;
    *pPixelsLeft= count;

    return;
    }

/************************************************************************/
/*  Find a collection of colors using 'median cut'.			*/
/************************************************************************/
static int bmFindColors(	HistogramEntry *	histogram,
				RGB8Color *		colors,
				CutNode *		cutNodes,
				int			colorCount,
				int			pixelCount,
				int			maxcolors 	)
    {
    ColorBox *			boxes;
    int				i;
    int				j;
    int				boxCount;
    int				nodeCount= 0;

    boxes= (ColorBox *)malloc( maxcolors* sizeof(ColorBox) );
    if  ( ! boxes )
	{ LLDEB(maxcolors,boxes); return -1; }

    boxes[0].cbFirst= 0;
    boxes[0].cbColorCount= colorCount;
    boxes[0].cbPixelCount= pixelCount;
    boxes[0].cbCutNumber= 0;
    boxCount= 1;

    cutNodes[0].cnComponent= CN_LEAF;
    cutNodes[0].cnValue= 0;		/*  Irrelevant !	*/
    cutNodes[0].cnLeft= 0;		/*  Both to this color.	*/
    cutNodes[0].cnRight= 0;		/*  Both to this color.	*/
    nodeCount= 1;

    while( boxCount < maxcolors )
	{
	int	biggestBox;
	int	colorsLeft;
	int	pixelsLeft;

	int	cutComponent;
	int	cutValue;

	for ( i= 0; i < boxCount; i++ )
	    {
	    if  ( boxes[i].cbColorCount > 1 )
		{ break;	}
	    }
	if  ( i >= boxCount )
	    { LLDEB(i,boxCount); break;	}
	biggestBox= i;
	for ( i= biggestBox+ 1; i < boxCount; i++ )
	    {
	    if  ( boxes[i].cbColorCount > 1				&&
		  boxes[i].cbPixelCount > boxes[biggestBox].cbPixelCount )
		{ biggestBox= i;	}
	    }

	bmSplitBox( histogram+ boxes[biggestBox].cbFirst,
				    boxes[biggestBox].cbColorCount,
				    boxes[biggestBox].cbPixelCount,
				    &colorsLeft, &pixelsLeft,
				    &cutComponent, &cutValue );

	/****************/
	/*  Cut Nodes	*/
	/****************/
	i= boxes[biggestBox].cbCutNumber;
	cutNodes[i].cnComponent= cutComponent;
	cutNodes[i].cnValue= cutValue;
	cutNodes[i].cnLeft= nodeCount;
	cutNodes[i].cnRight= nodeCount+ 1;

	cutNodes[nodeCount].cnComponent= CN_LEAF;
	cutNodes[nodeCount].cnValue= 0;		/*  Irrelevant !	*/
	cutNodes[nodeCount].cnLeft= biggestBox;	/*  Both to this color.	*/
	cutNodes[nodeCount].cnRight= biggestBox;/*  Both to this color.	*/

	cutNodes[nodeCount+1].cnComponent= CN_LEAF;
	cutNodes[nodeCount+1].cnValue= 0;	/*  Irrelevant !	*/
	cutNodes[nodeCount+1].cnLeft= boxCount;	/*  Both to this color.	*/
	cutNodes[nodeCount+1].cnRight= boxCount;/*  Both to this color.	*/

	/****************/
	/*  Boxes	*/
	/****************/
	boxes[boxCount].cbFirst= boxes[biggestBox].cbFirst+ colorsLeft;
	boxes[boxCount].cbColorCount=
				boxes[biggestBox].cbColorCount- colorsLeft;
	boxes[boxCount].cbPixelCount=
				boxes[biggestBox].cbPixelCount- pixelsLeft;
	boxes[boxCount].cbCutNumber= nodeCount+ 1;

	boxes[biggestBox].cbColorCount= colorsLeft;
	boxes[biggestBox].cbPixelCount= pixelsLeft;
	boxes[biggestBox].cbCutNumber= nodeCount;

	nodeCount += 2; boxCount++;
	}

    for ( i= 0; i < boxCount; i++ )
	{
	long			sR= 0;
	long			sG= 0;
	long			sB= 0;
	long			N= 0;
	HistogramEntry *	he= histogram+ boxes[i].cbFirst;

	for ( j= 0; j < boxes[i].cbColorCount; j++, he++ )
	    {
	    sR += he->heCount* he->heRed;
	    sG += he->heCount* he->heGreen;
	    sB += he->heCount* he->heBlue;
	    N  += he->heCount;
	    }
	colors[i].rgb8Red= sR/ N;
	colors[i].rgb8Green= sG/ N;
	colors[i].rgb8Blue= sB/ N;
	}

    free( (char *)boxes );

    return boxCount;
    }

/************************************************************************/
/*  Various translation routines.					*/
/************************************************************************/
static void bm24to8or4(	int			pixelsHigh,
			int			pixelsWide,
			int			inBytesPerRow,
			int			outBytesPerRow,
			const unsigned char *	bufIn,
			unsigned char *		bufOut,
			CutNode *		cutNodes,
			int			bitsPerPixel		)
    {
    int				row;
    int				col;
    int				i;

    unsigned char		r;
    unsigned char		g;
    unsigned char		b;

    const unsigned char *	from;
    unsigned char *		to;

    for ( row= 0; row < pixelsHigh; row++ )
	{
	from= bufIn+ row* inBytesPerRow;
	to= bufOut+ row* outBytesPerRow;

	for ( col= 0; col < pixelsWide; col++ )
	    {
	    r= *(from++); g= *(from++); b= *(from++);
	    i= 0;
	    for (;;)
		{
		switch( cutNodes[i].cnComponent )
		    {
		    case CN_RED:
			if  ( r < cutNodes[i].cnValue )
			    { i= cutNodes[i].cnLeft;	continue; }
			else{ i= cutNodes[i].cnRight;	continue; }
		    case CN_GREEN:
			if  ( g < cutNodes[i].cnValue )
			    { i= cutNodes[i].cnLeft;	continue; }
			else{ i= cutNodes[i].cnRight;	continue; }
		    case CN_BLUE:
			if  ( b < cutNodes[i].cnValue )
			    { i= cutNodes[i].cnLeft;	continue; }
			else{ i= cutNodes[i].cnRight;	continue; }
		    case CN_LEAF:
			break;
		    default:
			LDEB(cutNodes[i].cnComponent); return;
		    }
		break;
		}

	    switch( bitsPerPixel )
		{
		case 8:
		    *(to++)= cutNodes[i].cnLeft; break;
		case 4:
		    if  ( col % 2 )
			{ *(to++) |=  cutNodes[i].cnLeft;	 break; }
		    else{ *(to  )=  ( cutNodes[i].cnLeft << 4 ); break; }
		default:
		    LDEB(bitsPerPixel); return;
		}
	    }
	}
    }

/************************************************************************/
/*  Hash the colors in a 24 bit image. Do not return more than a	*/
/*  given number of colors.						*/
/*  1)  Make a hash table for the colors.				*/
/************************************************************************/
static int bmHashColors24(	HashBucket ***		pHashTable,
				int *			pColorCount,
				int			pixelsHigh,
				int			pixelsWide,
				int			bytesPerRow,
				const unsigned char *	bufIn,
				int			maxcolors,
				int			mask		)
    {
    unsigned int		r;
    unsigned int		g;
    unsigned int		b;

    int				row;
    int				col;

    HashBucket **		hashTable;
    HashBucket *		hashBucket;
    int				hash;

    int				colorCount= 0;

    const unsigned char *	from;

    /*  1  */
    hashTable= (HashBucket **)malloc( HASH_SIZE* sizeof(HashBucket *) );
    if  ( ! hashTable )
	{ XDEB(hashTable); return -1;	}
    for ( row= 0; row < HASH_SIZE; row++ )
	{ hashTable[row]= (HashBucket *)0;	}

    for ( row= 0; row < pixelsHigh; row++ )
	{
	from= bufIn+ row* bytesPerRow;

	for ( col= 0; col < pixelsWide; col++ )
	    {
	    r= *(from++) & mask; g= *(from++) & mask; b= *(from++) & mask;
	    hash= ppm_hash( r, g, b );
	    hashBucket= hashTable[hash];
	    while( hashBucket )
		{
		if  ( hashBucket->hbRed == r	&&
		      hashBucket->hbGreen == g	&&
		      hashBucket->hbBlue == b	)
		    { break;	}
		hashBucket= hashBucket->hbNext;
		}
	    if  ( hashBucket )
		{ hashBucket->hbCount++; continue;	}
	    if  ( bmInsertHash( hashTable, hash, r, g, b ) )
		{ LDEB(1); return -1;	}

	    colorCount++;

	    if  ( colorCount > maxcolors )
		{ bmFreeColorHash( (void *)hashTable ); return 1; }
	    }
	}

    *pColorCount= colorCount;
    *pHashTable= hashTable;

    return 0;
    }

/************************************************************************/
/*  Reduce the number of colors in an image to at most 'maxcolors'.	*/
/*  1)  Make a hash table for the colors.				*/
/*  2)  Store and count all the colors in the hash table.		*/
/*  3)  Translate the hash table to a histogram.			*/
/*  4)  Allocate memory for the palette and for the tree that will be	*/
/*	used to classify the colors.					*/
/*  5)  Classify colors: Yields a palette and a tree.			*/
/*  6)  Allocate and initialise output.					*/
/*  7)  Translate to representative of box.				*/
/************************************************************************/

int bmColorReduce(	BitmapDescription *		bdOut,
			const BitmapDescription *	bdIn,
			unsigned char **		pBufOut,
			const unsigned char *		bufIn,
			int				maxcolors )
    {
    int				row;
    int				i;

    HashBucket **		hashTable;
    HashBucket *		hashBucket;

    HistogramEntry *		histogram;
    RGB8Color *			colors;
    CutNode *			cutNodes;

    int				colorCount= 0;
    int				colorsFound;

    int				bitsPerPixel;

    if  ( maxcolors > 256 )
	{ LDEB(maxcolors); return -1;	}
    if  ( maxcolors <= 16 )
	{ bitsPerPixel= 4;	}
    else{ bitsPerPixel= 8;	}

    /*  1, 2  */
    switch( bdIn->bdBitsPerPixel )
	{
	case 24:
	    for ( i= 0; i < 8; i++ )
		{
		row= bmHashColors24( &hashTable, &colorCount,
				bdIn->bdPixelsHigh, bdIn->bdPixelsWide,
				bdIn->bdBytesPerRow, bufIn, 32768, 
				~( 0xff >> ( 8- i ) ) );
		if  ( row < 0 )
		    { LDEB(row); return -1;	}
		if  ( row == 0 )
		    { break;	}
		}
	    break;
	default:
	    LDEB(bdIn->bdBitsPerPixel); return -1;
	}

    /*  3  */
    histogram= (HistogramEntry *)malloc( colorCount* sizeof(HistogramEntry) );
    if  ( ! histogram )
	{
	LLDEB(colorCount,histogram); 
	bmFreeColorHash( (void *)hashTable ); return -1;
	}

    row= 0;
    for ( i= 0; i < HASH_SIZE; i++ )
	{
	hashBucket= hashTable[i];

	while( hashBucket )
	    {
	    histogram[row++]= hashBucket->hbHistogramEntry;
	    hashBucket= hashBucket->hbNext;
	    }
	}

    bmFreeColorHash( (void *)hashTable );

    /*  4  */
    colors= (RGB8Color *)malloc( maxcolors* sizeof(RGB8Color) );
    if  ( ! colors )
	{
	LLDEB(maxcolors,colors);
	free( (char *)histogram );
	return -1;
	}

    cutNodes= (CutNode *)malloc( 2* maxcolors* sizeof(CutNode) );
    if  ( ! cutNodes )
	{
	LLDEB(maxcolors,cutNodes);
	free( (char *)histogram ); free( (char *)colors );
	return -1;
	}

    memset( colors, 0, maxcolors* sizeof(RGB8Color) );
    memset( cutNodes, 0, 2* maxcolors* sizeof(CutNode) );

    /*  5  */
    colorsFound= bmFindColors( histogram, colors, cutNodes, colorCount,
			bdIn->bdPixelsHigh* bdIn->bdPixelsWide, maxcolors );
    if  ( colorsFound < 1 )
	{
	LLDEB(maxcolors,colorsFound);
	free( (char *)histogram ); free( (char *)colors ); free( cutNodes );
	return -1;
	}

    free( histogram );

    /*  6  */
    bdOut->bdPixelsWide= bdIn->bdPixelsWide;
    bdOut->bdPixelsHigh= bdIn->bdPixelsHigh;

    bdOut->bdBitsPerSample= bdIn->bdBitsPerSample;
    bdOut->bdSamplesPerPixel= 3;
    bdOut->bdBitsPerPixel= bitsPerPixel;
    bdOut->bdBytesPerRow= ( bdOut->bdBitsPerPixel* bdOut->bdPixelsWide+ 7 )/8;
    bdOut->bdBufferLength= bdOut->bdPixelsHigh* bdOut->bdBytesPerRow;

    bdOut->bdXResolution= bdIn->bdXResolution;
    bdOut->bdYResolution= bdIn->bdYResolution;
    bdOut->bdUnit= bdIn->bdUnit;

    bdOut->bdColorEncoding= BMcoRGB8PALETTE;
    bdOut->bdColorCount= colorsFound;

    bdOut->bdHasAlpha= 0;

    bdOut->bdRGB8Palette= colors;

    *pBufOut= malloc( bdOut->bdBufferLength );
    if  ( ! *pBufOut )
	{
	LLDEB( bdOut->bdBufferLength,*pBufOut);
	free( (char *)histogram ); free( (char *)colors ); free( cutNodes );
	return -1;
	}

    /*  7  */
    switch( bdIn->bdBitsPerPixel )
	{
	case 24:
	    switch( bitsPerPixel )
		{
		case 8:
		case 4:
		    bm24to8or4( bdIn->bdPixelsHigh, bdIn->bdPixelsWide,
			    bdIn->bdBytesPerRow, bdOut->bdBytesPerRow,
			    bufIn, *pBufOut, cutNodes, bitsPerPixel );
		    break;
		default:
		    LLDEB(bdIn->bdBitsPerPixel,bdOut->bdBitsPerPixel);
		    return -1;
		}
	    break;
	default:
	    LLDEB(bdIn->bdBitsPerPixel,bdOut->bdBitsPerPixel); return -1;
	}

    free( cutNodes );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Make a palette for a bitmap description.				*/
/*									*/
/************************************************************************/

int bmMakeGrayPalette(		const BitmapDescription *	bd,
				int *				pColorCount,
				RGB8Color *			palette )
    {
    int		i;
    int		colorCount;

    switch( bd->bdColorEncoding )
	{
	case BMcoRGB8PALETTE:
	    LDEB(bd->bdColorEncoding); return -1;
	    break;

	case BMcoBLACKWHITE:
	    switch( bd->bdBitsPerPixel )
		{
		case 1: case 2: case 4: case 8:
		    colorCount= 1 << bd->bdBitsPerPixel;

		    for ( i= 0; i < colorCount; i++ )
			{
			palette[colorCount- i- 1].rgb8Red=
			palette[colorCount- i- 1].rgb8Green=
			palette[colorCount- i- 1].rgb8Blue=
					    ( i* 255 )/ ( colorCount- 1 );
			}

		    *pColorCount= colorCount; return 0;

		default:
		    LLDEB(bd->bdColorEncoding,bd->bdBitsPerPixel);
		    return -1;
		}

	case BMcoWHITEBLACK:
	    switch( bd->bdBitsPerPixel )
		{
		case 1: case 2: case 4: case 8:
		    colorCount= 1 << bd->bdBitsPerPixel;

		    for ( i= 0; i < colorCount; i++ )
			{
			palette[i].rgb8Red=
			palette[i].rgb8Green=
			palette[i].rgb8Blue= ( i* 255 )/ ( colorCount- 1 );
			}

		    *pColorCount= colorCount; return 0;

		default:
		    LLDEB(bd->bdColorEncoding,bd->bdBitsPerPixel);
		    return -1;
		}
	    break;

	case BMcoRGB:
	    LDEB(bd->bdColorEncoding); return -1;
	    break;

	default:
	    LDEB(bd->bdColorEncoding); return -1;
	}

    return -1;
    }

/************************************************************************/
/*									*/
/*  Inflate a scan line to 8 bits per pixel.				*/
/*									*/
/************************************************************************/

int bmInflateTo8bit(		unsigned char *			to,
				const unsigned char *		from,
				const BitmapDescription *	bd )
    {
    int			col;
    unsigned char	b;

    switch( bd->bdBitsPerPixel )
	{
	case 1:
	    for ( col= 0; col < bd->bdPixelsWide; col += 8 )
		{
		b= *(from++);

		*(to++)= ( b & 0x80 ) >> 7;
		*(to++)= ( b & 0x40 ) >> 6;
		*(to++)= ( b & 0x20 ) >> 5;
		*(to++)= ( b & 0x10 ) >> 4;
		*(to++)= ( b & 0x08 ) >> 3;
		*(to++)= ( b & 0x04 ) >> 2;
		*(to++)= ( b & 0x02 ) >> 1;
		*(to++)= ( b & 0x01 )     ;
		}
	    return 0;

	case 2:
	    for ( col= 0; col < bd->bdPixelsWide; col += 4 )
		{
		b= *(from++);

		*(to++)= ( b & 0xc0 ) >> 6;
		*(to++)= ( b & 0x30 ) >> 4;
		*(to++)= ( b & 0x0c ) >> 2;
		*(to++)= ( b & 0x03 )     ;
		}
	    return 0;

	case 4:
	    for ( col= 0; col < bd->bdPixelsWide; col += 2 )
		{
		b= *(from++);

		*(to++)= ( b & 0xf0 ) >> 4;
		*(to++)= ( b & 0x0f )     ;
		}
	    return 0;

	default:
	    LDEB(bd->bdBitsPerPixel); return -1;
	}
    }

int bmInflateTo8bitGray(	unsigned char *			to,
				const unsigned char *		from,
				const BitmapDescription *	bd )
    {
    int			col;
    unsigned char	b;

    switch( bd->bdBitsPerPixel )
	{
	case 1:
	    for ( col= 0; col < bd->bdPixelsWide; col += 8 )
		{
		b= *(from++);

		*(to++)= ( 255* ( ( b & 0x80 ) >> 7 ) );
		*(to++)= ( 255* ( ( b & 0x40 ) >> 6 ) );
		*(to++)= ( 255* ( ( b & 0x20 ) >> 5 ) );
		*(to++)= ( 255* ( ( b & 0x10 ) >> 4 ) );
		*(to++)= ( 255* ( ( b & 0x08 ) >> 3 ) );
		*(to++)= ( 255* ( ( b & 0x04 ) >> 2 ) );
		*(to++)= ( 255* ( ( b & 0x02 ) >> 1 ) );
		*(to++)= ( 255* ( ( b & 0x01 )      ) );
		}
	    return 0;

	case 2:
	    for ( col= 0; col < bd->bdPixelsWide; col += 4 )
		{
		b= *(from++);

		*(to++)= ( 255* ( ( b & 0xc0 ) >> 6 ) )/ 3;
		*(to++)= ( 255* ( ( b & 0x30 ) >> 4 ) )/ 3;
		*(to++)= ( 255* ( ( b & 0x0c ) >> 2 ) )/ 3;
		*(to++)= ( 255* ( ( b & 0x03 )      ) )/ 3;
		}
	    return 0;

	case 4:
	    for ( col= 0; col < bd->bdPixelsWide; col += 2 )
		{
		b= *(from++);

		*(to++)= ( 255* ( ( b & 0xf0 ) >> 4 ) )/ 15;
		*(to++)= ( 255* ( ( b & 0x0f )      ) )/ 15;
		}
	    return 0;

	default:
	    LDEB(bd->bdBitsPerPixel); return -1;
	}
    }
