/************************************************************************/
/*									*/
/*  Run file choosers for edit applications.				*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<locale.h>

#   include	<Xm/Xm.h>
#   include	<Xm/FileSB.h>
#   include	<Xm/Form.h>
#   include	<Xm/Text.h>
#   include	<Xm/TextF.h>
#   include	<Xm/DialogS.h>

#   include	<Xm/MwmUtil.h>

#   include	<Xm/Protocols.h>

#   include	<appSystem.h>

#   include	<appFrame.h>
#   include	"appUtil.h"
#   include	<debugon.h>

/************************************************************************/
/*									*/
/*  For saving documents.						*/
/*  The architecture is dictated by Motif.				*/
/*									*/
/************************************************************************/

#   define	ACIrespNONE	0
#   define	ACIrespSAVE	1
#   define	ACIrespCANCEL	2
#   define	ACIrespOPEN	3
#   define	ACIrespFAILURE	4

typedef struct AppChooserInformation
    {
    int				aciResponse;
    char *			aciFilename;
    void *			aciThrough;
    APP_OPEN_DOCUMENT		aciOpenDocument;
    const AppFileExtension *	aciExtensions;
    int				aciExtensionCount;
    Widget			aciChooser;
    Widget			aciFilterPulldown;
    Widget			aciFilterMenu;
    EditApplication *		aciApplication;
    Widget			aciOption;
    int				aciFormat;
    char *			aciExtension;

    int				aciFilterMenuWidth;

    char *			aciNoFilenameMessage;
    char *			aciIsDirecoryMessage;
    char *			aciNotWritableMessage;
    char *			aciNotReadableMessage;
    char *			aciOverwriteMessage;
    char *			aciNoSuchDirMessage;
    } AppChooserInformation;

/************************************************************************/
/*  Error messages.							*/
/************************************************************************/
# define xx(x)	x,x

static XtResource APP_ChooserResourceTable[]=
    {
	{ xx("chooserNoFilename"), XtRString, sizeof(char *),
		    offsetof(AppChooserInformation,aciNoFilenameMessage),
		    XtRString, "Please Give a Name" },
	{ xx("chooserIsDirectory"), XtRString, sizeof(char *),
		    offsetof(AppChooserInformation,aciIsDirecoryMessage),
		    XtRString, "Is a directory" },
	{ xx("chooserNotWritable"), XtRString, sizeof(char *),
		    offsetof(AppChooserInformation,aciNotWritableMessage),
		    XtRString, "You have no permission to write this file." },
	{ xx("chooserNotReadable"), XtRString, sizeof(char *),
		    offsetof(AppChooserInformation,aciNotReadableMessage),
		    XtRString, "You have no permission to read this file." },
	{ xx("chooserOverwrite"), XtRString, sizeof(char *),
		    offsetof(AppChooserInformation,aciOverwriteMessage),
		    XtRString, "Do you want to overwrite this file?" },
	{ xx("chooserNoSuchDir"), XtRString, sizeof(char *),
		    offsetof(AppChooserInformation,aciNoSuchDirMessage),
		    XtRString, "This directory does not exist." },
    };

/************************************************************************/
/*  Cleanup when a chooser is destroyed.				*/
/************************************************************************/
static void appFreeAci(		Widget		fileChooser,
				XtPointer	voidaci,
				XtPointer	voidcbs	 )
    { free( voidaci );	}

static void appInitAci(		AppChooserInformation *	aci )
    {
    aci->aciResponse= ACIrespNONE;
    aci->aciFilename= (char *)0;
    aci->aciThrough= (void *)0;
    aci->aciOpenDocument= (APP_OPEN_DOCUMENT)0;
    aci->aciExtensions= (AppFileExtension *)0;
    aci->aciExtensionCount= 0;
    aci->aciChooser= (Widget)0;
    aci->aciFilterPulldown= (Widget)0;
    aci->aciFilterMenu= (Widget)0;
    aci->aciApplication= (EditApplication *)0;
    aci->aciOption= (Widget)0;
    aci->aciFormat= -1;
    aci->aciExtension= (char *)0;

    aci->aciIsDirecoryMessage= (char *)0;
    aci->aciNotWritableMessage= (char *)0;
    aci->aciNotReadableMessage= (char *)0;
    aci->aciOverwriteMessage= (char *)0;
    aci->aciNoSuchDirMessage= (char *)0;
    }

/************************************************************************/
/*  Callback for cancel buttons on choosers.				*/
/************************************************************************/
static void appChooserCancel(	Widget		fileChooser,
				XtPointer	voidaci,
				XtPointer	voidcbs	 )
    {
    AppChooserInformation *	aci= (AppChooserInformation *)voidaci;

    aci->aciResponse= ACIrespCANCEL;

    return;
    }

/************************************************************************/
/*									*/
/*  Retrieve the file name from a chooser.				*/
/*									*/
/************************************************************************/

static int appChooserGetFilename(	void *			voidcbs,
					Widget			fileChooser,
					AppChooserInformation *	aci )
    {
    EditApplication *			ea= aci->aciApplication;
    Widget				option= aci->aciOption;

    XmFileSelectionBoxCallbackStruct *	cbs;
    char *				extension= (char *)0;
    char *				slash;
    char *				relative;

    cbs= (XmFileSelectionBoxCallbackStruct *)voidcbs;

    if  ( ! XmStringGetLtoR( cbs->value,
			    XmFONTLIST_DEFAULT_TAG, &(aci->aciFilename) ) )
	{
	aci->aciResponse= ACIrespFAILURE;
	return -1;
	}

    if  ( aci->aciExtensionCount > 0 )
	{
	if  ( aci->aciFormat >= 0 && aci->aciFormat < aci->aciExtensionCount )
	    {
	    const AppFileExtension *	afe= aci->aciExtensions+ aci->aciFormat;

	    extension= afe->afeExtension;
	    }
	}
    else{ extension= aci->aciExtension;	}

    slash= strrchr( aci->aciFilename, '/' );
    if  ( ! slash )
	{ relative= aci->aciFilename;	}
    else{ relative= slash+ 1;		}

    if  ( ! relative[0] )
	{
	appQuestionRunErrorDialog( ea, XtParent( fileChooser ), option,
						    aci->aciNoFilenameMessage );
	XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;
	/* NO: just stay in the loop.
	aci->aciResponse= ACIrespFAILURE;
	*/
	return -1;
	}

    if  ( extension && extension[0] )
	{
	char *	dot;

	dot= strrchr( relative, '.' );

	if  ( ! dot				||
	      strcmp( dot+ 1, extension )	)
	    {
	    int		ln;
	    int		le;

	    ln= strlen( aci->aciFilename );
	    le= strlen( extension );

	    aci->aciFilename= XtRealloc( aci->aciFilename, ln+ le+ 2 );

	    if  ( dot && ! dot[1] )
		{
		strcpy( aci->aciFilename+ ln,    extension );
		}
	    else{
		strcpy( aci->aciFilename+ ln   , "." );
		strcpy( aci->aciFilename+ ln+ 1, extension );
		}
	    }
	}

    return 0;
    }

/************************************************************************/
/*  Set the directory for a chooser, based on the file name.		*/
/*  Set a file name derived from the previous name and an extension.	*/
/************************************************************************/
static void appChooserSetText(		Widget  text,
					char *	s )
    {
    if  ( XmIsText( text ) )
	{
	XmTextSetString( text, s );
	XmTextSetInsertionPosition( text, strlen( s ) );

	return;
	}

    if  ( XmIsTextField( text ) )
	{
	XmTextFieldSetString( text, s );
	XmTextFieldSetInsertionPosition( text, strlen( s ) );

	return;
	}

    SDEB( (char *)text );
    }

static void appChooserSetDirectory(	char *	filename,
					Widget	chooser )
    {
    char *	slash= strrchr( filename, '/' );

    if  ( slash )
	{
	XmString	directoryString;

	*slash= '\0';
	directoryString= XmStringCreateLocalized( filename );
	*slash= '/';

	XtVaSetValues( chooser,
			XmNdirectory,	directoryString,
			NULL );

	XmStringFree( directoryString );
	}

    return;
    }

static void appChooserSetFileExtension(	char *	filename,
					char *	extension,
					Widget	text )
    {
    char *	slash= strrchr( filename, '/' );
    char *	dot;

    if  ( slash )
	{ dot= strrchr( slash+ 1, '.' );	}
    else{ dot= strrchr( filename, '.' );	}

    if  ( dot )
	{
	int	l;
	char *	s;

	l= strlen( filename )+ strlen( extension )+ 2;
	s= (char *)malloc( l );
	strcpy( s, filename );
	strcpy( s+ ( dot- filename )+ 1, extension );

	appChooserSetText( text, s );

	free( s );
	}

    return;
    }

/************************************************************************/
/*  Callback from the file chooser to open a file.			*/
/************************************************************************/
static void appFileChooserOk(	Widget		fileChooser,
				XtPointer	voidaci,
				XtPointer	voidcbs	 )
    {
    AppChooserInformation *	aci= (AppChooserInformation *)voidaci;
    EditApplication *		ea= aci->aciApplication;
    Widget			option= aci->aciOption;

    char *			scratch;
    int				isDir= 0;
    int				fileExists= 0;
    int				fileReadable= 0;
    int				fileNameLength;

    if  ( appChooserGetFilename( voidcbs, fileChooser, aci ) )
	{ return;	}

    fileNameLength= strlen( aci->aciFilename );
    scratch= (char *)malloc( fileNameLength+ 11 );
    if  ( ! scratch )
	{
	XDEB(scratch);
	XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;
	aci->aciResponse= ACIrespFAILURE;
	return;
	}

    fileExists= appTestFileExists( aci->aciFilename ) == 0;

    if  ( fileExists )
	{
	fileReadable= appTestFileReadable( aci->aciFilename ) == 0;
	}
    else{ isDir= appTestDirectory( aci->aciFilename ) == 0; }

    free( scratch );

    if  ( isDir )
	{
	appQuestionRunSubjectErrorDialog( ea, XtParent( fileChooser ), option,
		    aci->aciFilename, aci->aciIsDirecoryMessage );
	XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;
	return;
	}

    if  ( ! fileExists )
	{
	AppFileMessageResources *	afmr= &(ea->eaFileMessageResources);

	appQuestionRunSubjectErrorDialog( ea, XtParent( fileChooser ), option,
		    aci->aciFilename, afmr->afmrNoSuchFileMessage );

	XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;
	return;
	}

    if  ( ! fileReadable )
	{
	appQuestionRunSubjectErrorDialog( ea, XtParent( fileChooser ), option,
		    aci->aciFilename, aci->aciNotReadableMessage );
	XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;
	return;
	}

    if  ( (*aci->aciOpenDocument)( aci->aciThrough,
					    XtParent( fileChooser ), option,
					    aci->aciFilename ) )
	{ XtFree( aci->aciFilename ); aci->aciFilename= (char *)0; return; }

    aci->aciResponse= ACIrespOPEN;

    XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;
    return;
    }

/************************************************************************/
/*									*/
/*  Add a pulldown with filters to a file chooser.			*/
/*									*/
/************************************************************************/
static void appFileFilterChosen(	Widget		w,
					XtPointer	voidaci,
					XtPointer	voidpbcs	 )
    {
    AppChooserInformation *	aci= (AppChooserInformation *)voidaci;
    const AppFileExtension *	afe= aci->aciExtensions;
    Widget			chooser= aci->aciChooser;
    Widget			text;

    int				previousExtension= aci->aciFormat;
    short			extensionChosen= -1;

    char *			filter;
    XmString			filterString;

    char *			fileSelected;

    text= XmFileSelectionBoxGetChild( chooser, XmDIALOG_TEXT );
    fileSelected= XmTextGetString( text );

    XtVaGetValues( w,	XmNpositionIndex,	&extensionChosen,
			NULL );

    if  ( extensionChosen < 0 )
	{ LDEB(extensionChosen); return;	}

    filter= afe[extensionChosen].afeFilter;
    filterString= XmStringCreateLocalized( filter );

    XtVaSetValues( chooser,
			XmNpattern,		filterString,
			NULL );

    if  ( fileSelected )
	{
	char *	slash= strrchr( fileSelected, '/' );
	char *	dot;

	if  ( slash )
	    { dot= strrchr( slash+ 1, '.' );		}
	else{ dot= strrchr( fileSelected, '.' );	}

	if  ( dot							&&
	      previousExtension >= 0					&&
	      afe[previousExtension].afeExtension			&&
	      afe[extensionChosen].afeExtension				&&
	      ! strcmp( dot+ 1, afe[previousExtension].afeExtension )	)
	    {
	    char *	ext= afe[extensionChosen].afeExtension;

	    appChooserSetFileExtension( fileSelected, ext, text );
	    }
	else{ appChooserSetText( text, fileSelected ); }

	XtFree( fileSelected );
	}

    XmStringFree( filterString );

    aci->aciFormat= extensionChosen;

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a filter dropdown in a file chooser.				*/
/*									*/
/************************************************************************/
static Widget appFileMakeFilter(	AppChooserInformation *	aci,
					Widget			chooser )
    {
    Arg				al[20];
    int				ac= 0;

    Widget			child;

    ac= 0;
    child= XmCreateForm( chooser, WIDGET_NAME, al, ac );

    appMakePulldownList( &aci->aciFilterPulldown,
						&aci->aciFilterMenu, child );

    XtManageChild( aci->aciFilterMenu );
    XtManageChild( child );

    return aci->aciFilterMenu;
    }

static void appFileFillFilter(	EditApplication *	ea,
				Widget			chooser,
				Widget			filterMenu,
				AppChooserInformation *	aci,
				AppFileExtension *	extensions,
				int			extensionCount,
				char *			defaultFilter )
    {
    XtResource *	xtr;
    AppFileExtension *	afe;

    Dimension		width;
    int			i;
    Widget		filterPulldown;

    Widget		child0= (Widget)0;
    char *		child0Filter= (char *)0;

    xtr= (XtResource *)malloc( extensionCount* sizeof(XtResource) );
    if  ( ! xtr )
	{ LXDEB(extensionCount,xtr); return;	}

    afe= extensions;
    for ( i= 0; i < extensionCount; afe++, i++ )
	{
	xtr[i].resource_name= afe->afeId;
	xtr[i].resource_class= afe->afeId;
	xtr[i].resource_type= XtRString;
	xtr[i].resource_size= sizeof(char *);
	xtr[i].resource_offset= i* sizeof(AppFileExtension)+
				    offsetof(AppFileExtension,afeDescription);
	xtr[i].default_type= XtRString;
	xtr[i].default_addr= afe->afeDescription;
	}

    XtGetApplicationResources( ea->eaTopWidget, extensions,
						xtr, extensionCount, NULL, 0 );

    free( xtr );

    XtVaGetValues( filterMenu,
			XmNsubMenuId,		&filterPulldown,
			XmNwidth,		&width,
			NULL );

    appEmptyPulldownList( filterPulldown );

    afe= extensions;
    for ( i= 0; i < extensionCount; afe++, i++ )
	{
	Widget	fresh;

	fresh= appPulldownMakeOption( filterPulldown, afe->afeDescription,
				    width, appFileFilterChosen, (void *)aci );

	if  ( ! child0 )
	    {
	    if  ( defaultFilter )
		{
		if  ( ! strcmp( afe->afeFilter, defaultFilter ) )
		    {
		    child0= fresh;
		    child0Filter= afe->afeFilter;
		    }
		}
	    else{
		child0= fresh;
		child0Filter= afe->afeFilter;
		}
	    }
	}

    if  ( child0 )
	{
	XmString	filterString= (XmString)0;

	XtVaSetValues( filterMenu,
			    XmNmenuHistory,	child0,
			    NULL );

	if  ( child0Filter )
	    {
	    filterString= XmStringCreateLocalized( child0Filter );

	    XtVaSetValues( chooser,
				XmNpattern,	filterString,
				NULL );

	    XmStringFree( filterString );
	    }
	}

    aci->aciFilterMenuWidth= width;
    aci->aciExtensions= extensions;
    aci->aciExtensionCount= extensionCount;

    /* Postpone:
    appPulldownSetWidth( filterMenu, width );
    */
    }

/************************************************************************/
/*									*/
/*  Make a chooser.							*/
/*									*/
/************************************************************************/
static Widget appMakeFileChooser( AppChooserInformation **	pAci,
				Widget				relative,
				EditApplication *		ea,
				const char *			defaultFilter,
				Widget				option )
    {
    Widget			chooser;

    Display *			display= ea->eaDisplay;
    int				screen= DefaultScreen( display );
    Widget			text;

    Arg				al[20];
    int				ac= 0;

    XmString			filterString= (XmString)0;

    Widget			helpButton;
    AppChooserInformation *	aci;

    MwmHints			hints;

    aci= (AppChooserInformation *)malloc( sizeof(AppChooserInformation) );
    if  ( ! aci )
	{ XDEB(aci); return (Widget)0;	}

    appInitAci( aci );

    XtGetApplicationResources( ea->eaTopWidget, (void *)aci,
		APP_ChooserResourceTable, XtNumber(APP_ChooserResourceTable),
		NULL, 0 );

    hints.flags= MWM_HINTS_FUNCTIONS|MWM_HINTS_DECORATIONS;
    hints.functions=	MWM_FUNC_MOVE		|
			MWM_FUNC_RESIZE		;
    hints.decorations=	MWM_DECOR_BORDER	|
			MWM_DECOR_RESIZEH	|
			MWM_DECOR_TITLE		|
			MWM_DECOR_MENU		;

    ac= 0;
    XtSetArg( al[ac], XmNmwmDecorations,	hints.decorations ); ac++;
    XtSetArg( al[ac], XmNmwmFunctions,		hints.functions ); ac++;
    XtSetArg( al[ac], XmNdeleteResponse,	XmDO_NOTHING ); ac++;
    XtSetArg( al[ac], XmNallowShellResize,	True ); ac++;

    if  ( relative )
	{ ac += appSetRelativeArgs( al+ ac, relative );	}

    XtSetArg( al[ac], XmNchildPlacement,XmPLACE_ABOVE_SELECTION ); ac++;
    XtSetArg( al[ac], XmNuserData,	(XtPointer)aci ); ac++;
    XtSetArg( al[ac], XmNdialogStyle,	XmDIALOG_APPLICATION_MODAL ); ac++;

    if  ( defaultFilter )
	{
	filterString= XmStringCreateLocalized( (char *)defaultFilter );
	XtSetArg( al[ac], XmNpattern, filterString ); ac++;
	}

    chooser= XmCreateFileSelectionDialog( ea->eaTopWidget,
							WIDGET_NAME, al, ac );

    if  ( filterString )
	{ XmStringFree( filterString );	}

    helpButton= XmFileSelectionBoxGetChild( chooser,
						XmDIALOG_HELP_BUTTON );
    XtUnmanageChild( helpButton );

    text= XmFileSelectionBoxGetChild( chooser, XmDIALOG_FILTER_TEXT );
    XtVaSetValues( text,
			XmNbackground,	WhitePixel( display, screen ),
			NULL );

    text= XmFileSelectionBoxGetChild( chooser, XmDIALOG_TEXT );
    XtVaSetValues( text,
			XmNbackground,	WhitePixel( display, screen ),
			NULL );

    appSetShellTitle( XtParent( chooser ), option, ea->eaApplicationName );

    XtAddEventHandler( XtParent( chooser ), StructureNotifyMask, False,
					appSetSizeAsMinimum, (void *)0 );

    XtAddCallback( chooser,
		    XmNcancelCallback, appChooserCancel, (XtPointer)aci );

    XtAddCallback( chooser,
		    XmNdestroyCallback, appFreeAci, (XtPointer)aci );

    aci->aciChooser= chooser;
    aci->aciApplication= ea;

    *pAci= aci;
    return chooser;
    }

/************************************************************************/
/*									*/
/*  Run some kind of a file chooser.					*/
/*									*/
/************************************************************************/

static void appRunFileChooser(	XtAppContext			appContext,
				AppChooserInformation *		aci,
				Widget				chooser,
				Widget				relative )
    {
    aci->aciResponse= ACIrespNONE;
    aci->aciFilename= (char *)0;

    if  ( relative )
	{ appSetRelativeCallback( relative, chooser );	}

    appDialogRelative( relative, XtParent( chooser ) );

    appSetFocusCallback( chooser );

    XtManageChild( chooser );

    if  ( aci->aciFilterMenu )
	{
	appPulldownSetWidth( aci->aciFilterMenu, aci->aciFilterMenuWidth );

	XtVaSetValues( XtParent( aci->aciFilterMenu ),
				XmNresizePolicy,	XmRESIZE_NONE,
				NULL );
	}

    while( aci->aciResponse == ACIrespNONE )
	{ XtAppProcessEvent( appContext, XtIMAll ); }

    XtUnmanageChild( chooser );

    return;
    }

/************************************************************************/
/*									*/
/*  Open a document.							*/
/*									*/
/************************************************************************/

void appRunOpenChooser(	Widget *		pChooser,
			Widget			option,
			Widget			relative,
			int			extensionCount,
			AppFileExtension *	extensions,
			char *			defaultFilter,
			void *			through,
			APP_OPEN_DOCUMENT	openDocument,
			EditApplication *	ea	)
    {
    Widget			filterMenu= (Widget)0;
    AppChooserInformation *	aci= (AppChooserInformation *)0;

    if  ( ! *pChooser )
	{
	*pChooser= appMakeFileChooser( &aci, relative, ea,
						    defaultFilter, option );

	XtAddCallback( *pChooser,
			XmNokCallback, appFileChooserOk, (XtPointer)aci );

	if  ( extensionCount > 0 )
	    {
	    filterMenu= appFileMakeFilter( aci, *pChooser );

	    XtManageChild( *pChooser );

	    appFileFillFilter( ea, *pChooser, filterMenu, aci,
				extensions, extensionCount, defaultFilter );
	    }
	}
    else{
	XtVaGetValues( *pChooser,
			    XmNuserData,	&aci,
			    NULL );
	if  ( ! aci )
	    { XDEB(aci); return;	}
	}

    aci->aciOption= option;
    aci->aciThrough= through;
    aci->aciOpenDocument= openDocument;
    aci->aciResponse= ACIrespNONE;
    aci->aciFilename= (char *)0;

    if  ( extensionCount > 0 )
	{
	WidgetList		children;
	Cardinal		childCount= 0;
	AppFileExtension *	afe;

	unsigned		i;

	XtVaGetValues( aci->aciFilterPulldown,
			    XmNchildren,		&children,
			    XmNnumChildren,		&childCount,
			    NULL );

	afe= extensions;
	for ( i= 0; i < childCount; afe++, i++ )
	    {
	    if  ( ! ( afe->afeUseFlags & APPFILE_CAN_OPEN )	||
		   ( afe->afeUseFlags & APPFILE_IS_BASIC_TYPE )	)
		{ XtUnmanageChild( children[i] ); continue;	}

	    XtManageChild( children[i] );
	    }
	}

    appRunFileChooser( ea->eaContext, aci, *pChooser, relative );

    return;
    }

static int appChooserOpenDocument(	void *		through,
					Widget		relative,
					Widget		option,
					const char *	filename )
    {
    const int	readonly= 0;

    if  ( ! appOpenDocument( (EditApplication *)through,
				    relative, option, readonly, filename ) )
	{ return -1;	}

    return 0;
    }

void appAppFileOpen(	Widget		option,
			XtPointer	voidea,
			XtPointer	call_data	 )
    {
    EditApplication *	ea= (EditApplication *)voidea;

    appRunOpenChooser( &ea->eaOpenChooser, option, ea->eaTopWidget,
			ea->eaFileExtensionCount, ea->eaFileExtensions,
			ea->eaDefaultFileFilter,
			(void *)ea, appChooserOpenDocument, ea );
    }

void appDocFileOpen(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    appRunOpenChooser( &ea->eaOpenChooser, option, ed->edTopWidget,
			ea->eaFileExtensionCount, ea->eaFileExtensions,
			ea->eaDefaultFileFilter,
			(void *)ea, appChooserOpenDocument, ea );
    }

static void appSaveOkPushed(		Widget		fileChooser,
					XtPointer	voidaci,
					XtPointer	voidcbs	 )
    {
    AppChooserInformation *		aci= (AppChooserInformation *)voidaci;
    EditApplication *			ea= aci->aciApplication;
    Widget				option= aci->aciOption;

    char *				scratch;
    int					isDir= 0;
    int					fileExists= 0;
    int					fileWritable= 0;
    int					dirExists= 1;

    if  ( appChooserGetFilename( voidcbs, fileChooser, aci ) )
	{ return;	}

    scratch= (char *)malloc( strlen( aci->aciFilename )+ 11 );
    if  ( ! scratch )
	{
	XDEB(scratch);
	aci->aciResponse= ACIrespFAILURE;
	return;
	}

    fileExists= appTestFileExists( aci->aciFilename ) == 0;

    if  ( fileExists )
	{ fileWritable= appTestFileWritable( aci->aciFilename ) == 0; }
    else{
	isDir= appTestDirectory( aci->aciFilename ) == 0;
	}

    if  ( ! fileExists && ! isDir )
	{
	char *	slash;

	strcpy( scratch, aci->aciFilename );
	slash= strrchr( scratch, '/' );
	if  ( slash )
	    {
	    *slash= '\0';
	    dirExists= appTestDirectory( scratch ) == 0;
	    }
	}

    free( scratch );

    if  ( isDir )
	{
	appQuestionRunSubjectErrorDialog( ea, XtParent( fileChooser ), option,
		    aci->aciFilename, aci->aciIsDirecoryMessage );

	XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;

	return;
	}

    if  ( fileExists )
	{
	if  ( ! fileWritable )
	    {
	    appQuestionRunSubjectErrorDialog(
			ea, XtParent( fileChooser ), option,
			aci->aciFilename, aci->aciNotWritableMessage );

	    XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;
	    return;
	    }
	else{
	    int		rcc;

	    rcc= appQuestionRunSubjectYesNoCancelDialog(
			ea, XtParent( fileChooser ), option,
			aci->aciFilename, aci->aciOverwriteMessage,
			(char *)0, (char *)0, (char *)0 );

	    switch( rcc )
		{
		case AQDrespYES:
		    aci->aciResponse= ACIrespSAVE;
		    return;
		case AQDrespNO:
		    XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;
		    return;
		default:
		    LDEB(rcc);
		    /*FALLTHROUGH*/
		case AQDrespCANCEL:
		    aci->aciResponse= ACIrespCANCEL;
		    return;
		}
	    }
	}
    else{
	if  ( ! dirExists )
	    {
	    char *	slash= strrchr( aci->aciFilename+ 1, '/' );

	    if  ( slash )
		{
		*slash= '\0';

		appQuestionRunSubjectErrorDialog(
			    ea, XtParent( fileChooser ), option,
			    aci->aciFilename, aci->aciNoSuchDirMessage );

		XtFree( aci->aciFilename ); aci->aciFilename= (char *)0;
		return;
		}
	    }
	}

    aci->aciResponse= ACIrespSAVE;

    return;
    }

/************************************************************************/
/*									*/
/*  Run Filechooser for save.						*/
/*									*/
/************************************************************************/

static int appRunSaveChooser(	Widget			option,
				Widget			relative,
				EditApplication *	ea,
				const EditDocument *	ed,
				unsigned int		useFlags,
				int *			pFormat,
				char **			pFilename )
    {
    Widget			filterMenu= (Widget)0;
    AppChooserInformation *	aci= (AppChooserInformation *)0;

    if  ( ! ea->eaSaveChooser )
	{
	ea->eaSaveChooser= appMakeFileChooser( &aci, relative, ea,
					    ea->eaDefaultFileFilter, option );

	XtAddCallback( ea->eaSaveChooser,
			    XmNokCallback, appSaveOkPushed, (XtPointer)aci );

	if  ( ea->eaFileExtensionCount > 0 )
	    {
	    filterMenu= appFileMakeFilter( aci, ea->eaSaveChooser );

	    XtManageChild( ea->eaSaveChooser );

	    appFileFillFilter( ea, ea->eaSaveChooser, filterMenu, aci,
			ea->eaFileExtensions, ea->eaFileExtensionCount,
			ea->eaDefaultFileFilter );
	    }
	}
    else{
	XtVaGetValues( ea->eaSaveChooser,
			    XmNuserData,	&aci,
			    NULL );
	if  ( ! aci )
	    { XDEB(aci); return ACIrespFAILURE;	}
	}

    aci->aciOption= option;

    if  ( ed->edFilename )
	{ appChooserSetDirectory( ed->edFilename, ea->eaSaveChooser ); }

    if  ( ea->eaFileExtensionCount > 0 )
	{
	WidgetList		children;
	Cardinal		childCount= 0;
	AppFileExtension *	afe;

	int			choice= -1;
	int			i;

	XtVaGetValues( aci->aciFilterPulldown,
			    XmNchildren,		&children,
			    XmNnumChildren,		&childCount,
			    NULL );

	afe= ea->eaFileExtensions;
	for ( i= 0; i < (int)childCount; afe++, i++ )
	    {
	    if  ( ( afe->afeUseFlags & useFlags ) != useFlags )
		{ XtUnmanageChild( children[i] ); continue;	}

	    if  ( ea->eaCanSaveDocument				&&
		  (*ea->eaCanSaveDocument)( ea, ed, i )		)
		{ XtUnmanageChild( children[i] ); continue;	}

	    XtManageChild( children[i] );

	    if  ( i == ed->edFormat || choice < 0 )
		{ choice= i;	}
	    }

	if  ( choice >= 0 )
	    {
	    XmString	filterString;

	    filterString= XmStringCreateLocalized(
				    ea->eaFileExtensions[choice].afeFilter );

	    XtVaSetValues( aci->aciFilterMenu,
			    XmNmenuHistory,		children[choice],
			    NULL );

	    XtVaSetValues( ea->eaSaveChooser,
			    XmNpattern,			filterString,
			    NULL );

	    XmStringFree( filterString );

	    if  ( ed->edFilename )
		{
		Widget	text;

		text= XmFileSelectionBoxGetChild( ea->eaSaveChooser,
							    XmDIALOG_TEXT );

		if  ( choice == ed->edFormat )
		    { appChooserSetText( text, ed->edFilename ); }
		else{
		    char *	ext= ea->eaFileExtensions[choice].afeExtension;

		    if  ( ext )
			{
			appChooserSetFileExtension( ed->edFilename,
								ext, text );
			}
		    }
		}

	    aci->aciFormat= choice;
	    }
	}

    appRunFileChooser( ea->eaContext, aci, ea->eaSaveChooser, ed->edTopWidget );

    if  ( aci->aciResponse == ACIrespSAVE )
	{
	*pFilename= aci->aciFilename;
	*pFormat= aci->aciFormat;
	}

    return aci->aciResponse;
    }

/************************************************************************/
/*  Callback for the file menu.						*/
/************************************************************************/

void appDocFileSaveAs(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    int			response;
    int			format;
    char *		filename;

    if  ( ! ea->eaSaveDocument )
	{ XDEB(ea->eaSaveDocument); return;	}

    response= appRunSaveChooser( option, ed->edTopWidget, ea, ed,
		( APPFILE_CAN_OPEN|APPFILE_CAN_SAVE ),
		&format, &filename );

    switch( response )
	{
	case ACIrespCANCEL:
	    break;
	case ACIrespSAVE:
	    if  ( (*ea->eaSaveDocument)( ea, ed, format, filename ) )
		{
		appReportSaveFailure( ea, option, ed->edTopWidget, filename );
		XtFree( filename ); return;
		}

	    appDocumentChanged( ea, ed, 0 );

	    if  ( appSetDocumentFilename( ed, filename ) )
		{ SDEB(filename);	}
	    if  ( appSetDocumentTitle( ed, filename ) )
		{ SDEB(filename);	}

	    ed->edFileReadOnly= 0;
	    ed->edFormat= format;

	    XtFree( filename );
	    break;
	default:
	    LDEB(response); break;
	}
    }

void appDocFileSaveTo(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    int			response;
    int			format;
    char *		filename;

    if  ( ! ea->eaSaveDocument )
	{ XDEB(ea->eaSaveDocument); return;	}

    response= appRunSaveChooser( option, ed->edTopWidget, ea, ed,
					APPFILE_CAN_SAVE, &format, &filename );

    switch( response )
	{
	case ACIrespCANCEL:
	    break;
	case ACIrespSAVE:
	    if  ( (*ea->eaSaveDocument)( ea, ed, format, filename ) )
		{
		appReportSaveFailure( ea, option, ed->edTopWidget, filename );
		XtFree( filename ); return;
		}

	    appDocumentChanged( ea, ed, 0 );

	    XtFree( filename );
	    break;
	default:
	    LDEB(response); break;
	}
    }

/************************************************************************/
/*  Run Filechooser for print to file.					*/
/************************************************************************/
static int appRunPrintToFileChooser(	Widget			option,
					Widget			panel,
					EditApplication *	ea,
					const EditDocument *	ed,
					char **			pFilename )
    {
    AppChooserInformation *	aci= (AppChooserInformation *)0;

    if  ( ! ea->eaPrintToFileChooser )
	{
	ea->eaPrintToFileChooser=
			appMakeFileChooser( &aci, panel, ea, "*.ps", option );

	XtAddCallback( ea->eaPrintToFileChooser,
			XmNokCallback, appSaveOkPushed, (XtPointer)aci );

	aci->aciExtension= "ps";
	}
    else{
	XtVaGetValues( ea->eaPrintToFileChooser,
			    XmNuserData,	&aci,
			    NULL );
	if  ( ! aci )
	    { XDEB(aci); return ACIrespFAILURE;	}
	}

    aci->aciOption= option;

    aci->aciResponse= ACIrespNONE;
    aci->aciFilename= (char *)0;

    if  ( panel )
	{ appSetRelativeCallback( panel, ea->eaPrintToFileChooser ); }

    if  ( ed->edFilename )
	{
	Widget	text;

	text= XmFileSelectionBoxGetChild(
				    ea->eaPrintToFileChooser, XmDIALOG_TEXT );

	appChooserSetDirectory( ed->edFilename, ea->eaPrintToFileChooser );
	appChooserSetFileExtension( ed->edFilename, "ps", text );
	}

    appRunFileChooser( ea->eaContext, aci, ea->eaPrintToFileChooser, panel );

    if  ( aci->aciResponse == ACIrespSAVE )
	{ *pFilename= aci->aciFilename; }

    return aci->aciResponse;
    }

void appDocPrintToFile(	Widget			option,
			Widget			panel,
			EditApplication *	ea,
			EditDocument *		ed )
    {
    int			response;
    char *		filename;

    FILE *		f;

    if  ( ! ea->eaPrintDocument )
	{ XDEB(ea->eaPrintDocument); return;	}

    response= appRunPrintToFileChooser( option, panel, ea, ed, &filename );

    switch( response )
	{
	case ACIrespCANCEL:
	    break;
	case ACIrespSAVE:
	    f= fopen( filename, "w" );

	    if  ( ! f )
		{ SXDEB(filename,f); XtFree( filename ); return;	}

	    (*ea->eaPrintDocument)( f, ed );

	    fclose( f );

	    XtFree( filename );
	    break;
	default:
	    LDEB(response); break;
	}
    }
