#   include	"config.h"

#   include	<stdlib.h>
#   include	<stdio.h>

#   include	<X11/IntrinsicP.h>
#   include	<X11/StringDefs.h>

#   include	"tedRuler.h"
#   include	<appDraw.h>

#   include	<debugon.h>

#   define	BACK_MARG(s)	((s)/3)
#   define	WHITE_TOP(s)	BACK_MARG(s)
#   define	WHITE_BOTTOM(s)	((s)-BACK_MARG(s))
#   define	WHITE_HEIGHT(s)	(WHITE_BOTTOM(s)-WHITE_TOP(s))

#   define	TAG_DIST(s)	1
#   define	TAG_TOP(s)	(WHITE_TOP(s)+TAG_DIST(s))
#   define	TAG_BOTTOM(s)	(WHITE_BOTTOM(s)-TAG_DIST(s))
#   define	TAG_HEIGHT(s)	(TAG_BOTTOM(s)-TAG_TOP(s))

#   define	FONT_HEIGHT(s)	(WHITE_HEIGHT(s)-2)
#   define	FONT_BOTTOM(s)	(TAG_BOTTOM(s)-1)

#   define	TICK_DIST(s)	((s)/12)
#   define	TICK_TOP(s)	(WHITE_TOP(s)+TICK_DIST(s))
#   define	TICK_BOTTOM(s)	(WHITE_BOTTOM(s)-TICK_DIST(s))
#   define	TICK_HEIGHT(s)	(TICK_BOTTOM(s)-TICK_TOP(s))

#   define	BUTTON_MARG		1

#   define	BUTTON_DIST(s)		(((s)/6)-1)
#   define	BUTTON_RECTH(s)		((s)/6)
#   define	BUTTON_TRIH(s)		((s)/6)
#   define	BUTTON_HWIDTH(s)	((s)/6)


#   define	HBUTTON_BASE(s)		BUTTON_DIST(s)
#   define	HBUTTON_APEX(s)		(HBUTTON_BASE(s)+ \
					    BUTTON_RECTH(s)+BUTTON_TRIH(s))

#   define	SBUTTON_BASE(s)		((s)-BUTTON_DIST(s))
#   define	SBUTTON_APEX(s)		(SBUTTON_BASE(s)- \
					    BUTTON_RECTH(s)-BUTTON_TRIH(s))

#   define	BUTTON_LEFT(v,s)	((v)-BUTTON_HWIDTH((s))-BUTTON_MARG)
#   define	BUTTON_RIGHT(v,s)	((v)+BUTTON_HWIDTH((s))+BUTTON_MARG)

#   define	TAB_HEIGHT(s)		((s)/6)
#   define	TAB_TOP(s)		(WHITE_BOTTOM(s)-TAB_HEIGHT(s))
#   define	TAB_WIDTH(s)		((s)/6)
#   define	TAB_THICK(s)		3

typedef struct TedRuler
    {
    int			trDocBackX0;
    int			trDocBackX1;
    int			trRightRulerWidth;
					/********************************/
					/*  Describe an area that is	*/
					/*  not to be used.		*/
					/********************************/
    int			trDocX0;
    int			trDocX1;

    int			trColX0;
    int			trColX1;
					/********************************/
					/*  Describe the area that is	*/
					/*  accessible as values.	*/
					/********************************/

    int			trDocVisX0;		/*  pixels.		*/
    int			trDocVisX1;		/*  pixels.		*/

    int			trMaximum;		/*  pixels.		*/

    int			trSize;			/*  wide, high: pixels	*/
	
    int			trFirstIndent;		/*  pixels		*/
    int			trLeftIndent;		/*  pixels		*/
    int			trRightIndent;		/*  pixels		*/

    int			trTabCount;
    TabStop *		trTabStops;
    int			trTabKind;		/************************/
						/*  tsKind for fresh	*/
						/*  tabs.		*/
						/************************/

    int			trColumnCount;
    ColumnSeparator *	trColumns;

    int			trTicksPerTag;
    int			trTagsPerText;
    int			trUnitsPerTag;
    double		trTickDistance;		/*  pixels.		*/

    Pixel		trForePixel;
    Pixel		trBackPixel;
    Pixel		trTopPixel;
    Pixel		trBottomPixel;

    XFontStruct *	trFontStruct;
    GC			trGc;
    Pixmap		trBackStipple;

    Dimension		trWidth;		/*  Width of the widget	*/
    } TedRuler;

void * tedMakeRuler(	int			height,
			double			pixPerMM,
			double			magnification,

			Pixel			forePixel,
			Pixel			backPixel,
			Pixel			topPixel,
			Pixel			bottomPixel,

			int			leftRulerWidth,
			int			rightRulerWidth,
			int			docX0,
			int			docX1,

			int			docVisX0,
			int			docVisX1,
			int			maximum,
			int			whatUnit )
    {
    double		pixPerUnit;

    int			tagsPerText;
    int			ticksPerTag;
    int			unitsPerTag;
    double		tickDistance;

    TedRuler *		tr;

    switch( whatUnit )
	{
	case UNITtyMM:	
	    pixPerUnit= pixPerMM;
	    tickDistance= ( 10* pixPerUnit )/2;
	    ticksPerTag= 2;
	    tagsPerText= 5;
	    unitsPerTag= 10;
	    break;
	case UNITtyCM:	
	    pixPerUnit= pixPerMM* 10;
	    tickDistance= pixPerUnit/2;
	    ticksPerTag= 2;
	    tagsPerText= 5;
	    unitsPerTag= 1;
	    break;
	case UNITtyINCH:
	    pixPerUnit= pixPerMM* 25.4;
	    tickDistance= pixPerUnit/4;
	    ticksPerTag= 4;
	    tagsPerText= 2;
	    unitsPerTag= 1;
	    break;
	case UNITtyPOINTS:
	    pixPerUnit= pixPerMM* ( 25.4/ 72.0 );
	    tickDistance= pixPerUnit* 18;
	    ticksPerTag= 4;
	    tagsPerText= 2;
	    unitsPerTag= 72;
	    break;
	case UNITtyPICAS:
	    pixPerUnit= pixPerMM* ( 25.4/ 6.0 );
	    tickDistance= pixPerUnit* 2;
	    ticksPerTag= 2;
	    tagsPerText= 2;
	    unitsPerTag= 8;
	    break;
	default:
	    LDEB(whatUnit); return (void *)0;
	}

    pixPerUnit *= magnification;
    tickDistance *= magnification;

    tr= (TedRuler *)malloc( sizeof( TedRuler ) );
    if  ( ! tr )
	{ XDEB(tr); return (void *)0;	}

    tr->trDocBackX0= leftRulerWidth;
    tr->trDocBackX1= 0;
    tr->trRightRulerWidth= rightRulerWidth;
    tr->trWidth= 0;

    tr->trDocX0= docX0;
    tr->trDocX1= docX1;

    tr->trColX0= docX0;
    tr->trColX1= docX1;

    tr->trDocVisX0= docVisX0;
    tr->trDocVisX1= docVisX1;

    tr->trMaximum= maximum;

    tr->trSize= height;

    tr->trTicksPerTag= ticksPerTag;
    tr->trTagsPerText= tagsPerText;
    tr->trUnitsPerTag= unitsPerTag;
    tr->trTickDistance= tickDistance;

    tr->trForePixel= forePixel;
    tr->trBackPixel= backPixel;
    tr->trTopPixel= topPixel;
    tr->trBottomPixel= bottomPixel;

    tr->trTabCount= 0;
    tr->trTabStops= (TabStop *)0;
    tr->trTabKind= DOCtkLEFT;

    tr->trColumnCount= 0;
    tr->trColumns= (ColumnSeparator *)0;

    tr->trFontStruct= (XFontStruct *)0;
    tr->trGc= (GC)0;
    tr->trBackStipple= None;

    tr->trFirstIndent= -1;
    tr->trLeftIndent= -1;
    tr->trRightIndent= -1;

    return (void *)tr;
    }

void tedFreeRuler(	Display *	display,
			void *		voidtr )
    {
    TedRuler *		tr= (TedRuler *)voidtr;

    if  ( tr->trTabStops )
	{ free( tr->trTabStops );			}

    if  ( tr->trFontStruct )
	{ XFreeFont( display, tr->trFontStruct );	}

    if  ( tr->trGc )
	{ XFreeGC( display, tr->trGc );	}

    if  ( tr->trBackStipple )
	{ XFreePixmap( display, tr->trBackStipple );	}

    free( tr );

    return;
    }

/************************************************************************/
/*									*/
/*  Work around a bug in XDrawLines().					*/
/*									*/
/************************************************************************/
static void editDrawLines(	Display *	display,
				Window		win,
				GC		gc,
				XPoint *	points,
				int		npoints )
    {
    int		i;

    /*
    XDrawLines( display, win, gc, points, npoints, CoordModeOrigin );
    */

    npoints--;

    for ( i= 0; i < npoints; points++, i++ )
	{
	XDrawLine( display, win, gc,
				points[0].x, points[0].y,
				points[1].x, points[1].y );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Draw the little button for the 'First Indent' of a paragraph	*/
/*									*/
/*	       v							*/
/*	4 ---- . ---- 0 = 5						*/
/*	|      .      |							*/
/*	|      .      |							*/
/*	3      .      1							*/
/*	 \     .     / 							*/
/*	  \    .    /  							*/
/*	   \   .   /   							*/
/*	    \  .  /    							*/
/*	     \ . /     							*/
/*	       2       							*/
/*									*/
/************************************************************************/

static void tedRulerDrawHangingButton(	int		value,
					int		size,
					Pixel		forePixel,
					Pixel		backPixel,
					Pixel		topPixel,
					Pixel		bottomPixel,
					Pixmap		stipple,
					Display *	display,
					Window		win,
					GC		gc )
    {
    XPoint	points[6];
    int		w= BUTTON_MARG;

    points[0].x= value+ BUTTON_HWIDTH(size);
    points[0].y= HBUTTON_BASE(size);

    points[1].x= points[0].x;
    points[1].y= HBUTTON_BASE(size)+ BUTTON_RECTH(size);

    points[2].x= value;
    points[2].y= HBUTTON_APEX(size);

    points[3].x= value- BUTTON_HWIDTH(size);
    points[3].y= points[1].y;

    points[4].x= points[3].x;
    points[4].y= points[0].y;

    points[5]= points[0];

    XSetForeground( display, gc, backPixel );
    if  ( stipple )
	{ XSetFillStyle( display, gc, FillStippled ); }

    XFillPolygon( display, win, gc, points, 6, Convex, CoordModeOrigin );

    if  ( stipple )
	{ XSetFillStyle( display, gc, FillSolid ); }

    XSetForeground( display, gc, forePixel );
    editDrawLines( display, win, gc, points, 6 );

    if  ( ! stipple )
	{
	points[0].x -= w; points[0].y += (w+1)/2;
	points[1].x -= w;
	points[2].y -= w;
	points[3].x += w;
	points[4].x += w; points[4].y += w;
	points[5].x -= w; points[5].y += w;

	XSetForeground( display, gc, bottomPixel );
	editDrawLines( display, win, gc, points, 4 );
	XSetForeground( display, gc, topPixel );
	editDrawLines( display, win, gc, points+ 3, 3 );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Draw the little buttons for the 'Left, Right Indent' of a paragraph	*/
/*									*/
/*	       3   							*/
/*	     / . \     							*/
/*	    /  .  \    							*/
/*	   /   .   \   							*/
/*	  /    .    \  							*/
/*	 /     .     \ 							*/
/*	4      .      2							*/
/*	|      .      |							*/
/*	|      .      |							*/
/*  5 = 0 ---- . ---- 1							*/
/*	       v							*/
/*									*/
/************************************************************************/

static void tedRulerDrawStandingButton(	int		value,
						int		size,
						Pixel		forePixel,
						Pixel		backPixel,
						Pixel		topPixel,
						Pixel		bottomPixel,
						Pixmap		stipple,
						Display *	display,
						Window		win,
						GC		gc )
    {
    XPoint	points[6];
    int		w= BUTTON_MARG;

    points[0].x= value- BUTTON_HWIDTH(size);
    points[0].y= SBUTTON_BASE(size);

    points[1].x= value+ BUTTON_HWIDTH(size);
    points[1].y= points[0].y;

    points[2].x= points[1].x;
    points[2].y= SBUTTON_BASE(size)- BUTTON_RECTH(size);

    points[3].x= value;
    points[3].y= SBUTTON_APEX(size);

    points[4].x= points[0].x;
    points[4].y= points[2].y;

    points[5]= points[0];

    XSetForeground( display, gc, backPixel );
    if  ( stipple )
	{ XSetFillStyle( display, gc, FillStippled ); }

    XFillPolygon( display, win, gc, points, 6, Convex, CoordModeOrigin );

    if  ( stipple )
	{ XSetFillStyle( display, gc, FillSolid ); }

    XSetForeground( display, gc, forePixel );
    editDrawLines( display, win, gc, points, 6 );

    if  ( ! stipple )
	{
	points[0].x += (w+1)/2; points[0].y -= w;
	points[1].x -= w; points[1].y -= w;
	points[2].x -= w;
	points[3].y += (w+1)/2;
	points[4].x += w;
	points[5].x += w; points[5].y -= w;

	XSetForeground( display, gc, bottomPixel );
	editDrawLines( display, win, gc, points, 4 );
	XSetForeground( display, gc, topPixel );
	editDrawLines( display, win, gc, points+ 3, 3 );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Draw the little holes for the table columns on the ruler.		*/
/*									*/
/*	3-------------2							*/
/*	|      .      |							*/
/*	|      .      |							*/
/*	|      .      |							*/
/*	|      .      |							*/
/*	|      .      |							*/
/*  4 = 0 ---- . ---- 1							*/
/*	       v							*/
/*									*/
/************************************************************************/

static void tedRulerDrawColumnControl(	int		x0,
					int		x1,
					int		size,
					Pixel		forePixel,
					Pixel		backPixel,
					Pixel		topPixel,
					Pixel		bottomPixel,
					Pixmap		stipple,
					Display *	display,
					Window		win,
					GC		gc )
    {
    XPoint	points[5];
    int		w= BUTTON_MARG;

    if  ( x1- x0 < BUTTON_HWIDTH( size ) )
	{
	int	m= ( x1+ x0 )/ 2;

	x0= m- BUTTON_HWIDTH( size )/2;
	x1= m+ BUTTON_HWIDTH( size )/2;
	}

    points[0].x= x0;
    points[0].y= WHITE_BOTTOM(size);

    points[1].x= x1;
    points[1].y= points[0].y;

    points[2].x= points[1].x;
    points[2].y= WHITE_TOP(size);

    points[3].x= points[0].x;
    points[3].y= points[2].y;

    points[4]= points[0];

    XSetForeground( display, gc, backPixel );
    if  ( stipple )
	{ XSetFillStyle( display, gc, FillStippled ); }

    XFillPolygon( display, win, gc, points, 5, Convex, CoordModeOrigin );

    if  ( stipple )
	{ XSetFillStyle( display, gc, FillSolid ); }

    XSetForeground( display, gc, forePixel );
    editDrawLines( display, win, gc, points, 5 );

    if  ( ! stipple )
	{
	points[0].x += w; points[0].y -= w;
	points[1].x -= w; points[1].y -= w;
	points[2].x -= w; points[2].y += w;
	points[3].x += w; points[3].y += w;
	points[4].x += w; points[4].y -= w;

	XSetForeground( display, gc, topPixel );
	editDrawLines( display, win, gc, points, 3 );
	XSetForeground( display, gc, bottomPixel );
	editDrawLines( display, win, gc, points+ 2, 2 );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Draw a 'Tab' symbol.						*/
/*									*/
/************************************************************************/

static void tedDrawTabSymbol(	Display *	display,
				Window		win,
				GC		gc,
				int		x,
				int		y,
				int		size,
				int		kind )
    {
    int		thick= TAB_THICK(size);
    int		tabWidth= TAB_WIDTH(size);
    int		tabHeight= TAB_HEIGHT(size);

    int		xx;

    xx= x- thick/2;
    XFillRectangle( display, win, gc, xx, y- tabHeight, thick, tabHeight );

    if  ( kind == DOCtkLEFT		||
	  kind == DOCtkCENTRE		||
	  kind == DOCtkDECIMAL		)
	{
	xx= x- thick/2;

	/*  laying right	*/
	XFillRectangle( display, win, gc, xx, y- thick, tabWidth, thick );
	}

    if  ( kind == DOCtkRIGHT		||
	  kind == DOCtkCENTRE		||
	  kind == DOCtkDECIMAL		)
	{
	xx= x+ (thick+1)/2- tabWidth;

	/*  laying left	*/
	XFillRectangle( display, win, gc, xx, y- thick, tabWidth, thick );
	}

    if  ( kind == DOCtkDECIMAL	)
	{ LDEB(kind); }

    return;
    }

static void tedTabButtonRectangle(	DocumentRectangle *	dr,
					int			x,
					int			y,
					int			size )
    {
    int		tw= TAB_WIDTH(size);
    int		bh= 2* TAB_WIDTH(size);

    dr->drX0= x- tw- 1;
    dr->drX1= x+ tw+ 1;
    dr->drY0= y- bh;
    dr->drY1= y;

    return;
    }

static void tedDrawTabButton(	Display *	display,
				Window		win,
				GC		gc,
				Pixel		forePixel,
				Pixel		topPixel,
				Pixel		bottomPixel,
				int		x,
				int		y,
				int		size,
				int		kind )
    {
    int			tw= TAB_WIDTH(size);
    DocumentRectangle	dr;

    tedTabButtonRectangle( &dr, x, y, size );

    XSetForeground( display, gc, forePixel );
    /*  top, bottom */
    XFillRectangle( display, win, gc,
		dr.drX0,	dr.drY0,	dr.drX1- dr.drX0, 1 );
    XFillRectangle( display, win, gc,
		dr.drX0,	dr.drY1,	dr.drX1- dr.drX0, 1 );
    /*  left, right */
    XFillRectangle( display, win, gc,
		dr.drX0,	dr.drY0,	1, dr.drY1- dr.drY0+ 1 );
    XFillRectangle( display, win, gc,
		dr.drX1,	dr.drY0,	1, dr.drY1- dr.drY0+ 1 );

    XSetForeground( display, gc, topPixel );
    /*  top, left */
    XFillRectangle( display, win, gc, dr.drX0+ 1,
		dr.drY0+ 1,	dr.drX1- dr.drX0- 1, 1 );
    XFillRectangle( display, win, gc, dr.drX0+ 1,
		dr.drY0+ 1,	1, dr.drY1- dr.drY0- 1 );

    XSetForeground( display, gc, bottomPixel );
    /*  bottom, right */
    XFillRectangle( display, win, gc,
		dr.drX0+ 2,	dr.drY1- 1,	dr.drX1- dr.drX0- 2, 1 );
    XFillRectangle( display, win, gc,
		dr.drX1- 1,	dr.drY0+ 2,	1, dr.drY1- dr.drY0- 2 );

    XSetForeground( display, gc, forePixel );

    tedDrawTabSymbol( display, win, gc, x, y- tw/2+ 1, size, kind );
    }

/************************************************************************/
/*									*/
/*  Draw a horizontal ruler.						*/
/*									*/
/************************************************************************/
static void tedDrawHorizontalRuler(	Widget			w,
					TedRuler *		tr,
					DocumentRectangle *	drClip,
					int			ox,
					Window			win,
					Display *		display )
    {
    double		item;
    int			tag;
    int			units;

    int			tab;
    TabStop *		ts;

    int			screen= DefaultScreen( display );

    int			size= tr->trSize;

    int			whiteY= WHITE_TOP(size);
    int			whiteH= WHITE_HEIGHT(size);

    int			tagY= TAG_TOP(size);
    int			tagH= TAG_HEIGHT(size);

    int			tickY= TICK_TOP(size);
    int			tickH= TICK_HEIGHT(size);

    int			tabWidth= TAB_WIDTH(size);

    int			fontBottom= FONT_BOTTOM(size);

    ColumnSeparator *	cs;

    DocumentRectangle	drWhite;

    /*  2  */
    XSetForeground( display, tr->trGc, tr->trBackPixel );
    if  ( tr->trBackStipple )
	{ XSetFillStyle( display, tr->trGc, FillStippled ); }

    if  ( tr->trWidth == 0 )
	{
	XtVaGetValues( w, XtNwidth, &tr->trWidth, NULL );
	tr->trDocBackX1= tr->trWidth- tr->trRightRulerWidth;
	}

    XFillRectangle( display, win, tr->trGc,
		    drClip->drX0- ox, drClip->drY0,
		    drClip->drX1- drClip->drX0, drClip->drY1- drClip->drY0 );

    if  ( tr->trBackStipple )
	{ XSetFillStyle( display, tr->trGc, FillSolid ); }

    drWhite.drX0= 0;
    drWhite.drX1= tr->trMaximum;
    drWhite.drY0= whiteY;
    drWhite.drY1= whiteY+ whiteH;

    if  ( docIntersectRectangle( &drWhite, &drWhite, drClip ) )
	{
	int	x0= drWhite.drX0- ox;
	int	x1= drWhite.drX1- ox;

	XSetForeground( display, tr->trGc, WhitePixel( display, screen ) );

	if  ( x0 < tr->trDocBackX0 )
	    { x0=  tr->trDocBackX0;	}
	if  ( x1 > tr->trDocBackX1 )
	    { x1=  tr->trDocBackX1;	}

	XFillRectangle( display, win, tr->trGc,
		    x0, drWhite.drY0,
		    x1- x0, drWhite.drY1- drWhite.drY0 );
	}

    tedDrawTabButton( display, win, tr->trGc,
			tr->trForePixel, tr->trTopPixel, tr->trBottomPixel,
			tr->trDocBackX0, whiteY,
			size, tr->trTabKind );

    XSetForeground( display, tr->trGc, tr->trForePixel );

    item= 0; units= 0; tag= 0;
    while( item < drClip->drX1 )
	{
	int	tick;

	if  ( item- ox >= tr->trDocBackX0	&&
	      item- ox <  tr->trDocBackX1	)
	    {
	    if  ( item == 0 || tag % tr->trTagsPerText )
		{
		XFillRectangle( display, win, tr->trGc,
						(int)item- ox, tagY, 1, tagH );
		}
	    else{
		char	scratch[30];
		int	len;
		int	wide;

		sprintf ( scratch, "%d", units );
		len= strlen( scratch );

		if  ( tr->trFontStruct )
		    { wide= XTextWidth( tr->trFontStruct, scratch, len ); }
		else{ wide= (6*FONT_HEIGHT(size)*len)/10;		   }

		if  ( item- ox- wide/2 >= tr->trDocBackX0	&&
		      item- ox+ wide/2 <  tr->trDocBackX1	)
		    {
		    XDrawString( display, win, tr->trGc, 
			    (int)item- ox- wide/2, fontBottom, scratch, len );
		    }
		else{
		    XFillRectangle( display, win, tr->trGc,
						(int)item- ox, tagY, 1, tagH );
		    }
		}
	    }

	tag++; units += tr->trUnitsPerTag;
	item += tr->trTickDistance;

	for ( tick= 1; tick < tr->trTicksPerTag; tick++ )
	    {
	    if  ( item- ox >= tr->trDocBackX0	&&
		  item- ox <  tr->trDocBackX1	)
		{
		XFillRectangle( display, win, tr->trGc,
					    (int)item- ox, tickY, 1, tickH );
		}

	    item += tr->trTickDistance;
	    }
	}

    cs= tr->trColumns;
    for ( tab= 0; tab < tr->trColumnCount; cs++, tab++ )
	{
	if  ( cs->csX1 >= tr->trDocBackX0		||
	      cs->csX0 <  tr->trDocBackX1		)
	    {
	    tedRulerDrawColumnControl( cs->csX0- ox, cs->csX1- ox, size,
			    tr->trForePixel, tr->trBackPixel,
			    tr->trTopPixel, tr->trBottomPixel,
			    tr->trBackStipple,
			    display, win, tr->trGc );
	    }
	}

    if  ( tr->trFirstIndent >= 0					&&
	  tr->trFirstIndent- ox >= tr->trDocBackX0			&&
	  tr->trFirstIndent- ox <  tr->trDocBackX1			&&
	  BUTTON_LEFT( tr->trFirstIndent,size)  <= drClip->drX1		&&
	  BUTTON_RIGHT(tr->trFirstIndent,size)  >= drClip->drX0		)
	{
	tedRulerDrawHangingButton( tr->trFirstIndent- ox, size,
				tr->trForePixel, tr->trBackPixel,
				tr->trTopPixel, tr->trBottomPixel,
				tr->trBackStipple,
				display, win, tr->trGc );
	}

    if  ( tr->trLeftIndent >= 0						&&
	  tr->trLeftIndent- ox >= tr->trDocBackX0			&&
	  tr->trLeftIndent- ox <  tr->trDocBackX1			&&
	  BUTTON_LEFT( tr->trLeftIndent,size)  <= drClip->drX1		&&
	  BUTTON_RIGHT(tr->trLeftIndent,size)  >= drClip->drX0		)
	{
	tedRulerDrawStandingButton( tr->trLeftIndent- ox, size,
				tr->trForePixel, tr->trBackPixel,
				tr->trTopPixel, tr->trBottomPixel,
				tr->trBackStipple,
				display, win, tr->trGc );
	}

    if  ( tr->trRightIndent >= 0					&&
	  tr->trRightIndent- ox >= tr->trDocBackX0			&&
	  tr->trRightIndent- ox <  tr->trDocBackX1			&&
	  BUTTON_LEFT( tr->trRightIndent,size)  <= drClip->drX1		&&
	  BUTTON_RIGHT(tr->trRightIndent,size)  >= drClip->drX0		)
	{
	tedRulerDrawStandingButton( tr->trRightIndent- ox, size,
				tr->trForePixel, tr->trBackPixel,
				tr->trTopPixel, tr->trBottomPixel,
				tr->trBackStipple,
				display, win, tr->trGc );
	}

    XSetForeground( display, tr->trGc, tr->trForePixel );

    ts= tr->trTabStops;
    for ( tab= 0; tab < tr->trTabCount; ts++, tab++ )
	{
	if  ( ts->tsPixels >  drClip->drX1		||
	      ts->tsPixels+ tabWidth <  drClip->drX0	)
	    { continue;	}

	if  ( ts->tsPixels- ox < tr->trDocBackX0 )
	    { continue;	}
	if  ( ts->tsPixels- ox > tr->trDocBackX1 )
	    { break;	}

	tedDrawTabSymbol( display, win, tr->trGc,
				ts->tsPixels- ox, whiteY+ whiteH,
				size, ts->tsKind );
	}

    XSetClipMask( display, tr->trGc, None );

    return;
    }

void tedExposeHorizontalRuler(	Widget			w,
				void *			voidtr,
				XEvent *		event )
    {
    TedRuler *		tr= (TedRuler *)voidtr;

    Display *		display= XtDisplay( w );
    Window		win= XtWindow( w );

    int			ox= tr->trDocVisX0- tr->trDocBackX0;
    int			oy= 0;

    DocumentRectangle	drClip;

    if  ( ! tr->trGc )
	{
	int	screen= DefaultScreen( display );
	char	name[128];

	tr->trGc= XCreateGC( display, win, 0, NULL );
	XSetLineAttributes( display, tr->trGc,
				1, LineSolid, CapProjecting, JoinMiter );

	sprintf( name,
		    "-*-lucidatypewriter-medium-r-*-*-%d-*-*-*-m-*-iso8859-1",
						    FONT_HEIGHT(tr->trSize) );

	tr->trFontStruct= XLoadQueryFont( display, name );

	if  ( ! tr->trFontStruct )
	    {
	    SXDEB(name,tr->trFontStruct);
	    sprintf( name, "-*-*-medium-r-*-*-%d-*-*-*-m-*-iso8859-1",
						    FONT_HEIGHT(tr->trSize) );

	    tr->trFontStruct= XLoadQueryFont( display, name );
	    }

	if  ( tr->trFontStruct )
	    { XSetFont( display, tr->trGc, tr->trFontStruct->fid );	}
	else{ SXDEB(name,tr->trFontStruct);				}

	if  ( DefaultDepth( display, screen ) <= 2 )
	    {
	    int		i;

	    for ( i= 0; i < 16; i++ )
		{
		memset( name+ 8* i+ 0, 0xaa, 4 );
		memset( name+ 8* i+ 4, 0x55, 4 );
		}

	    tr->trBackStipple= XCreateBitmapFromData( display, win,
							    name, 32, 32 );
	    if  ( ! tr->trBackStipple )
		{ XDEB(tr->trBackStipple);	}

	    tr->trBackPixel= BlackPixel( display, screen );

	    XSetStipple( display, tr->trGc, tr->trBackStipple );
	    }
	}

    appCollectExposures( &drClip, ox, oy, display, win, tr->trGc, event );

    tedDrawHorizontalRuler( w, tr, &drClip, ox, win, display );
    }

void tedHorizontalRulerConfigure(	Widget			w,
					void *			voidtr,
					XEvent *		event,
					Boolean *		pRefused )
    {
    XConfigureEvent *	cevent= &(event->xconfigure);
    TedRuler *		tr= (TedRuler *)voidtr;

    if  ( ! tr )
	{ XDEB(tr); return;	}

    if  ( cevent->type != ConfigureNotify )
	{ return;	}

    tr->trWidth= cevent->width;
    tr->trDocBackX1= cevent->width- tr->trRightRulerWidth;

    return;
    }

/************************************************************************/
/*									*/
/*  Scroll the horizontal ruler.					*/
/*									*/
/************************************************************************/

void tedScrollHorizontalRuler(	void *			voidtr,
				Widget			w,
				int			d	)
    {
    TedRuler *			tr= (TedRuler *)voidtr;
    Display *			display= XtDisplay( w );
    Window			win= XtWindow( w );
    GC				gc= tr->trGc;

    int				wide= tr->trDocBackX1;
    int				high= tr->trSize;

    DocumentRectangle		drClip;

    int				ox= tr->trDocVisX0- tr->trDocBackX0;

    int				x0;

    tr->trDocVisX0 += d;
    tr->trDocVisX1 += d;

    ox= tr->trDocVisX0- tr->trDocBackX0;

    x0= tr->trDocBackX0;

    drClip.drY0= 0;
    drClip.drY1= high;

    if  ( d > 0 )
	{
	if  ( wide > d )
	    {
	    XCopyArea( display, win, win, gc,
					d+ x0, 0, wide- d- x0, high, x0, 0 );

	    drClip.drX0= tr->trDocVisX1- d;
	    drClip.drX1= tr->trDocVisX1+
				    BUTTON_HWIDTH(tr->trSize)+ BUTTON_MARG;
	    }
	else{
	    drClip.drX0= tr->trDocVisX0;
	    drClip.drX1= tr->trDocVisX1+
				    BUTTON_HWIDTH(tr->trSize)+ BUTTON_MARG;
	    }
	}
    else{
	if  ( wide+ d > 0 )
	    {
	    XCopyArea( display, win, win, gc,
					x0, 0, wide+ d- x0, high, x0- d, 0 );

	    drClip.drX0= tr->trDocVisX0-
				    BUTTON_HWIDTH(tr->trSize)- BUTTON_MARG;
	    drClip.drX1= tr->trDocVisX0- d;
	    }
	else{
	    drClip.drX0= tr->trDocVisX0-
				    BUTTON_HWIDTH(tr->trSize)- BUTTON_MARG;
	    drClip.drX1= tr->trDocVisX1;
	    }
	}

    ox= tr->trDocVisX0- tr->trDocBackX0;

    tedDrawHorizontalRuler( w, tr, &drClip, ox, win, display );
    }

void tedSetHorizontalRulerRange(	void *			voidtr,
					Widget			w,
					int			docVisX0,
					int			docVisX1,
					int			docBackX1 )
    {
    TedRuler *			tr= (TedRuler *)voidtr;

    Display *			display= XtDisplay( w );
    Window			win= XtWindow( w );

    tr->trDocVisX0= docVisX0;
    tr->trDocVisX1= docVisX1;
    tr->trDocBackX1= docBackX1;

    XClearArea( display, win, 0, 0, 0, 0, True );

    return;
    }

/************************************************************************/
/*									*/
/*  Let the user drag one of the controls on the top ruler.		*/
/*									*/
/*  1)  Draw a 'hair' over the document that corresponds to the		*/
/*	position of the control.					*/
/*  2)  Until the mouse button is released, observe events.		*/
/*  3)  Merge as many mouse motion events as possible.			*/
/*  4)  If the event is a mouse motion or a button release, redraw the	*/
/*	old an new position of the control on the ruler. (Assume that	*/
/*	the buttons are wider than the tab marks.			*/
/*  5)  Let the document move the hair.					*/
/*  6)  When the mouse button is released, the loop ends.		*/
/*  7)  Let the toolkit dispatch all events that are not relevant for	*/
/*	this routine.							*/
/*									*/
/************************************************************************/
static void tedHorizontalRulerDragItem(	int *		pV0,
						int *		pV1,
						int *		pChanged,
						TedRuler *	tr,
						int		x0,
						int		x1,
						int		ox,
						void *		voided,
						DragHair	dragHair,
						XEvent *	event,
						Widget		w )
    {
    Display *		display= XtDisplay( w );
    Window		win= XtWindow( w );

    int			v0= *pV0;
    int			v1= *pV0;
    int			v= event->xbutton.x+ ox;

    DocumentRectangle	drClip;

    /*  1  */
    (*dragHair)( voided, -1, v0 );
    if  ( pV1 )
	{
	v1= *pV1;
	(*dragHair)( voided, -1, v1 );
	}

    /*  2  */
    for (;;)
	{
	/*  3  */
	if  ( QLength( display ) > 0 )
	    {
	    while( QLength( display ) > 0 )
		{
		XNextEvent( display, event );

		if  ( event->type != MotionNotify )
		    { break;	}
		}
	    }
	else{ XNextEvent( display, event );	}

	if  ( event->type == MotionNotify )
	    { *pChanged= 1;	}

	/*  4  */
	if  ( event->type == ButtonRelease	||
	      event->type == MotionNotify	)
	    {
	    int		nv= event->xbutton.x+ ox;
	    int		nv0= v0+ nv- v;
	    int		nv1= v1+ nv- v;

	    if  ( nv0 < x0 )
		{ nv1 += x0- nv0; nv0 += x0- nv0; }
	    if  ( nv1 > x1 )
		{ nv0 -= nv1- x1; nv1 -= nv1- x1; }

	    if  ( nv0 == v0 && nv1 == v1 && event->type == MotionNotify )
		{ continue;	}

	    *pV0= nv0;
	    if  ( pV1 )
		{ *pV1= nv1;	}

	    if  ( v0 < nv0 )
		{
		drClip.drX0= v0-  BUTTON_HWIDTH(tr->trSize)- BUTTON_MARG;
		drClip.drX1= nv1+ BUTTON_HWIDTH(tr->trSize)+ BUTTON_MARG;
		drClip.drY0= 0;
		drClip.drY1= tr->trSize;
		}
	    else{
		drClip.drX0= nv0- BUTTON_HWIDTH(tr->trSize)- BUTTON_MARG;
		drClip.drX1= v1+  BUTTON_HWIDTH(tr->trSize)+ BUTTON_MARG;
		drClip.drY0= 0;
		drClip.drY1= tr->trSize;
		}

	    tedDrawHorizontalRuler( w, tr, &drClip, ox, win, display );

	    /*  5  */
	    (*dragHair)( voided, v0, nv0 );
	    if  ( pV1 )
		{ (*dragHair)( voided, v1, nv1 ); }

	    v= nv; v0= nv0; v1= nv1;
	    }

	if  ( event->type == MotionNotify )
	    { continue;	}

	/*  6  */
	if  ( event->type == ButtonRelease )
	    {
	    (*dragHair)( voided, *pV0, -1 );
	    if  ( pV1 )
		{ (*dragHair)( voided, *pV1, -1 );	}
	    return;
	    }

	/*  7  */
	XtDispatchEvent( event );
	}
    }

/************************************************************************/
/*									*/
/*  The user clicked on the top ruler, make her happy.			*/
/*									*/
/*  1)  Only accept 'left mouse down' events.				*/
/*  2)  If the click is on one of the tab marks, apparently the user	*/
/*	wants to move the tab.						*/
/*  3)  If the tab was dragged from the ruler, assume that the user	*/
/*	wants to delete it.						*/
/*  4)  Move first indent?						*/
/*  5)  Move left indent?						*/
/*  6)  Move right indent?						*/
/*  7)  Not on one of the controls.. If the click is on the white band,	*/
/*	assume she wants to set a new tab.				*/
/*  8)  Is it on one of the table column separators? Allow the user to	*/
/*	drag it.							*/
/*  9)  Is it on the button with the kind of thabs that is to be added?	*/
/*	Rotate the type.						*/
/*									*/
/************************************************************************/

static int tedCompareTabStopsPixels(	const void *	voidts1,
					const void *	voidts2	)
    {
    const TabStop *	ts1= (const TabStop *)voidts1;
    const TabStop *	ts2= (const TabStop *)voidts2;

    return ts1->tsPixels- ts2->tsPixels;
    }

/*  2  */
static void tedHorizontalRulerDragTab(		int		item,
						int *		pChanged,
						TedRuler *	tr,
						int		ox,
						void *		voided,
						DragHair	dragHair,
						XEvent *	event,
						Widget		w )
    {
    XButtonEvent *	bevent= &(event->xbutton);
    int			size= tr->trSize;

    TabStop *		ts= tr->trTabStops+ item;

    tedHorizontalRulerDragItem( &(ts->tsPixels), (int *)0, pChanged,
			tr, tr->trColX0, tr->trColX1, ox,
			voided, dragHair, event, w );

    /*  3  */
    if  ( bevent->y <  WHITE_TOP(size)				||
	  bevent->y >  WHITE_TOP(size)+ WHITE_HEIGHT(size)	)
	{
	int		i;

	Display *	display= XtDisplay( w );
	Window		win= XtWindow( w );

	tr->trTabCount--;
	for ( i= item; i < tr->trTabCount; i++ )
	    { tr->trTabStops[i]= tr->trTabStops[i+ 1]; }

	XClearArea( display, win, 0, 0, 0, 0, True );
	}

    qsort( tr->trTabStops, tr->trTabCount, sizeof(TabStop),
						tedCompareTabStopsPixels );

    return;
    }

void tedHorizontalRulerTrackMouse(	Widget		w,
					void *		voidtr,
					XEvent *	pEvent,
					unsigned int *	pChange,
					int *		pFirstIndent,
					int *		pLeftIndent,
					int *		pRightIndent,
					int *		pTabCount,
					TabStop **	pTabStops,
					int *		pCsCount,
					ColumnSeparator ** pCs,
					void *		voided,
					DragHair	dragHair )
    {
    Display *		display= XtDisplay( w );
    Window		win= XtWindow( w );

    TedRuler  *		tr= (TedRuler *)voidtr;
    int			size= tr->trSize;

    int			ox= tr->trDocVisX0- tr->trDocBackX0;

    int			item;
    int			tabWidth= TAB_WIDTH(size);

    int			whiteY= WHITE_TOP(size);
    int			whiteH= WHITE_HEIGHT(size);

    XEvent		event= *pEvent;
    XButtonEvent *	bevent= &(event.xbutton);

    int			changed= 0;

    DocumentRectangle	dr;

    /*  1  */
    if  ( event.type != ButtonPress	||
	  bevent->button != Button1	)
	{ *pChange= PPupdNONE; return; }

    /*  2  */
    if  ( bevent->y >= whiteY		&&
	  bevent->y <  whiteY+ whiteH	)
	{
	TabStop *		ts;

	ts= tr->trTabStops;
	for ( item= 0; item < tr->trTabCount; ts++, item++ )
	    {
	    if  ( bevent->x+ ox >= ts->tsPixels			&&
		  bevent->x+ ox <  ts->tsPixels+ tabWidth	)
		{
		tedHorizontalRulerDragTab( item, &changed, tr, ox,
						voided, dragHair, &event, w );

		if  ( changed )
		    {
		    *pTabCount= tr->trTabCount;
		    *pTabStops= tr->trTabStops;
		    *pChange= PPupdTAB_STOPS;
		    }
		else{ *pChange= PPupdNONE; }

		return;
		}
	    }
	}

    /*  4  */
    if  ( tr->trFirstIndent >= 0					&&
	  bevent->y < HBUTTON_APEX(size)				&&
	  bevent->x+ ox >= tr->trFirstIndent- BUTTON_HWIDTH(size)	&&
	  bevent->x+ ox <= tr->trFirstIndent+ BUTTON_HWIDTH(size)	)
	{
	tedHorizontalRulerDragItem( &tr->trFirstIndent, (int *)0, &changed,
				tr, tr->trColX0, tr->trColX1, ox,
				voided, dragHair, &event, w );

	if  ( changed )
	    {
	    *pFirstIndent= tr->trFirstIndent;
	    *pTabCount= tr->trTabCount;
	    *pTabStops= tr->trTabStops;
	    *pChange= PPupdFIRST_INDENT;
	    }
	else{ *pChange= PPupdNONE; }
	return;
	}

    /*  5  */
    if  ( tr->trLeftIndent >= 0						&&
	  bevent->y >= SBUTTON_APEX(size)				&&
	  bevent->y <= SBUTTON_BASE(size)				&&
	  bevent->x+ ox >= tr->trLeftIndent- BUTTON_HWIDTH(size)	&&
	  bevent->x+ ox <= tr->trLeftIndent+ BUTTON_HWIDTH(size)	)
	{
	tedHorizontalRulerDragItem( &tr->trLeftIndent, (int *)0, &changed,
				tr, tr->trColX0, tr->trColX1, ox,
				voided, dragHair, &event, w );

	if  ( changed )
	    {
	    *pLeftIndent= tr->trLeftIndent;
	    *pTabCount= tr->trTabCount;
	    *pTabStops= tr->trTabStops;
	    *pChange= PPupdLEFT_INDENT;
	    }
	else{ *pChange= PPupdNONE; }
	return;
	}

    /*  6  */
    if  ( tr->trRightIndent >= 0					&&
	  bevent->y >= SBUTTON_APEX(size)				&&
	  bevent->y <= SBUTTON_BASE(size)				&&
	  bevent->x+ ox >= tr->trRightIndent- BUTTON_HWIDTH(size)	&&
	  bevent->x+ ox <= tr->trRightIndent+ BUTTON_HWIDTH(size)	)
	{
	tedHorizontalRulerDragItem( &tr->trRightIndent, (int *)0, &changed,
				tr, tr->trColX0, tr->trColX1, ox,
				voided, dragHair, &event, w );

	if  ( changed )
	    {
	    *pRightIndent= tr->trRightIndent;
	    *pTabCount= tr->trTabCount;
	    *pTabStops= tr->trTabStops;
	    *pChange= PPupdRIGHT_INDENT;
	    }
	else{ *pChange= PPupdNONE; }
	return;
	}

    /*  7  */
    if  ( bevent->y >= whiteY			&&
	  bevent->y <  whiteY+ whiteH		&&
	  bevent->x+ ox >= tr->trColX0		&&
	  bevent->x+ ox <= tr->trColX1		)
	{
	int		i;
	TabStop *	fresh;

	fresh= (TabStop *)realloc( tr->trTabStops,
				    ( tr->trTabCount+ 1)* sizeof(TabStop) );
	if  ( ! fresh )
	    { LXDEB(tr->trTabCount,fresh); *pChange= PPupdNONE; return; }

	tr->trTabStops= fresh;

	for ( item= 0; item < tr->trTabCount; item++ )
	    {
	    if  ( tr->trTabStops[item].tsPixels >= bevent->x+ ox )
		{ break;	}
	    }

	for ( i= tr->trTabCount; i > item; i-- )
	    { tr->trTabStops[i]= tr->trTabStops[i-1];	}

	docInitTabStop( tr->trTabStops+ item );

	tr->trTabStops[item].tsPixels= bevent->x+ ox;
	tr->trTabStops[item].tsKind= tr->trTabKind;
	tr->trTabCount++;

	XClearArea( display, win, 0, 0, 0, 0, True );

	tedHorizontalRulerDragTab( item, &changed, tr, ox,
				voided, dragHair, &event, w );

	*pTabCount= tr->trTabCount;
	*pTabStops= tr->trTabStops;
	*pChange= PPupdTAB_STOPS; return;
	}

    /*  8  */
    if  ( bevent->y >= whiteY			&&
	  bevent->y <  whiteY+ whiteH		)
	{
	for ( item= 0; item < tr->trColumnCount; item++ )
	    {
	    if  ( bevent->x+ ox >= tr->trColumns[item].csX0		&&
		  bevent->x+ ox <  tr->trColumns[item].csX1		)
		{
		int		wide;
		int		x0;
		int		x1;

		wide= tr->trColumns[item].csX1- tr->trColumns[item].csX0;
		x0= tr->trDocX0- wide;
		x1= tr->trDocX1+ wide;

		if  ( item > 0 )
		    { x0= tr->trColumns[item- 1].csX1; }
		if  ( item < tr->trColumnCount- 1 )
		    { x1= tr->trColumns[item+ 1].csX0; }

		tedHorizontalRulerDragItem( &tr->trColumns[item].csX0,
				    &tr->trColumns[item].csX1,
				    &changed, tr, x0, x1, ox,
				    voided, dragHair, &event, w );

		if  ( changed )
		    {
		    *pCsCount= tr->trColumnCount; *pCs= tr->trColumns;
		    *pChange= PPupdCOLUMNS;
		    }
		else{ *pChange= PPupdNONE;	}
		return;
		}
	    }
	}

    /*  9  */
    tedTabButtonRectangle( &dr, tr->trDocBackX0, whiteY, size );

    if  ( bevent->x >= dr.drX0 && bevent->x <= dr.drX1	&&
	  bevent->y >= dr.drY0 && bevent->y <= dr.drY1	)
	{
	switch( tr->trTabKind )
	    {
	    case DOCtkLEFT:	tr->trTabKind= DOCtkRIGHT;	break;
	    case DOCtkRIGHT:	tr->trTabKind= DOCtkCENTRE;	break;
	    case DOCtkCENTRE:	tr->trTabKind= DOCtkLEFT;	break;
	    default:
		LDEB(tr->trTabKind);
	    }

	XClearArea( display, win,
			dr.drX0, dr.drY0,
			dr.drX1- dr.drX0+ 1, dr.drY1- dr.drY0+ 1, True );
	}

    *pChange= PPupdNONE; return;
    }

/************************************************************************/
/*									*/
/*  Adapt an horizontal ruler to the paragraph.				*/
/*									*/
/************************************************************************/
void tedAdaptHorizontalRuler(	void *		voidtr,
				Widget		w,
				int		docX0,
				int		docX1,
				int		firstIndent,
				int		leftIndent,
				int		rightIndent,
				int		maximum,
				int		tabCount,
				TabStop *	tabStops,
				int		andExpose )
    {
    TedRuler *		tr= (TedRuler *)voidtr;
    TabStop *		ts;

    int			changed= 0;

    int			i;

    tr->trDocX0= docX0;
    tr->trDocX1= docX1;

    if  ( tr->trFirstIndent != firstIndent )
	{ tr->trFirstIndent=   firstIndent; changed= 1;	}

    if  ( tr->trLeftIndent != leftIndent )
	{ tr->trLeftIndent=   leftIndent; changed= 1;	}

    if  ( tr->trRightIndent != rightIndent )
	{ tr->trRightIndent=   rightIndent; changed= 1;	}

    if  ( tr->trMaximum != maximum )
	{ tr->trMaximum=   maximum; changed= 1;	}

    if  ( tabCount != tr->trTabCount )
	{
	if  ( tabCount > tr->trTabCount )
	    {
	    TabStop *	fresh;

	    fresh= (TabStop *)realloc( tr->trTabStops,
						tabCount* sizeof(TabStop) );
	    if  ( ! fresh )
		{ LXDEB(tabCount,fresh); return;	}

	    tr->trTabStops= fresh;
	    }

	tr->trTabCount= tabCount;
	changed= 1;
	}

    ts= tr->trTabStops;
    for ( i= 0; i < tabCount; ts++, tabStops++, i++ )
	{
	int		different= 0;

	if  ( ts->tsPixels != tabStops->tsPixels )
	    { different= 1; changed= 1;	}
	if  ( ts->tsKind != tabStops->tsKind )
	    { different= 1; changed= 1;	}
	if  ( ts->tsLeader != tabStops->tsLeader )
	    { different= 1;	}

	if  ( different )
	    { *ts= *tabStops;	}
	}

    if  ( changed && andExpose )
	{
	Display *	display= XtDisplay( w );
	Window		win= XtWindow( w );

	XClearArea( display, win, 0, 0, 0, 0, True );
	}

    return;
    }

int tedSetRulerColumns(		Widget			w,
				void *			voidtr,
				int			colX0,
				int			colX1,
				ColumnSeparator *	cs,
				int			csCount )
    {
    TedRuler *		tr= (TedRuler *)voidtr;
    int			changed= 0;
    int			i;

    tr->trColX0= colX0;
    tr->trColX1= colX1;

    if  ( csCount != tr->trColumnCount )
	{
	if  ( csCount > tr->trColumnCount )
	    {
	    ColumnSeparator *	fresh;

	    fresh= (ColumnSeparator *)
		    realloc( tr->trColumns, csCount* sizeof(ColumnSeparator) );
	    if  ( ! fresh )
		{ LXDEB(csCount,fresh); return -1;	}

	    tr->trColumns= fresh;
	    }

	tr->trColumnCount= csCount;
	changed= 1;
	}

    for ( i= 0; i < csCount; i++ )
	{
	if  ( tr->trColumns[i].csX0 != cs[i].csX0 )
	    { tr->trColumns[i].csX0=   cs[i].csX0; changed= 1; }
	if  ( tr->trColumns[i].csX1 != cs[i].csX1 )
	    { tr->trColumns[i].csX1=   cs[i].csX1; changed= 1; }
	}

    if  ( changed && XtIsRealized( w ) )
	{
	Display *	display= XtDisplay( w );
	Window		win= XtWindow( w );

	XClearArea( display, win, 0, 0, 0, 0, True );
	}

    return 0;
    }
