/************************************************************************/
/*									*/
/*  Ted, main module.							*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<signal.h>

#   include	<Xm/Form.h>
#   include	<Xm/DrawingA.h>
#   include	<X11/cursorfont.h>

#   include	"tedApp.h"
#   include	"tedRuler.h"
#   include	"docPs.h"

#   include	<appSymbolPicker.h>
#   include	<appFindTool.h>
#   include	<appSpellTool.h>

#   include	<appGeoString.h>
#   include	<appPaper.h>

#   include	<sioStdio.h>
#   include	<sioMemory.h>

#   include	<debugon.h>

typedef struct TedAppResources
    {
    char *			tarFileNotRtf;

    char *			tarAppFileMenuText;
    char *			tarAppWinMenuText;
    char *			tarDocFileMenuText;
    char *			tarDocEditMenuText;
    char *			tarDocInsertMenuText;
    char *			tarDocWinMenuText;
    char *			tarDocFontMenuText;
    char *			tarDocFormatMenuText;
    char *			tarDocToolMenuText;
    char *			tarDocTableMenuText;
    char *			tarHelpMenuText;
    char *			tarAppHelpFileName;
    } TedAppResources;

# define xx(x)	x,x

static XtResource TEDApplicationResourceTable[]=
    {
	/****************************************************************/
	/*  Application defaults.					*/
	/*  2)  "magnification" is duplicate. It exists for the		*/
	/*	application and for the font chooser tool.		*/
	/****************************************************************/
	{ xx("fileFileNotRtf"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarFileNotRtf),
		    XtRString, "File is not in Rich Text Format." },

	/********************************/
	/*  Application file menu.	*/
	/*  Application windows menu.	*/
	/********************************/
	{ xx("appFileMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarAppFileMenuText),
		    XtRString, "File" },
	{ xx("appWinMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarAppWinMenuText),
		    XtRString, "Window" },

	/********************************/
	/*  Document file menu.		*/
	/*  Document edit menu.		*/
	/*  Document windows menu.	*/
	/*  Document Font menu.		*/
	/*  Document Format menu.	*/
	/*  Document Table menu.	*/
	/*  Document Tool menu.		*/
	/*  Document Help menu.		*/
	/********************************/
	{ xx("docFileMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarDocFileMenuText),
		    XtRString, "File" },
	{ xx("docEditMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarDocEditMenuText),
		    XtRString, "Edit" },
	{ xx("docInsertMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarDocInsertMenuText),
		    XtRString, "Insert" },
	{ xx("docWinMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarDocWinMenuText),
		    XtRString, "Window" },
	{ xx("docFontMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarDocFontMenuText),
		    XtRString, "Font" },
	{ xx("docFormatMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarDocFormatMenuText),
		    XtRString, "Format" },
	{ xx("docTableMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarDocTableMenuText),
		    XtRString, "Table" },
	{ xx("docToolsMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarDocToolMenuText),
		    XtRString, "Tools" },
	{ xx("helpMenuText"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarHelpMenuText),
		    XtRString, "Help" },

	{ xx("documentFileName"), XtRString, sizeof(char *),
		    offsetof(TedAppResources,tarAppHelpFileName),
		    XtRString, INFODIR "/TedDocument.rtf" },

    };


static XtResource TEDFileMessageResourceTable[]=
    {
	/********************************/
	/*  Format for 'New' title	*/
	/********************************/
	{ xx("namelessFormat"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrNamelessTitleFormat),
		    XtRString, "Nameless %d" },

	/********************************/
	/*  Really Close? Dialog.	*/
	/********************************/
	{ xx("reallyCloseQuestion"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrReallyCloseQuestion),
		    XtRString,  "Do you want to save the changes "
				"you made to this text?" },
	{ xx("reallyCloseSave"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrReallyCloseSaveIt),
		    XtRString, "Yes" },
	{ xx("reallyCloseDontSave"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrReallyCloseDontSave),
		    XtRString, "No" },
	{ xx("reallyCloseCancel"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrReallyCloseCancel),
		    XtRString, "Cancel" },

	/********************************/
	/*  Really Quit? Dialog.	*/
	/********************************/
	{ xx("reallyQuitQuestion"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrReallyQuitQuestion),
		    XtRString, "Do you want to save changes to texts?" },
	{ xx("reallyQuitReview"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrReallyQuitReview),
		    XtRString, "Yes" },
	{ xx("reallyQuitQuitAnyway"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrReallyQuitAnyway),
		    XtRString, "No" },
	{ xx("reallyQuitCancel"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrReallyQuitCancel),
		    XtRString, "Cancel" },

	/********************************/
	/*  Messages about  a file.	*/
	/********************************/
	{ xx("fileFileNoAccess"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrFileNoAccess),
		    XtRString, "Cannot open file." },
	{ xx("fileFileReadOnly"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrFileReadOnly),
		    XtRString, "File is Readonly." },
	{ xx("fileNoSuchFile"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrNoSuchFileMessage),
		    XtRString, "This file does not exist." },
	{ xx("fileNotWritable"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrFileNotWritable),
		    XtRString, "You cannot write to this file." },
	{ xx("fileMakeIt"), XtRString, sizeof(char *),
		    offsetof(AppFileMessageResources,afmrMakeItQuestion),
		    XtRString, "There is no such file. Make it?" },

    };

static TedAppResources			TEDResources;

EditDocument *		TED_ManualDocument;

/************************************************************************/
/*									*/
/*  Open a document.							*/
/*									*/
/*  1)  Open an input stream.						*/
/*  2)  Try to read as RTF.						*/
/*  3)  If this fials, try as plain text.				*/
/*	As we cannot save plain text, forget the file name.		*/
/*									*/
/************************************************************************/

static int tedOpenDocumentFile(	EditApplication *	ea,
				int *			pFormat,
				BufferDocument **	pBd,
				int *			pForgetName,
				const char *		filename,
				Widget			relative,
				Widget			option )
    {
    const char *		slash= strrchr( filename, '/' );
    const char *		dot;
    SimpleInputStream *		sis;

    BufferDocument *		bd;
    int				triedRtf= 0;

    int				resp;
    AppFileMessageResources *	afmr= &(ea->eaFileMessageResources);

    if  ( slash )
	{ dot= strrchr( slash+ 1, '.' );	}
    else{ dot= strrchr( filename, '.' );	}

    /*  1  */
    sis= sioInStdioOpen( filename );
    if  ( ! sis )
	{
	appQuestionRunSubjectErrorDialog( ea, relative, option,
				    filename, afmr->afmrFileNoAccess );

	return -1;
	}

    if  ( dot && ! strcmp( dot, ".rtf" ) )
	{
	/*  2  */
	bd= docRtfReadFile( sis );

	sioInClose( sis );
	
	if  ( bd )
	    { *pFormat= 0; *pBd= bd; return 0; }
	else{
	    resp= appQuestionRunSubjectOkCancelDialog( ea,
				    relative, option,
				    filename, TEDResources.tarFileNotRtf,
				    (char *)0, (char *)0 );

	    if  ( resp != AQDrespOK )
		{ return -1;	}

	    triedRtf= 1;
	    }
	}

    if  ( dot && ! strcmp( dot, ".txt" ) )
	{
	bd= docPlainReadFile( sis );

	sioInClose( sis );

	if  ( bd )
	    { *pFormat= 1; *pBd= bd; return 0; }
	else{ SXDEB(filename,bd); return -1;	}
	}

    /*  2  */
    if  ( ! triedRtf )
	{
	bd= docRtfReadFile( sis );
	if  ( bd )
	    { *pFormat= 0; *pBd= bd; return 0; }

	resp= appQuestionRunSubjectOkCancelDialog( ea,
					ea->eaTopWidget, option,
					filename, TEDResources.tarFileNotRtf,
					(char *)0, (char *)0 );

	if  ( resp != AQDrespOK )
	    { return -1;	}
	}

    /*  3  */
    *pForgetName= 1;

    sis= sioInStdioOpen( filename );
    if  ( ! sis )
	{ SXDEB(filename,sis); return -1;	}

    bd= docPlainReadFile( sis );

    sioInClose( sis );

    if  ( bd )
	{ *pFormat= 1; *pBd= bd; return 0; }

    SXDEB(filename,bd); return -1;
    }

/************************************************************************/
/*									*/
/*  Open a document.							*/
/*									*/
/*  1)  Open an input stream.						*/
/*  2)  Try to read as RTF.						*/
/*  3)  If this fials, try as plain text.				*/
/*	As we cannot save plain text, forget the file name.		*/
/*									*/
/************************************************************************/
static int tedOpenDocument(	EditApplication *	ea,
				EditDocument *		ed,
				Widget			relative,
				Widget			option,
				const char *		filename	)
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    int			forgetName= 0;

    if  ( tedOpenDocumentFile( ea, &ed->edFormat, &td->tdDocument, &forgetName,
					    filename, relative, option ) )
	{ /*SDEB(filename);*/ return -1;	}

    /*  3  */
    if  ( ed->edFilename && forgetName )
	{ free( ed->edFilename ); ed->edFilename= (char *)0;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Display the online manual.						*/
/*									*/
/************************************************************************/

static void tedManual(		Widget			option,
				EditApplication *	ea,
				XtPointer		voidpbcs,
				Widget			relative )
    {
    const int		readonly= 1;
    const char *	file= TEDResources.tarAppHelpFileName;

    if  ( TED_ManualDocument )
	{ appDocToFront( option, (void *)TED_ManualDocument, voidpbcs ); }
    else{
	TED_ManualDocument= appOpenDocument( ea, relative,
						    option, readonly, file );

	if  ( ! TED_ManualDocument )
	    { SXDEB(file,TED_ManualDocument);	}
	}

    return;
    }

static void tedDocManual(	Widget		option,
				XtPointer	voided,
				XtPointer	voidpbcs )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    tedManual( option, ea, voidpbcs, ed->edTopWidget );
    }

static void tedAppManual(	Widget		option,
				XtPointer	voidea,
				XtPointer	voidpbcs )
    {
    EditApplication *	ea= (EditApplication *)voidea;

    tedManual( option, ea, voidpbcs, ea->eaTopWidget );
    }

/************************************************************************/
/*									*/
/*  Handle a replace from one of the tools.				*/
/*									*/
/************************************************************************/
void tedAppReplace(	void *			voidea,
			const unsigned char *	word	)
    {
    EditApplication *	ea= (EditApplication *)voidea;
    EditDocument *	ed= ea->eaCurrentDocument;
    TedDocument *	td;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;

    if  ( tedHasSelection( td )		&&
	  ! tedHasIBarSelection( td )	&&
	  td->tdCanReplaceSelection	)
	{ tedAppReplaceSelection( ed, word, strlen( (char *)word ) ); }

    return;
    }

/************************************************************************/
/*									*/
/*  Callback for the 'Insert File' menu option.				*/
/*									*/
/************************************************************************/

static int tedInsertFile(	void *		voided,
				Widget		relative,
				Widget		option,
				const char *	filename )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    int				forgetName= 0;

    int				format;
    BufferDocument *		bd;

    if  ( tedOpenDocumentFile( ea, &format, &bd, &forgetName,
						filename, relative, option ) )
	{ SDEB(filename); return -1;	}

    switch( format )
	{
	case 0:
	    if  ( tedIncludeRtfDocument( ed->edDocumentWidget, ed, bd ) )
		{ LDEB(1); docFreeDocument( bd ); return -1;	}
	    break;
	case 1:
	    if  ( tedIncludePlainDocument( ed->edDocumentWidget, ed, bd ) )
		{ LDEB(1); docFreeDocument( bd ); return -1;	}
	    break;
	default:
	    LDEB(format);
	    docFreeDocument( bd ); return -1;
	}
	    
    docFreeDocument( bd );

    appDocumentChanged( ea, ed, 1 );

    return 0;
    }

void tedDocInsertFile(		Widget		option,
				XtPointer	voided,
				XtPointer	voidpbcs	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    static Widget		insertFileChooser;

    if  ( ! td->tdCanReplaceSelection )
	{ LDEB(td->tdCanReplaceSelection); return;	}

    appRunOpenChooser( &insertFileChooser, option, ed->edTopWidget,
			    ea->eaFileExtensionCount, ea->eaFileExtensions,
			    ea->eaDefaultFileFilter,
			    voided, tedInsertFile, ea );

    return;
    }

/************************************************************************/
/*  Font Menu.								*/
/************************************************************************/

static void tedDocFontPaste(	Widget		fontsOption,
				XtPointer	voided,
				XtPointer	voidpbcbs	 )
    {
    EditDocument *			ed= (EditDocument *)voided;
    XmPushButtonCallbackStruct *	pbcs=
					(XmPushButtonCallbackStruct *)voidpbcbs;

    appDocAskForPaste( ed, "FONT", pbcs->event->xbutton.time );
    }

static void tedDocToolFont(	Widget		fontsOption,
				XtPointer	voided,
				XtPointer	voidcbs	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    tedShowFontsTool( fontsOption, ea, voidcbs );
    }

/************************************************************************/
/*									*/
/*  Table related menu option callbacks.				*/
/*									*/
/************************************************************************/

static void tedDocTableInsertTable(	Widget		option,
					XtPointer	voided,
					XtPointer	voidpbcbs	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    int			rows= 2;
    int			columns= 2;

    if  ( tedInsertTable( ed->edDocumentWidget, ed, rows, columns ) )
	{ LLDEB(rows,columns);	}

    appDocumentChanged( ea, ed, 1 );

    return;
    }

static void tedDocTableAddRow(		Widget		option,
					XtPointer	voided,
					XtPointer	voidpbcbs	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;

    int				rows= 1;

    BufferItem *		sectBi;
    const RowProperties *	rp;

    int				col;
    int				row;
    int				row0;
    int				row1;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return;	}

    rp= &(sectBi->biGroupChildren[row]->biRowProperties);

    if  ( tedInsertRowsInTable( sectBi, rp, ed, row+ 1, rows ) )
	{ LDEB(rows);	}

    appDocumentChanged( ea, ed, 1 );

    return;
    }

static void tedDocTableAddColumn(	Widget		option,
					XtPointer	voided,
					XtPointer	voidpbcbs	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    if  ( tedAppendColumnToTable( ed->edDocumentWidget, ed ) )
	{ LDEB(1);	}

    appDocumentChanged( ea, ed, 1 );
    }

static void tedDocFormatTool(	Widget		option,
				XtPointer	voided,
				XtPointer	voidcbs	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    tedShowFormatTool( option, ea, "tedFormatTool", "tedtable" );

    if  ( ! TED_FormatTool )
	{ XDEB(TED_FormatTool); return;	}

    tedAdaptFormatToolToDocument( TED_FormatTool, ed );
    }

/************************************************************************/
/*									*/
/*  Run document properties dialog.					*/
/*									*/
/************************************************************************/
static void tedDocFileProps(	Widget		option,
				XtPointer	voided,
				XtPointer	call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    tedRunPropertyDialog( ea, ed, option, "props" );
    }

/************************************************************************/
/*									*/
/*  Custom 'Paste' functionality.					*/
/*									*/
/*  1)  The regular X11 manner.						*/
/*									*/
/************************************************************************/

static void tedDocEditPaste(	Widget		option,
				XtPointer	voided,
				XtPointer	voidpbcs	 )
    {
    EditDocument *			ed= (EditDocument *)voided;
    EditApplication *			ea= ed->edApplication;
    TedDocument *			td= (TedDocument *)ed->edPrivateData;
    XmPushButtonCallbackStruct *	pbcs=
					(XmPushButtonCallbackStruct *)voidpbcs;

    int					res;
    BitmapDescription			bd;
    unsigned char *			buffer;

    if  ( ! td->tdCanReplaceSelection )
	{ LDEB(td->tdCanReplaceSelection); return;	}

    res= appDocAskForPaste( ed, "PRIMARY", pbcs->event->xbutton.time );
    if  ( ! res )
	{ return;	}

    if  ( ! appImgGetXvClipboard( &bd, &buffer, ea->eaDisplay, ea->eaScreen ) )
	{
	AppBitmapImage *	abi;

	abi= (AppBitmapImage *)malloc( sizeof(AppBitmapImage) );
	if  ( ! abi )
	    { XDEB(abi); return;	}
	appInitBitmapImage( abi );

	if  ( bmCopyDescription( &(abi->abiBitmap), &bd ) )
	    { bmCleanDescription( &bd ); return; }

	bmCleanDescription( &bd );
	abi->abiBuffer= buffer;

	tedAppReplaceSelection( ed, (unsigned char *)"", 0 );

	if  ( tedObjectInsertBitmap( ed, abi ) )
	    { LDEB(1); return; }

	return;
	}

    return;
    }

/************************************************************************/
/*  Mail a document.							*/
/************************************************************************/
static void tedDocFileMail(	Widget		mailOption,
				XtPointer	voided,
				XtPointer	call_data	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    const unsigned char *	subject= bd->bdSubject;

    if  ( ! subject )
	{ subject= bd->bdTitle;	}

    appRunMailDialog( ea, ed, mailOption, "mail",
					subject, (void *)td->tdDocument );
    }

/************************************************************************/
/*  Application file menu.						*/
/************************************************************************/
static AppMenuItem TED_AppFileMenuItems[]=
{
    {
    "appFileMenuNewText",	"New",
    "appFileMenuNewKey",	"Ctrl <Key>n",
    "appFileMenuNewKeyText",	"^N",		ITEMtyOPTION, appAppFileNew,
    },
    {
    "appFileMenuOpenText",	"Open",
    "appFileMenuOpenKey",	"Ctrl <Key>o",
    "appFileMenuOpenKeyText",	"^O",		ITEMtyOPTION, appAppFileOpen,
    },
    {
    "appFileMenuMiniText",	"Minimize",
    "appFileMenuMiniKey",	"Ctrl <Key>m",
    "appFileMenuMiniKeyText",	"^M",		ITEMtyOPTION, appAppFileMini,
    },
    {
    "appFileMenuHideText",	"Hide",
    "appFileMenuHideKey",	"Ctrl <Key>h",
    "appFileMenuHideKeyText",	"^H",		ITEMtyOPTION, appAppFileHide,
    },
    {
    "appFileMenuQuitText",	"Quit",
    "appFileMenuQuitKey",	(char *)0,
    "appFileMenuQuitKeyText",	(char *)0,	ITEMtyOPTION, appAppFileQuit,
    },
};

/************************************************************************/
/*  Application windows menu.						*/
/************************************************************************/
static AppMenuItem TED_AppWinMenuItems[]=
{
    {
    "appWinMenuMiniText",	"Minimize",
    "appWinMenuMiniKey",	"Ctrl <Key>m",
    "appWinMenuMiniKeyText",	"^M",		ITEMtyOPTION, appAppFileMini,
    },
    {
    "appWinMenuHideText",	"Hide",
    "appWinMenuHideKey",	"Ctrl <Key>h",
    "appWinMenuHideKeyText",	"^H",		ITEMtyOPTION, appAppFileHide,
    },
};

/************************************************************************/
/*  Document file menu.							*/
/************************************************************************/
static AppMenuItem TED_DocFileMenuItems[]=
{
#   define	TEDmiDocFileNew		0
    {
    "docFileMenuNewText",	"New",
    "docFileMenuNewKey",	"Ctrl <Key>n",
    "docFileMenuNewKeyText",	"^N",		ITEMtyOPTION, appDocFileNew,
    },
#   define	TEDmiDocFileOpen	1
    {
    "docFileMenuOpenText",	"Open",
    "docFileMenuOpenKey",	"Ctrl <Key>o",
    "docFileMenuOpenKeyText",	"^O",		ITEMtyOPTION, appDocFileOpen,
    },
#   define	TEDmiDocFileSave	2
    {
    "docFileMenuSaveText",	"Save",
    "docFileMenuSaveKey",	"Ctrl <Key>s",
    "docFileMenuSaveKeyText",	"^S",		ITEMtyOPTION, appDocFileSave,
    },
#   define	TEDmiDocFileSaveAs	3
    {
    "docFileMenuSaveAsText",	"Save As ...",
    "docFileMenuSaveAsKey",	(char *)0,
    "docFileMenuSaveAsKeyText",	(char *)0,	ITEMtyOPTION, appDocFileSaveAs,
    },
#   define	TEDmiDocFileSaveTo	4
    {
    "docFileMenuSaveToText",	"Save To ...",
    "docFileMenuSaveToKey",	(char *)0,
    "docFileMenuSaveToKeyText",	(char *)0,	ITEMtyOPTION, appDocFileSaveTo,
    },
#   define	TEDmiDocFilePrint	5
    {
    "docFileMenuPrintText",	"Print ...",
    "docFileMenuPrintKey",	"Ctrl <Key>p",
    "docFileMenuPrintKeyText",	"^P",		ITEMtyOPTION, appDocFilePrint,
    },
#   define	TEDmiDocFileMail	6
    {
    "docFileMenuMailText",	"Mail ...",
    "docFileMenuMailKey",	(char *)0,
    "docFileMenuMailKeyText",	(char *)0,	ITEMtyOPTION, tedDocFileMail,
    },
#   define	TEDmiDocFileProperties	7
    {
    "docFileMenuPropsText",	"Properties ...",
    "docFileMenuPropsKey",	(char *)0,
    "docFileMenuPropsKeyText",	(char *)0,	ITEMtyOPTION, tedDocFileProps,
    },
#   define	TEDmiDocFileMinimize	8
    {
    "docFileMenuMiniText",	"Minimize",
    "docFileMenuMiniKey",	"Ctrl <Key>m",
    "docFileMenuMiniKeyText",	"^M",		ITEMtyOPTION, appDocFileMini,
    },
#   define	TEDmiDocFileHide	9
    {
    "docFileMenuHideText",	"Hide",
    "docFileMenuHideKey",	"Ctrl <Key>h",
    "docFileMenuHideKeyText",	"^H",		ITEMtyOPTION, appDocFileHide,
    },
#   define	TEDmiDocFileClose	10
    {
    "docFileMenuCloseText",	"Close",
    "docFileMenuCloseKey",	(char *)0,
    "docFileMenuCloseKeyText",	(char *)0,	ITEMtyOPTION, appDocFileClose,
    },
#   define	TEDmiDocFileQuit	11
    {
    "docFileMenuQuitText",	"Quit",
    "docFileMenuQuitKey",	(char *)0,
    "docFileMenuQuitKeyText",	(char *)0,	ITEMtyOPTION, appDocFileQuit,
    },
};

/************************************************************************/
/*  Document edit menu.							*/
/************************************************************************/
static AppMenuItem TED_DocEditMenuItems[]=
{
#   define	TEDmiDocEditCopy	0
    {
    "docEditMenuCopyText",	"Copy",
    "docEditMenuCopyKey",	"Ctrl <Key>c",
    "docEditMenuCopyKeyText",	"^C",		ITEMtyOPTION, appDocEditCopy,
    },
#   define	TEDmiDocEditCut		1
    {
    "docEditMenuCutText",	"Cut",
    "docEditMenuCutKey",	"Ctrl <Key>x",
    "docEditMenuCutKeyText",	"^X",		ITEMtyOPTION, appDocEditCut,
    },
#   define	TEDmiDocEditPaste	2
    {
    "docEditMenuPasteText",	"Paste",
    "docEditMenuPasteKey",	"Ctrl <Key>v",
    "docEditMenuPasteKeyText",	"^V",		ITEMtyOPTION, tedDocEditPaste,
    },
#   define	TEDmiDocEditSelectAll	3
    {
    "docEditMenuSelAllText",	"Select All",
    "docEditMenuSelAllKey",	"Ctrl <Key>a",
    "docEditMenuSelAllKeyText",	"^A",	ITEMtyOPTION, appDocEditSelAll,
    },
};

/************************************************************************/
/*  Document edit menu.							*/
/************************************************************************/
static AppMenuItem TED_DocInsertMenuItems[]=
{
#   define	TEDmiDocInsertInsPict		0
    {
    "docInsertMenuInsPictText",		"Insert Picture ...",
    "docInsertMenuInsPictKey",		(char *)0,
    "docInsertMenuInsPictKeyText",	(char *)0,
					ITEMtyOPTION, tedDocInsertPicture,
    },
#   define	TEDmiDocInsertInsSymbol		1
    {
    "docInsertMenuInsertSymbolText",	"Insert Symbol",
    "docInsertMenuInsertSymbolKey",	(char *)0,
    "docInsertMenuInsertSymbolKeyText",	(char *)0,
					ITEMtyOPTION, tedDocToolInsertSymbol,
    },
#   define	TEDmiDocInsertInsHyperlink	2
    {
    "docInsertMenuHyperlinkText",	"Hyperlink ...",
    "docInsertMenuHyperlinkKey",	"Ctrl <Key>k",
    "docInsertMenuHyperlinkKeyText",	"^K",
					ITEMtyOPTION, tedDocInsertLink,
    },
#   define	TEDmiDocInsertInsBookmark	3
    {
    "docInsertMenuBookmarkText",	"Bookmark ...",
    "docInsertMenuBookmarkKey",		(char *)0,
    "docInsertMenuBookmarkKeyText",	(char *)0,
					ITEMtyOPTION, tedDocInsertBookmark,
    },
#   define	TEDmiDocInsertInsFile		4
    {
    "docInsertMenuInsertFileText",	"Insert File",
    "docInsertMenuInsertFileKey",	(char *)0,
    "docInsertMenuInsertFileKeyText",	(char *)0,
					ITEMtyOPTION, tedDocInsertFile,
    },
#   define	TEDmiDocInsertInsTable		5
    {
    "docInsertMenuInsertTableText",	"Insert Table",
    "docInsertMenuInsertTableKey",	(char *)0,
    "docInsertMenuInsertTableKeyText",	(char *)0,
					ITEMtyOPTION, tedDocTableInsertTable,
    },
};

/************************************************************************/
/*  Document windows menu.						*/
/************************************************************************/
static AppMenuItem TED_DocWinMenuItems[]=
{
    {
    "docWinMenuMiniText",	"Minimize",
    "docWinMenuMiniKey",	"Ctrl <Key>m",
    "docWinMenuMiniKeyText",	"^M",		ITEMtyOPTION, appDocFileMini,
    },
    {
    "docWinMenuHideText",	"Hide",
    "docWinMenuHideKey",	"Ctrl <Key>h",
    "docWinMenuHideKeyText",	"^H",		ITEMtyOPTION, appDocFileHide,
    },
    {
    "docWinMenuCloseText",	"Close",
    "docWinMenuCloseKey",	(char *)0,
    "docWinMenuCloseKeyText",	(char *)0,	ITEMtyOPTION, appDocFileClose,
    },
};

/************************************************************************/
/*  Document font menu.							*/
/************************************************************************/

static void tedDocFontCopy(	Widget		fontsOption,
				XtPointer	voided,
				XtPointer	voidpbcbs );

static AppMenuItem TED_DocFontMenuItems[]=
{
#   define	TEDmiDocFontBold	0
    {
    "docFontMenuBoldText",	"Bold",
    "docFontMenuBoldKey",	(char *)0,
    "docFontMenuBoldKeyText",	(char *)0,
				ITEMtyTOGGLE_OFF, tedDocFontBold,
    },
#   define	TEDmiDocFontItalic	1
    {
    "docFontMenuItalicText",	"Italic",
    "docFontMenuItalicKey",	(char *)0,
    "docFontMenuItalicKeyText",	(char *)0,
				ITEMtyTOGGLE_OFF, tedDocFontItalic,
    },
#   define	TEDmiDocFontUnderlined	2
    {
    "docFontMenuUnderlText",	"Underlined",
    "docFontMenuUnderlKey",	(char *)0,
    "docFontMenuUnderlKeyText",	(char *)0,
				ITEMtyTOGGLE_OFF, tedDocFontUnderlined,
    },
#   define	TEDmiDocFontSuperscript	3
    {
    "docFontMenuSuperscriptText",	"Superscript",
    "docFontMenuSuperscriptKey",	(char *)0,
    "docFontMenuSuperscriptKeyText",	(char *)0,
				ITEMtyTOGGLE_OFF, tedDocFontSupersub,
    },
#   define	TEDmiDocFontSubscript	4
    {
    "docFontMenuSubscriptText",		"Subscript",
    "docFontMenuSubscriptKey",		(char *)0,
    "docFontMenuSubscriptKeyText",	(char *)0,
				ITEMtyTOGGLE_OFF, tedDocFontSupersub,
    },

    { "",	"", "", (char *)0, "", (char *)0, ITEMtySEPARATOR, },

    {
    "docFontMenuCopyText",	"Copy Font",
    "docFontMenuCopyKey",	(char *)0,
    "docFontMenuCopyKeyText",	(char *)0,	ITEMtyOPTION, tedDocFontCopy,
    },
    {
    "docFontMenuPasteText",	"Paste Font",
    "docFontMenuPasteKey",	(char *)0,
    "docFontMenuPasteKeyText",	(char *)0,	ITEMtyOPTION, tedDocFontPaste,
    },

    { "",	"", "", (char *)0, "", (char *)0, ITEMtySEPARATOR, },

    {
    "docFontMenuToolText",	"Font Tool",
    "docFontMenuToolKey",	(char *)0,
    "docFontMenuToolKeyText",	(char *)0,	ITEMtyOPTION, tedDocToolFont,
    },
};

/************************************************************************/
/*  Document 'Table' menu.						*/
/************************************************************************/
static AppMenuItem TED_DocTableMenuItems[]=
{
#   define	TEDmiDocTableInsertTable	0
    {
    "docTableInsertTableText",		"Insert Table",
    "docTableInsertTableKey",		(char *)0,
    "docTableInsertTableKeyText",	(char *)0,
					ITEMtyOPTION, tedDocTableInsertTable,
    },
#   define	TEDmiDocTableAddRow		1
    {
    "docTableAddRowText",		"Add Row",
    "docTableAddRowKey",		(char *)0,
    "docTableAddRowKeyText",		(char *)0,
					ITEMtyOPTION, tedDocTableAddRow,
    },
#   define	TEDmiDocTableAddColumn		2
    {
    "docTableAddColumnText",		"Add Column",
    "docTableAddColumnKey",		(char *)0,
    "docTableAddColumnKeyText",	(char *)0,
					ITEMtyOPTION, tedDocTableAddColumn,
    },

    { "",	"", "", (char *)0, "", (char *)0, ITEMtySEPARATOR, },

#   define	TEDmiDocTableSelectTable	4
    {
    "docTableSelectTableText",		"Select Table",
    "docTableSelectTableKey",		(char *)0,
    "docTableSelectTableKeyText",	(char *)0,
					ITEMtyOPTION, tedDocTableSelectTable,
    },
#   define	TEDmiDocTableSelectRow		5
    {
    "docTableSelectRowText",		"Select Row",
    "docTableSelectRowKey",		(char *)0,
    "docTableSelectRowKeyText",	(char *)0,
					ITEMtyOPTION, tedDocTableSelectRow,
    },
#   define	TEDmiDocTableSelectColumn	6
    {
    "docTableSelectColumnText",		"Select Column",
    "docTableSelectColumnKey",		(char *)0,
    "docTableSelectColumnKeyText",	(char *)0,
					ITEMtyOPTION, tedDocTableSelectColumn,
    },

    { "",	"", "", (char *)0, "", (char *)0, ITEMtySEPARATOR, },

#   define	TEDmiDocTableDrawGrid		8
    {
    "docTableDrawGridText",		"Draw Table Grid",
    "docTableDrawGridKey",		(char *)0,
    "docTableDrawGridKeyText",		(char *)0,
					ITEMtyTOGGLE_ON, tedDocTableDrawGrid,
    },
};

/************************************************************************/
/*  Document format menu.						*/
/************************************************************************/
static AppMenuItem TED_DocFormatMenuItems[]=
{
    /*  0  */
    { "",	"", "", (char *)0, "", (char *)0, ITEMtySEPARATOR, },

#   define	TEDmiDocFormatOnePara		1
    {
    "docFormatMenuOneParaText",		"Make One Paragraph",
    "docFormatMenuOneParaKey",		(char *)0,
    "docFormatMenuOneParaKeyText",	(char *)0,
					ITEMtyOPTION, tedDocFormatOnePara,
    },

    /*  2  */
    { "",	"", "", (char *)0, "", (char *)0, ITEMtySEPARATOR, },

    /*  3  */
    {
    "docFormatMenuCopyRulText",		"Copy ruler",
    "docFormatMenuCopyRulKey",		(char *)0,
    "docFormatMenuCopyRulKeyText",	(char *)0,
					ITEMtyOPTION, tedDocFormatCopyRul,
    },
    /*  4  */
    {
    "docFormatMenuPasteRulText",	"Paste ruler",
    "docFormatMenuPasteRulKey",		(char *)0,
    "docFormatMenuPasteRulKeyText",	(char *)0,
					ITEMtyOPTION, tedDocFormatPasteRul,
    },

    /*  5  */
    { "",	"", "", (char *)0, "", (char *)0, ITEMtySEPARATOR, },

#   define	TEDmiDocFormatTool		6
    {
    "docToolMenuFormatText",		"Format Tool",
    "docToolMenuFormatKey",		(char *)0,
    "docToolMenuFormatKeyText",		(char *)0,
					ITEMtyOPTION, tedDocFormatTool,
    }

};

/************************************************************************/
/*									*/
/*  Document tools menu.						*/
/*									*/
/************************************************************************/

static AppMenuItem TED_DocToolMenuItems[]=
{
#   define	TEDmiDocToolsFontTool		0
    {
    "docToolMenuFontText",		"Font Tool",
    "docToolMenuFontKey",		(char *)0,
    "docToolMenuFontKeyText",		(char *)0,
					ITEMtyOPTION, tedDocToolFont,
    },
#   define	TEDmiDocToolsFind		1
    {
    "docToolMenuFindText",		"Find",
    "docToolMenuFindKey",		"Ctrl <Key>f",
    "docToolMenuFindKeyText",		"^F",
					ITEMtyOPTION, tedDocToolFind,
    },
#   define	TEDmiDocToolsSpelling		2
    {
    "docToolMenuSpellText",		"Spelling",
    "docToolMenuSpellKey",		(char *)0,
    "docToolMenuSpellKeyText",		(char *)0,
					ITEMtyOPTION, tedDocToolSpell,
    },
#   define	TEDmiDocToolsPageLayout		3
    {
    "docToolMenuPageLayoutText",	"Page Layout",
    "docToolMenuPageLayoutKey",		(char *)0,
    "docToolMenuPageLayoutKeyText",	(char *)0,
					ITEMtyOPTION, tedDocShowPageTool,
    },
#   define	TEDmiDocToolsInsertSymbol	4
    {
    "docToolMenuInsertSymbolText",	"Insert Symbol",
    "docToolMenuInsertSymbolKey",	(char *)0,
    "docToolMenuInsertSymbolKeyText",	(char *)0,
					ITEMtyOPTION, tedDocToolInsertSymbol,
    },
#   define	TEDmiDocToolsFormatTool		5
    {
    "docToolMenuFormatText",		"Format Tool",
    "docToolMenuFormatKey",		(char *)0,
    "docToolMenuFormatKeyText",		(char *)0,
					ITEMtyOPTION, tedDocFormatTool,
    },
};

/************************************************************************/
/*									*/
/*  Document, Application help menu.					*/
/*									*/
/************************************************************************/

static AppMenuItem TED_DocHelpMenuItems[]=
{
    {
    "docWinMenuAboutText",	"About Ted",
    "docWinMenuAboutKey",	(char *)0,
    "docWinMenuAboutKeyText",	(char *)0,
				ITEMtyOPTION, appDocAbout,
    },
    {
    "docHelpMenuDocumentText",	"Document",
    "docHelpMenuDocumentKey",	(char *)0,
    "docHelpMenuDocumentKeyText", (char *)0,
				ITEMtyOPTION, tedDocManual,
    },
};

static AppMenuItem TED_AppHelpMenuItems[]=
{
    {
    "docHelpMenuDocumentText",	"Document",
    "docHelpMenuDocumentKey",	(char *)0,
    "docHelpMenuDocumentKeyText", (char *)0,
				ITEMtyOPTION, tedAppManual,
    },
};

/************************************************************************/
/*									*/
/*  Make a document readonly.						*/
/*									*/
/************************************************************************/
void tedMakeDocumentReadonly(	EditDocument *	ed )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    WidgetList		children;
    Cardinal		childCount= 0;

    XtVaGetValues( ed->edFileMenu,
			    XmNchildren,		&children,
			    XmNnumChildren,		&childCount,
			    NULL );

    XtUnmanageChild( children[TEDmiDocFileNew] );
    XtUnmanageChild( children[TEDmiDocFileOpen] );
    XtUnmanageChild( children[TEDmiDocFileSave] );
    XtUnmanageChild( children[TEDmiDocFileSaveAs] );
    XtUnmanageChild( children[TEDmiDocFileSaveTo] );
    XtUnmanageChild( children[TEDmiDocFileMail] );
    XtUnmanageChild( children[TEDmiDocFileProperties] );
    XtUnmanageChild( children[TEDmiDocFileQuit] );

    XtVaGetValues( ed->edEditMenu,
			    XmNchildren,		&children,
			    XmNnumChildren,		&childCount,
			    NULL );

    XtUnmanageChild( children[TEDmiDocEditCut] );
    XtUnmanageChild( children[TEDmiDocEditPaste] );

    XtUnmanageChild( td->tdInsPictOption );
    XtUnmanageChild( td->tdInsSymbolWidget );
    XtUnmanageChild( td->tdInsFileOption );
    XtUnmanageChild( td->tdInsInsertTableWidget );

    XtUnmanageChild( td->tdFontMenuButton );
    XtUnmanageChild( td->tdTableMenuButton );
    XtUnmanageChild( td->tdFormatMenuButton );

    XtVaGetValues( td->tdToolsMenu,
			    XmNchildren,		&children,
			    XmNnumChildren,		&childCount,
			    NULL );

    XtUnmanageChild( children[TEDmiDocToolsFontTool] );
    XtUnmanageChild( children[TEDmiDocToolsSpelling] );
    XtUnmanageChild( children[TEDmiDocToolsPageLayout] );
    XtUnmanageChild( children[TEDmiDocToolsInsertSymbol] );
    XtUnmanageChild( children[TEDmiDocToolsFormatTool] );

    XtUnmanageChild( ed->edHelpMenuButton );
    }

/************************************************************************/
/*									*/
/*  Make private menus per document.					*/
/*									*/
/*  Remember Widget pointers of the menu options to be able to turn	*/
/*  on and off when appropriate.					*/
/*									*/
/************************************************************************/
static void tedMakePrivateDocumentMenus( EditApplication *	ea,
					EditDocument *		ed,
					Widget			menubar )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    td->tdCopyWidget=
	    TED_DocEditMenuItems[TEDmiDocEditCopy].amiOptionWidget;
    td->tdCutWidget=
	    TED_DocEditMenuItems[TEDmiDocEditCut].amiOptionWidget;
    td->tdPasteWidget=
	    TED_DocEditMenuItems[TEDmiDocEditPaste].amiOptionWidget;

    td->tdInsertMenu= appMakeMenu( &(td->tdInsertMenuButton),
			ea, menubar, TEDResources.tarDocInsertMenuText, 0,
			TED_DocInsertMenuItems,
			sizeof(TED_DocInsertMenuItems)/sizeof(AppMenuItem),
			(void *)ed );

    td->tdInsPictOption=
	    TED_DocInsertMenuItems[TEDmiDocInsertInsPict].amiOptionWidget;
    td->tdInsFileOption=
	    TED_DocInsertMenuItems[TEDmiDocInsertInsFile].amiOptionWidget;
    td->tdInsSymbolWidget=
	    TED_DocInsertMenuItems[TEDmiDocInsertInsSymbol].amiOptionWidget;
    td->tdInsHyperlinkWidget=
	    TED_DocInsertMenuItems[TEDmiDocInsertInsHyperlink].amiOptionWidget;
    td->tdInsBookmarkWidget=
	    TED_DocInsertMenuItems[TEDmiDocInsertInsBookmark].amiOptionWidget;
    td->tdInsInsertTableWidget=
	    TED_DocInsertMenuItems[TEDmiDocInsertInsTable].amiOptionWidget;

    td->tdFontMenu= appMakeMenu( &(td->tdFontMenuButton), ea, menubar,
			TEDResources.tarDocFontMenuText, 0,
			TED_DocFontMenuItems,
			sizeof(TED_DocFontMenuItems)/sizeof(AppMenuItem),
			(void *)ed );

    td->tdFontBoldOption=
	    TED_DocFontMenuItems[TEDmiDocFontBold].amiOptionWidget;
    td->tdFontItalicOption=
	    TED_DocFontMenuItems[TEDmiDocFontItalic].amiOptionWidget;
    td->tdFontUnderlinedOption=
	    TED_DocFontMenuItems[TEDmiDocFontUnderlined].amiOptionWidget;
    td->tdFontSuperscriptOption=
	    TED_DocFontMenuItems[TEDmiDocFontSuperscript].amiOptionWidget;
    td->tdFontSubscriptOption=
	    TED_DocFontMenuItems[TEDmiDocFontSubscript].amiOptionWidget;

    td->tdTableMenu= appMakeMenu( &(td->tdTableMenuButton), ea, menubar,
			TEDResources.tarDocTableMenuText, 0,
			TED_DocTableMenuItems,
			sizeof(TED_DocTableMenuItems)/sizeof(AppMenuItem),
			(void *)ed );

    td->tdTabInsertTableOption=
	    TED_DocTableMenuItems[TEDmiDocTableInsertTable].amiOptionWidget;
    td->tdTabAddRowOption=
	    TED_DocTableMenuItems[TEDmiDocTableAddRow].amiOptionWidget;
    td->tdTabAddColumnOption=
	    TED_DocTableMenuItems[TEDmiDocTableAddColumn].amiOptionWidget;

    td->tdSelectTableWidget=
	    TED_DocTableMenuItems[TEDmiDocTableSelectTable].amiOptionWidget;
    td->tdSelectRowWidget=
	    TED_DocTableMenuItems[TEDmiDocTableSelectRow].amiOptionWidget;
    td->tdSelectColumnWidget=
	    TED_DocTableMenuItems[TEDmiDocTableSelectColumn].amiOptionWidget;
    td->tdDrawTableGridWidgetWidget=
	    TED_DocTableMenuItems[TEDmiDocTableDrawGrid].amiOptionWidget;

    td->tdFormatMenu= appMakeMenu( &(td->tdFormatMenuButton), ea, menubar,
			TEDResources.tarDocFormatMenuText, 0,
			TED_DocFormatMenuItems,
			sizeof(TED_DocFormatMenuItems)/sizeof(AppMenuItem),
			(void *)ed );

    td->tdFormatOneParaOption=
	    TED_DocFormatMenuItems[TEDmiDocFormatOnePara].amiOptionWidget;

    td->tdToolsMenu= appMakeMenu( &(td->tdToolsMenuButton),
			ea, menubar, TEDResources.tarDocToolMenuText, 0,
			TED_DocToolMenuItems,
			sizeof(TED_DocToolMenuItems)/sizeof(AppMenuItem),
			(void *)ed );

    return;
    }

/************************************************************************/
/*									*/
/*  Turn tools on/off depending on the number of visible documents.	*/
/*									*/
/************************************************************************/

static void tedVisibleDocumentCountChanged(	EditApplication *	ea,
						int			from,
						int			to )
    {
    if  ( from == 0 && to > 0 )
	{
	/* No !
	if  ( TED_FormatTool )
	    { tedEnableFormatTool( TED_FormatTool, 1 ); }
	*/
	if  ( ea->eaSymbolPicker )
	    { appEnableSymbolPicker( ea->eaSymbolPicker, 1 ); }
	if  ( ea->eaSpellTool )
	    { appEnableSpellTool( ea->eaSpellTool, 1 ); }
	if  ( ea->eaFontsTool )
	    { appEnableFontsTool( ea->eaFontsTool, 1 ); }
	if  ( ea->eaFindTool )
	    { appEnableFindTool( ea->eaFindTool, 1 ); }
	if  ( ea->eaPageTool )
	    { appEnablePageTool( ea->eaPageTool, 1 ); }
	}

    if  ( from > 0 && to == 0 )
	{
	if  ( TED_FormatTool )
	    { tedEnableFormatTool( TED_FormatTool, 0 ); }
	if  ( ea->eaSymbolPicker )
	    { appEnableSymbolPicker( ea->eaSymbolPicker, 0 ); }
	if  ( ea->eaSpellTool )
	    { appEnableSpellTool( ea->eaSpellTool, 0 ); }
	if  ( ea->eaFontsTool )
	    { appEnableFontsTool( ea->eaFontsTool, 0 ); }
	if  ( ea->eaFindTool )
	    { appEnableFindTool( ea->eaFindTool, 0 ); }
	if  ( ea->eaPageTool )
	    { appEnablePageTool( ea->eaPageTool, 0 ); }
	}

    return;
    }

/************************************************************************/
/*  Ted, description of the application.				*/
/************************************************************************/
static AppFileExtension	TedFileExtensions[]=
{
    { "rtfFiles",	"*.rtf",
			"Rich Text Files ( *.rtf )",
			"rtf"	 ,
			( APPFILE_CAN_OPEN|APPFILE_CAN_SAVE ),
			},
    { "txtFiles",	"*.txt",
			"Text Files ( *.txt )",	
			"txt"	 ,
			( APPFILE_CAN_OPEN|APPFILE_CAN_SAVE ),
			},
    { "htmlFiles",	"*.html",
			"HTML Files ( *.html )",
			"html",
			( APPFILE_CAN_SAVE ),
    },
    { "allFiles",	"*",
			"All Files ( *.* )",
			(char *)0,
			( APPFILE_CAN_OPEN ),
    },
};

static int tedWriteRtfMail(	SimpleOutputStream *		sos,
				void *				voidbd )
    {
    const int	saveBookmarks= 1;

    return docRtfSaveDocument( sos,
	    (BufferDocument *)voidbd, (BufferSelection *)0, saveBookmarks );
    }

static int tedWriteHtmlMail(	SimpleOutputStream *		sos,
				void *				voidbd )
    {
    return docHtmlSaveDocument( sos, (BufferDocument *)voidbd, (char *)0 );
    }

static int tedWritePlainMail(	SimpleOutputStream *		sos,
				void *				voidbd )
    {
    return docPlainSaveDocument( sos,
			(BufferDocument *)voidbd, (BufferSelection *)0, 1, 1 );
    }

static MailContent	TedMailContents[]=
    {
	{ "mailPlain",	"text/plain",
			"text", "plain",	tedWritePlainMail	},
	{ "mailRtf",	"application/rtf",
			"application", "rtf",	tedWriteRtfMail		},
	{ "mailHtml",	"text/html",
			"text", "html",		tedWriteHtmlMail	},
    };

static AppDocSelectionTargetType TedPrimaryTargets[]=
    {
	{ "RTF",	None,		tedPastePrimaryRtf,
					tedCopyPrimaryRtf		},
	{ "STRING",	XA_STRING,	tedPastePrimaryString,
					tedCopyPrimaryString		},
	{ "PNG",	None,		tedPastePrimaryPng,
					tedCopyPrimaryPng		},
	{ "PIXMAP",	XA_PIXMAP,	tedPastePrimaryPixmap,
					tedCopyPrimaryPixmap		},
	{ "BITMAP",	XA_BITMAP,	tedPastePrimaryPixmap,
					tedCopyPrimaryPixmap		},
    };

AppDocSelectionTargetType TedPrimaryTextTargets[]=
    {
	{ "RTF",	None,		tedPastePrimaryRtf,
					tedCopyPrimaryRtf		},
	{ "STRING",	XA_STRING,	tedPastePrimaryString,
					tedCopyPrimaryString		},
    };

const int TedPrimaryTextTargetCount=
	    sizeof(TedPrimaryTextTargets)/sizeof(AppDocSelectionTargetType);

AppDocSelectionTargetType TedPrimaryPictureTargets[]=
    {
	{ "RTF",	None,		tedPastePrimaryRtf,
					tedCopyPrimaryRtf		},
	{ "PNG",	None,		tedPastePrimaryPng,
					tedCopyPrimaryPng		},
	{ "PIXMAP",	XA_PIXMAP,	tedPastePrimaryPixmap,
					tedCopyPrimaryPixmap		},
	{ "BITMAP",	XA_BITMAP,	tedPastePrimaryPixmap,
					tedCopyPrimaryPixmap		},
    };

const int TedPrimaryPictureTargetCount=
	    sizeof(TedPrimaryPictureTargets)/sizeof(AppDocSelectionTargetType);

AppDocSelectionTargetType TedRulerTargets[]=
    {
	{ "TED",	None,		tedPasteRulerTed,
					tedCopyRulerTed		},
    };

const int TedRulerTargetCount=
	    sizeof(TedRulerTargets)/sizeof(AppDocSelectionTargetType);

static AppDocSelectionTargetType TedFontTargets[]=
    {
	{ "TED",	None,		tedPasteFontTed,
					tedCopyFontTed		},
    };

static AppDocSelectionType TedDocSelectionTypes[]=
    {
	{ "PRIMARY", XA_PRIMARY,
		TedPrimaryTargets,
		sizeof(TedPrimaryTargets)/sizeof(AppDocSelectionTargetType),
		tedPrimaryLost	},
	{ "FONT", XA_FONT,
		TedFontTargets,
		sizeof(TedFontTargets)/sizeof(AppDocSelectionTargetType), },
	{ "RULER", None,
		TedRulerTargets,
		sizeof(TedRulerTargets)/sizeof(AppDocSelectionTargetType), },
    };

/************************************************************************/
/*									*/
/*  'Copy Font' menu option.						*/
/*									*/
/************************************************************************/

static void tedDocFontCopy(	Widget		fontsOption,
				XtPointer	voided,
				XtPointer	voidpbcbs )
    {
    EditDocument *			ed= (EditDocument *)voided;
    TedDocument *			td= (TedDocument *)ed->edPrivateData;
    SimpleOutputStream *		sos;
    XmPushButtonCallbackStruct *	pbcs=
					(XmPushButtonCallbackStruct *)voidpbcbs;

    char				scratch[120];

    if  ( ! td->tdSelection.bsBegin.bpBi )
	{ XDEB(td->tdSelection.bsBegin.bpBi); return;	}

    if  ( tedGetDocumentAttributeString( scratch, td ) )
	{ LDEB(1); return;	}

    sos= sioOutMemoryOpen( &(td->tdCopiedFont) );
    if  ( ! sos )
	{ XDEB(sos); return;	}

    sioOutPutString( scratch, sos );

    if  ( sioOutClose( sos ) )
	{ LDEB(1); return;	}

    appDocOwnSelection( ed, "FONT",
		    TedFontTargets,
		    sizeof(TedFontTargets)/sizeof(AppDocSelectionTargetType),
		    pbcs->event->xbutton.time );
    }

static EditApplication	TedApplication=
    {
		    /****************************************************/
		    /*  Ted Application name for Xt.			*/
		    /*  Version information for application window.	*/
		    /*  Reference to site for application window.	*/
		    /*  Document shell name for Xt.			*/
		    /*  Page Tool shell name for Xt.			*/
		    /*  Print Dialog shell name for Xt.			*/
		    /*  Mail Dialog shell name for Xt.			*/
		    /*  Message Dialog shell name for Xt.		*/
		    /*  Name of the icon for the application window.	*/
		    /*  Name of the picture for the application window.	*/
		    /****************************************************/
    "Ted",
    "Ted, Version 2.6, September 30, 1999",
    "http://www.nllgg.nl/Ted",


    "tedDocument",
    "tedPageTool",
    "tedPrintDialog",
    "tedMailDialog",
    "tedMessageDialog",
    "tedmain",
    "tedabout",
		    /****************************************************/
		    /*  File extensions, count.				*/
		    /****************************************************/
    TedFileExtensions, sizeof(TedFileExtensions)/sizeof(AppFileExtension),
		    /****************************************************/
		    /*  Default filter for file choosers.		*/
		    /****************************************************/
    "*.rtf",
    "teddoc",
    (void *)&TEDResources,
    TEDApplicationResourceTable,
    sizeof(TEDApplicationResourceTable)/sizeof(XtResource),
    TEDFileMessageResourceTable,
    sizeof(TEDFileMessageResourceTable)/sizeof(XtResource),
		    /****************************************************/
		    /*  Create new document from command line?		*/
		    /****************************************************/
    1,

    0,
    0,
    0.0,
    0.0,
		    /****************************************************/
		    /*  Default Geometry resources.			*/
		    /****************************************************/
    "inch",
    "letter",
    "90pt",
    "72pt",
    "90pt",
    "72pt",
    UNITtyINCH,
	{ 12240, 15840, },
		    /****************************************************/
		    /*  Application File Menu.				*/
		    /****************************************************/
    &(TEDResources.tarAppFileMenuText),
    TED_AppFileMenuItems,
    sizeof(TED_AppFileMenuItems)/sizeof(AppMenuItem),
		    /****************************************************/
		    /*  Application Window Menu.			*/
		    /****************************************************/
    &(TEDResources.tarAppWinMenuText),
    TED_AppWinMenuItems,
    sizeof(TED_AppWinMenuItems)/sizeof(AppMenuItem),
		    /****************************************************/
		    /*  Document File Menu.				*/
		    /****************************************************/
    &(TEDResources.tarDocFileMenuText),
    TED_DocFileMenuItems,
    sizeof(TED_DocFileMenuItems)/sizeof(AppMenuItem),
		    /****************************************************/
		    /*  Document Edit Menu.				*/
		    /****************************************************/
    &(TEDResources.tarDocEditMenuText),
    TED_DocEditMenuItems,
    sizeof(TED_DocEditMenuItems)/sizeof(AppMenuItem),
		    /****************************************************/
		    /*  Document Window Menu.				*/
		    /****************************************************/
    &(TEDResources.tarDocWinMenuText),
    TED_DocWinMenuItems,
    sizeof(TED_DocWinMenuItems)/sizeof(AppMenuItem),
		    /****************************************************/
		    /*  Document, Application Help Menu.		*/
		    /****************************************************/
    &(TEDResources.tarHelpMenuText),
    TED_DocHelpMenuItems,
    sizeof(TED_DocHelpMenuItems)/sizeof(AppMenuItem),

    &(TEDResources.tarHelpMenuText),
    TED_AppHelpMenuItems,
    sizeof(TED_AppHelpMenuItems)/sizeof(AppMenuItem),
		    /****************************************************/
		    /*  Document Widgets.				*/
		    /*  Their initialisation.				*/
		    /*  How to save them.				*/
		    /****************************************************/
    tedSetPrivateData,
    tedMakeDocumentWidget,
    tedOpenDocument,
    tedNewDocument,
    tedSetupDocument,
    tedFinishDocumentSetup,
    NULL,					/*  CanSave		*/
    tedSaveDocument,				/*  Save		*/
    psPrintDocument,				/*  PrintDocument	*/
    tedDrawRectangle,
		    /****************************************************/
		    /*  Track document count.				*/
		    /****************************************************/
    tedVisibleDocumentCountChanged,
		    /****************************************************/
		    /*  Private application,document menus.		*/
		    /****************************************************/
    NULL,
    tedMakePrivateDocumentMenus,
		    /****************************************************/
		    /*  Cut and paste functions. (Document Edit Menu)	*/
		    /****************************************************/
    tedDocCopy,
    tedDocCut,
    tedDocSelAll,

    tedDocHorizontalScrollbarCallback,
    tedDocVerticalScrollbarCallback,

    tedScrollHorizontalRuler,
    tedSetHorizontalRulerRange,
    NULL,				/*  ScrollVerticalRuler		*/
    NULL,				/*  SetVerticalRulerRange	*/

    TedDocSelectionTypes,
    sizeof(TedDocSelectionTypes)/ sizeof(AppDocSelectionType),

    (AppAppSelectionType *)0,
    0/ sizeof(AppAppSelectionType),

    0,0,0,

    TedMailContents,
    sizeof(TedMailContents)/sizeof(MailContent),
    0,
    };

int main(	int		argc,
		char *		argv[]	)
    {
    (void) signal( SIGHUP, SIG_IGN );
    (void) signal( SIGINT, SIG_IGN );

    tedGetNamedPictures( &TedApplication );

    return appMain( &TedApplication, argc, argv );
    }
