#include <m4ri/m4ri.h>
#include "gf2e.h"

gf2e *gf2e_init(const word minpoly) {
  gf2e *ff = (gf2e*)m4ri_mm_calloc(1, sizeof(gf2e));

  for(int i=0; i<=M4RIE_MAX_DEGREE; i++)
    if(1<<i & minpoly)
      ff->degree = i;

  ff->minpoly = minpoly;

  const unsigned int order = __M4RI_TWOPOW(ff->degree);

  /** red **/
  ff->red = (word*)m4ri_mm_calloc(order, sizeof(word));
  for(unsigned int i=1; i<order; i++) {
    word tmp = 0;
    for(deg_t j=0; j<ff->degree; j++)
      if (__M4RI_TWOPOW(j) & i)
        tmp ^= minpoly<<j;
    assert(ff->red[tmp>>ff->degree] == 0);
    ff->red[tmp>>ff->degree] = tmp;
  }

  /** pow_gen: X^i **/
  deg_t n = 2*ff->degree-1;
  ff->pow_gen = (word*)m4ri_mm_malloc( n * sizeof(word));
  for(deg_t i=0; i<n; i++) {
    ff->pow_gen[i] = 1<<i;
    for(deg_t j=i; j>=ff->degree; j--) {
      if (ff->pow_gen[i] & 1<<j)
        ff->pow_gen[i] ^= ff->minpoly<<(j - ff->degree);
    }
  }

  if(ff->degree <= 8) {
    /** mul tables **/
    ff->_mul = (word **)m4ri_mm_calloc(order, sizeof(word *));
    ff->_mul[0] = (word *)m4ri_mm_calloc(order, sizeof(word));
    for(unsigned int i = 1; i<order; i++) {
      ff->_mul[i] = (word *)m4ri_mm_calloc(order, sizeof(word));
      for(unsigned int j=1; j<order; j++) {
        word res = gf2x_mul(i,j, ff->degree);
        ff->_mul[i][j] = res ^ ff->red[res>>ff->degree];
      }
    }
    ff->mul = _gf2e_mul_table;
  } else {
    ff->mul = _gf2e_mul_arith;
  }
  ff->inv = gf2e_inv;
  return ff;
}

void gf2e_free(gf2e *ff) {
  if (ff->_mul) {
    for(size_t i=0; i<__M4RI_TWOPOW(ff->degree); i++) {
      m4ri_mm_free(ff->_mul[i]);
    }
    m4ri_mm_free(ff->_mul);
  }
  m4ri_mm_free(ff->pow_gen);
  m4ri_mm_free(ff->red);
}

const word _irreducible_polynomials_degree_01[   3]  = {    2, 0x00002, 0x00003 };
const word _irreducible_polynomials_degree_02[   2]  = {    1, 0x00007 };
const word _irreducible_polynomials_degree_03[   3]  = {    2, 0x0000b, 0x0000d };
const word _irreducible_polynomials_degree_04[   4]  = {    3, 0x00013, 0x00019, 0x0001f };
const word _irreducible_polynomials_degree_05[   7]  = {    6, 0x00025, 0x00029, 0x0002f, 0x00037, 0x0003b, 0x0003d };
const word _irreducible_polynomials_degree_06[  10]  = {    9, 0x00043, 0x00049, 0x00057, 0x0005b, 0x00061, 0x00067, 0x0006d, 0x00073, 0x00075 };
const word _irreducible_polynomials_degree_07[  19]  = {   18, 0x00083, 0x00089, 0x0008f, 0x00091, 0x0009d, 0x000a7, 0x000ab, 0x000b9, 0x000bf, 0x000c1, 0x000cb, 0x000d3, 0x000d5, 0x000e5, 0x000ef, 0x000f1, 0x000f7, 0x000fd };
const word _irreducible_polynomials_degree_08[  31]  = {   30, 0x0011b, 0x0011d, 0x0012b, 0x0012d, 0x00139, 0x0013f, 0x0014d, 0x0015f, 0x00163, 0x00165, 0x00169, 0x00171, 0x00177, 0x0017b, 0x00187, 0x0018b, 0x0018d, 0x0019f, 0x001a3, 0x001a9, 0x001b1, 0x001bd, 0x001c3, 0x001cf, 0x001d7, 0x001dd, 0x001e7, 0x001f3, 0x001f5, 0x001f9 };
const word _irreducible_polynomials_degree_09[  57]  = {   56, 0x00203, 0x00211, 0x00217, 0x0021b, 0x00221, 0x0022d, 0x00233, 0x0024b, 0x00259, 0x0025f, 0x00265, 0x00269, 0x0026f, 0x00277, 0x0027d, 0x00287, 0x00295, 0x00299, 0x002a3, 0x002a5, 0x002af, 0x002b7, 0x002bd, 0x002cf, 0x002d1, 0x002db, 0x002f5, 0x002f9, 0x00301, 0x00313, 0x00315, 0x0031f, 0x00323, 0x00331, 0x0033b, 0x00349, 0x0034f, 0x0035b, 0x00361, 0x0036b, 0x0036d, 0x00373, 0x0037f, 0x00385, 0x0038f, 0x003a1, 0x003b5, 0x003b9, 0x003c7, 0x003cb, 0x003cd, 0x003d5, 0x003d9, 0x003e3, 0x003e9, 0x003fb };
const word _irreducible_polynomials_degree_10[ 100]  = {   99, 0x00409, 0x0040f, 0x0041b, 0x0041d, 0x00427, 0x0042d, 0x00435, 0x00447, 0x00453, 0x00463, 0x00465, 0x0046f, 0x00481, 0x0048b, 0x00499, 0x004a9, 0x004af, 0x004c5, 0x004c9, 0x004d7, 0x004e7, 0x004ed, 0x004f3, 0x004ff, 0x0050b, 0x0050d, 0x00519, 0x0051f, 0x00523, 0x00531, 0x0053d, 0x00543, 0x00557, 0x00561, 0x00567, 0x0056b, 0x00585, 0x0058f, 0x00597, 0x0059b, 0x005a1, 0x005ab, 0x005b9, 0x005c1, 0x005c7, 0x005e5, 0x005f7, 0x005fb, 0x00613, 0x00615, 0x00623, 0x00625, 0x00631, 0x00637, 0x00643, 0x0064f, 0x00651, 0x0065b, 0x00679, 0x0067f, 0x00685, 0x00689, 0x006a7, 0x006ad, 0x006b5, 0x006bf, 0x006c1, 0x006cd, 0x006d3, 0x006df, 0x006f7, 0x006fd, 0x0070f, 0x00711, 0x00717, 0x0071d, 0x00721, 0x0072b, 0x00735, 0x00739, 0x00747, 0x0074d, 0x00755, 0x00759, 0x00763, 0x0077b, 0x0077d, 0x00781, 0x00787, 0x0078d, 0x00793, 0x007a9, 0x007b1, 0x007c5, 0x007db, 0x007eb, 0x007f3, 0x007f9, 0x007ff };
const word _irreducible_polynomials_degree_11[ 187]  = {  186, 0x00805, 0x00817, 0x0082b, 0x0082d, 0x00847, 0x00863, 0x00865, 0x00871, 0x0087b, 0x0088d, 0x00895, 0x0089f, 0x008a9, 0x008b1, 0x008c3, 0x008cf, 0x008d1, 0x008e1, 0x008e7, 0x008eb, 0x008f5, 0x0090d, 0x00913, 0x00925, 0x00929, 0x00937, 0x0093b, 0x0093d, 0x00945, 0x00949, 0x00951, 0x0095b, 0x00973, 0x00975, 0x0097f, 0x00983, 0x0098f, 0x009ab, 0x009ad, 0x009b9, 0x009c7, 0x009d9, 0x009e5, 0x009ef, 0x009f7, 0x00a01, 0x00a07, 0x00a13, 0x00a15, 0x00a29, 0x00a49, 0x00a61, 0x00a6d, 0x00a79, 0x00a7f, 0x00a85, 0x00a91, 0x00a9d, 0x00aa7, 0x00aab, 0x00ab3, 0x00ab5, 0x00ad5, 0x00adf, 0x00ae3, 0x00ae9, 0x00aef, 0x00af1, 0x00afb, 0x00b03, 0x00b09, 0x00b11, 0x00b33, 0x00b3f, 0x00b41, 0x00b4b, 0x00b59, 0x00b5f, 0x00b65, 0x00b6f, 0x00b7d, 0x00b87, 0x00b8b, 0x00b93, 0x00b95, 0x00baf, 0x00bb7, 0x00bbd, 0x00bc9, 0x00bdb, 0x00bdd, 0x00be7, 0x00bed, 0x00c0b, 0x00c0d, 0x00c19, 0x00c1f, 0x00c31, 0x00c57, 0x00c61, 0x00c6b, 0x00c73, 0x00c75, 0x00c85, 0x00c89, 0x00c97, 0x00c9b, 0x00c9d, 0x00cb3, 0x00cbf, 0x00cc7, 0x00ccd, 0x00cd3, 0x00cd5, 0x00ce3, 0x00ce9, 0x00cf7, 0x00d03, 0x00d0f, 0x00d1d, 0x00d27, 0x00d2d, 0x00d41, 0x00d47, 0x00d55, 0x00d59, 0x00d63, 0x00d6f, 0x00d71, 0x00d93, 0x00d9f, 0x00da9, 0x00dbb, 0x00dbd, 0x00dc9, 0x00dd7, 0x00ddb, 0x00de1, 0x00de7, 0x00df5, 0x00dff, 0x00e05, 0x00e1d, 0x00e21, 0x00e27, 0x00e2b, 0x00e33, 0x00e39, 0x00e47, 0x00e4b, 0x00e55, 0x00e5f, 0x00e71, 0x00e7b, 0x00e7d, 0x00e81, 0x00e93, 0x00e9f, 0x00ea3, 0x00ebb, 0x00ec9, 0x00ecf, 0x00edd, 0x00ef3, 0x00ef9, 0x00f0b, 0x00f19, 0x00f31, 0x00f37, 0x00f5d, 0x00f6b, 0x00f6d, 0x00f75, 0x00f79, 0x00f83, 0x00f91, 0x00f97, 0x00f9b, 0x00fa7, 0x00fad, 0x00fb5, 0x00fcd, 0x00fd3, 0x00fe5, 0x00fe9, 0x00ffb };
const word _irreducible_polynomials_degree_12[ 336]  = {  335, 0x01009, 0x01017, 0x01021, 0x01033, 0x01035, 0x0103f, 0x0104d, 0x01053, 0x01069, 0x01077, 0x0107b, 0x0107d, 0x01081, 0x0108b, 0x01099, 0x010a3, 0x010a5, 0x010cf, 0x010d1, 0x010eb, 0x010ed, 0x010ff, 0x01107, 0x0111f, 0x01123, 0x01131, 0x01137, 0x0113b, 0x0114f, 0x01157, 0x01161, 0x0116b, 0x0116d, 0x01179, 0x01183, 0x01185, 0x01191, 0x011ab, 0x011b3, 0x011d9, 0x011df, 0x011e3, 0x011ef, 0x011f1, 0x01201, 0x0120d, 0x01213, 0x01219, 0x01225, 0x01237, 0x0123b, 0x0123d, 0x01243, 0x01245, 0x01267, 0x0126d, 0x01273, 0x0127f, 0x012b5, 0x012b9, 0x012c1, 0x012cb, 0x01309, 0x0130f, 0x0131b, 0x0131d, 0x01321, 0x01333, 0x01339, 0x0133f, 0x0134d, 0x01365, 0x01371, 0x01399, 0x0139f, 0x013a3, 0x013a9, 0x013af, 0x013bb, 0x013c3, 0x013d1, 0x013d7, 0x013dd, 0x013f3, 0x013f5, 0x01407, 0x01413, 0x01431, 0x01437, 0x01449, 0x0144f, 0x0145b, 0x0145d, 0x01467, 0x0146b, 0x01475, 0x0147f, 0x01489, 0x014a1, 0x014a7, 0x014ad, 0x014b5, 0x014cd, 0x014d3, 0x014d9, 0x014df, 0x014e5, 0x0150f, 0x0151d, 0x0154b, 0x0154d, 0x01581, 0x01593, 0x015a5, 0x015a9, 0x015bb, 0x015c5, 0x015d7, 0x015dd, 0x015eb, 0x015f9, 0x01603, 0x01609, 0x0161b, 0x0163f, 0x01641, 0x01647, 0x0164b, 0x01655, 0x01659, 0x01663, 0x01665, 0x0166f, 0x01693, 0x016a5, 0x016bd, 0x016c3, 0x016c9, 0x016d1, 0x016e1, 0x016e7, 0x016f3, 0x016ff, 0x0170b, 0x01715, 0x01719, 0x0173d, 0x01743, 0x01745, 0x0174f, 0x01757, 0x0175d, 0x01773, 0x01775, 0x01779, 0x01789, 0x0178f, 0x01797, 0x0179d, 0x017ad, 0x017b3, 0x017bf, 0x017c1, 0x017df, 0x017fb, 0x01807, 0x0180d, 0x0181f, 0x01823, 0x01831, 0x01837, 0x01849, 0x01857, 0x0185d, 0x0186d, 0x01879, 0x0187f, 0x01883, 0x01891, 0x01897, 0x018a1, 0x018b9, 0x018cb, 0x018cd, 0x018ef, 0x018f1, 0x018fb, 0x01905, 0x01909, 0x0191b, 0x0192d, 0x01935, 0x01941, 0x0194b, 0x0195f, 0x01965, 0x0196f, 0x0197b, 0x01981, 0x0198b, 0x01999, 0x019b1, 0x019b7, 0x019bd, 0x019c9, 0x019cf, 0x019dd, 0x019e7, 0x019ed, 0x019f9, 0x01a1b, 0x01a1d, 0x01a21, 0x01a2b, 0x01a33, 0x01a4d, 0x01a53, 0x01a55, 0x01a5f, 0x01a63, 0x01a69, 0x01a7b, 0x01a8b, 0x01ab1, 0x01ac5, 0x01ad1, 0x01ae1, 0x01ae7, 0x01aeb, 0x01af5, 0x01b0b, 0x01b0d, 0x01b13, 0x01b19, 0x01b1f, 0x01b2f, 0x01b45, 0x01b57, 0x01b89, 0x01b8f, 0x01b91, 0x01ba7, 0x01bb5, 0x01bb9, 0x01bbf, 0x01bc1, 0x01bcb, 0x01bd3, 0x01be3, 0x01bfd, 0x01c03, 0x01c05, 0x01c0f, 0x01c11, 0x01c17, 0x01c27, 0x01c4d, 0x01c5f, 0x01c87, 0x01c9f, 0x01ca5, 0x01cb7, 0x01cbb, 0x01cc5, 0x01cc9, 0x01ccf, 0x01cd7, 0x01ceb, 0x01ced, 0x01cf3, 0x01d01, 0x01d07, 0x01d23, 0x01d2f, 0x01d3d, 0x01d43, 0x01d51, 0x01d5b, 0x01d5d, 0x01d67, 0x01d75, 0x01d79, 0x01d83, 0x01d85, 0x01d89, 0x01d91, 0x01da7, 0x01db3, 0x01dc1, 0x01def, 0x01e07, 0x01e15, 0x01e19, 0x01e2f, 0x01e3b, 0x01e3d, 0x01e45, 0x01e51, 0x01e5d, 0x01e61, 0x01e67, 0x01e73, 0x01e8f, 0x01e97, 0x01e9b, 0x01eb9, 0x01ebf, 0x01ecd, 0x01ed3, 0x01ee3, 0x01ef1, 0x01ef7, 0x01f03, 0x01f11, 0x01f1b, 0x01f27, 0x01f39, 0x01f47, 0x01f4b, 0x01f53, 0x01f65, 0x01f71, 0x01f7d, 0x01f81, 0x01f8d, 0x01f99, 0x01faf, 0x01fbb, 0x01fbd, 0x01fc3, 0x01fc5, 0x01fc9, 0x01fe1, 0x01fed, 0x01fff };
const word _irreducible_polynomials_degree_13[ 631]  = {  630, 0x0201b, 0x02027, 0x02035, 0x02053, 0x02065, 0x0206f, 0x0208b, 0x0208d, 0x0209f, 0x020a5, 0x020af, 0x020bb, 0x020bd, 0x020c3, 0x020c9, 0x020e1, 0x020f3, 0x0210d, 0x02115, 0x02129, 0x0212f, 0x0213b, 0x02143, 0x02167, 0x0216b, 0x02179, 0x02189, 0x02197, 0x0219d, 0x021bf, 0x021c1, 0x021c7, 0x021cd, 0x021df, 0x021e3, 0x021f1, 0x021fb, 0x02219, 0x02225, 0x02237, 0x0223d, 0x02243, 0x0225b, 0x0225d, 0x02279, 0x0227f, 0x02289, 0x02297, 0x0229b, 0x022b3, 0x022bf, 0x022cd, 0x022ef, 0x022f7, 0x022fb, 0x02305, 0x02327, 0x0232b, 0x02347, 0x02355, 0x02359, 0x0236f, 0x02371, 0x0237d, 0x02387, 0x0238d, 0x02395, 0x023a3, 0x023a9, 0x023b1, 0x023b7, 0x023bb, 0x023e1, 0x023ed, 0x023f9, 0x0240b, 0x02413, 0x0241f, 0x02425, 0x02429, 0x0243d, 0x02451, 0x02457, 0x02461, 0x0246d, 0x0247f, 0x02483, 0x0249b, 0x0249d, 0x024b5, 0x024bf, 0x024c1, 0x024c7, 0x024cb, 0x024e3, 0x02509, 0x02517, 0x0251d, 0x02521, 0x0252d, 0x02539, 0x02553, 0x02555, 0x02563, 0x02571, 0x02577, 0x02587, 0x0258b, 0x02595, 0x02599, 0x0259f, 0x025af, 0x025bd, 0x025c5, 0x025cf, 0x025d7, 0x025eb, 0x02603, 0x02605, 0x02611, 0x0262d, 0x0263f, 0x0264b, 0x02653, 0x02659, 0x02669, 0x02677, 0x0267b, 0x02687, 0x02693, 0x02699, 0x026b1, 0x026b7, 0x026bd, 0x026c3, 0x026eb, 0x026f5, 0x02713, 0x02729, 0x0273b, 0x0274f, 0x02757, 0x0275d, 0x0276b, 0x02773, 0x02779, 0x02783, 0x02791, 0x027a1, 0x027b9, 0x027c7, 0x027cb, 0x027df, 0x027ef, 0x027f1, 0x02807, 0x02819, 0x0281f, 0x02823, 0x02831, 0x0283b, 0x0283d, 0x02845, 0x02867, 0x02875, 0x02885, 0x028ab, 0x028ad, 0x028bf, 0x028cd, 0x028d5, 0x028df, 0x028e3, 0x028e9, 0x028fb, 0x02909, 0x0290f, 0x02911, 0x0291b, 0x0292b, 0x02935, 0x0293f, 0x02941, 0x0294b, 0x02955, 0x02977, 0x0297d, 0x02981, 0x02993, 0x0299f, 0x029af, 0x029b7, 0x029bd, 0x029c3, 0x029d7, 0x029f3, 0x029f5, 0x02a03, 0x02a0f, 0x02a1d, 0x02a21, 0x02a33, 0x02a35, 0x02a4d, 0x02a69, 0x02a6f, 0x02a71, 0x02a7b, 0x02a7d, 0x02aa5, 0x02aa9, 0x02ab1, 0x02ac5, 0x02ad7, 0x02adb, 0x02aeb, 0x02af3, 0x02b01, 0x02b15, 0x02b23, 0x02b25, 0x02b2f, 0x02b37, 0x02b43, 0x02b49, 0x02b6d, 0x02b7f, 0x02b85, 0x02b97, 0x02b9b, 0x02bad, 0x02bb3, 0x02bd9, 0x02be5, 0x02bfd, 0x02c0f, 0x02c21, 0x02c2b, 0x02c2d, 0x02c3f, 0x02c41, 0x02c4d, 0x02c71, 0x02c8b, 0x02c8d, 0x02c95, 0x02ca3, 0x02caf, 0x02cbd, 0x02cc5, 0x02cd1, 0x02cd7, 0x02ce1, 0x02ce7, 0x02ceb, 0x02d0d, 0x02d19, 0x02d29, 0x02d2f, 0x02d37, 0x02d3b, 0x02d45, 0x02d5b, 0x02d67, 0x02d75, 0x02d89, 0x02d8f, 0x02da7, 0x02dab, 0x02db5, 0x02de3, 0x02df1, 0x02dfd, 0x02e07, 0x02e13, 0x02e15, 0x02e29, 0x02e49, 0x02e4f, 0x02e5b, 0x02e5d, 0x02e61, 0x02e6b, 0x02e8f, 0x02e91, 0x02e97, 0x02e9d, 0x02eab, 0x02eb3, 0x02eb9, 0x02edf, 0x02efb, 0x02efd, 0x02f05, 0x02f09, 0x02f11, 0x02f17, 0x02f3f, 0x02f41, 0x02f4b, 0x02f4d, 0x02f59, 0x02f5f, 0x02f65, 0x02f69, 0x02f95, 0x02fa5, 0x02faf, 0x02fb1, 0x02fcf, 0x02fdd, 0x02fe7, 0x02fed, 0x02ff5, 0x02fff, 0x03007, 0x03015, 0x03019, 0x0302f, 0x03049, 0x0304f, 0x03067, 0x03079, 0x0307f, 0x03091, 0x030a1, 0x030b5, 0x030bf, 0x030c1, 0x030d3, 0x030d9, 0x030e5, 0x030ef, 0x03105, 0x0310f, 0x03135, 0x03147, 0x0314d, 0x0315f, 0x03163, 0x03171, 0x0317b, 0x031a3, 0x031a9, 0x031b7, 0x031c5, 0x031c9, 0x031db, 0x031e1, 0x031eb, 0x031ed, 0x031f3, 0x031ff, 0x03209, 0x0320f, 0x0321d, 0x03227, 0x03239, 0x0324b, 0x03253, 0x03259, 0x03265, 0x03281, 0x03293, 0x03299, 0x0329f, 0x032a9, 0x032b7, 0x032bb, 0x032c3, 0x032d7, 0x032db, 0x032e7, 0x03307, 0x03315, 0x0332f, 0x03351, 0x0335d, 0x03375, 0x03397, 0x0339b, 0x033ab, 0x033b9, 0x033c1, 0x033c7, 0x033d5, 0x033e3, 0x033e5, 0x033f7, 0x033fb, 0x03409, 0x0341b, 0x03427, 0x03441, 0x0344d, 0x0345f, 0x03469, 0x03477, 0x0347b, 0x03487, 0x03493, 0x03499, 0x034a5, 0x034bd, 0x034c9, 0x034db, 0x034e7, 0x034f9, 0x0350d, 0x0351f, 0x03525, 0x03531, 0x03537, 0x03545, 0x0354f, 0x0355d, 0x0356d, 0x03573, 0x0357f, 0x0359d, 0x035a1, 0x035b9, 0x035cd, 0x035d5, 0x035d9, 0x035e3, 0x035e9, 0x035ef, 0x03601, 0x0360b, 0x0361f, 0x03625, 0x0362f, 0x0363b, 0x03649, 0x03651, 0x0365b, 0x03673, 0x03675, 0x03691, 0x0369b, 0x0369d, 0x036ad, 0x036cb, 0x036d3, 0x036d5, 0x036e3, 0x036ef, 0x03705, 0x0370f, 0x0371b, 0x03721, 0x0372d, 0x03739, 0x03741, 0x03747, 0x03753, 0x03771, 0x03777, 0x0378b, 0x03795, 0x03799, 0x037a3, 0x037c5, 0x037cf, 0x037d1, 0x037d7, 0x037dd, 0x037e1, 0x037f3, 0x03803, 0x03805, 0x03817, 0x0381d, 0x03827, 0x03833, 0x0384b, 0x03859, 0x03869, 0x03871, 0x038a3, 0x038b1, 0x038bb, 0x038c9, 0x038cf, 0x038e1, 0x038f3, 0x038f9, 0x03901, 0x03907, 0x0390b, 0x03913, 0x03931, 0x0394f, 0x03967, 0x0396d, 0x03983, 0x03985, 0x03997, 0x039a1, 0x039a7, 0x039ad, 0x039cb, 0x039cd, 0x039d3, 0x039ef, 0x039f7, 0x039fd, 0x03a07, 0x03a29, 0x03a2f, 0x03a3d, 0x03a51, 0x03a5d, 0x03a61, 0x03a67, 0x03a73, 0x03a75, 0x03a89, 0x03ab9, 0x03abf, 0x03acd, 0x03ad3, 0x03ad5, 0x03adf, 0x03ae5, 0x03ae9, 0x03afb, 0x03b11, 0x03b2b, 0x03b2d, 0x03b35, 0x03b3f, 0x03b53, 0x03b59, 0x03b63, 0x03b65, 0x03b6f, 0x03b71, 0x03b77, 0x03b8b, 0x03b99, 0x03ba5, 0x03ba9, 0x03bb7, 0x03bbb, 0x03bd1, 0x03be7, 0x03bf3, 0x03bff, 0x03c0d, 0x03c13, 0x03c15, 0x03c1f, 0x03c23, 0x03c25, 0x03c3b, 0x03c4f, 0x03c5d, 0x03c6d, 0x03c83, 0x03c8f, 0x03c9d, 0x03ca7, 0x03cab, 0x03cb9, 0x03cc7, 0x03ce9, 0x03cfb, 0x03cfd, 0x03d03, 0x03d17, 0x03d1b, 0x03d21, 0x03d2d, 0x03d33, 0x03d35, 0x03d41, 0x03d4d, 0x03d65, 0x03d69, 0x03d7d, 0x03d81, 0x03d95, 0x03db1, 0x03db7, 0x03dc3, 0x03dd1, 0x03ddb, 0x03de7, 0x03deb, 0x03df9, 0x03e05, 0x03e09, 0x03e0f, 0x03e1b, 0x03e2b, 0x03e3f, 0x03e41, 0x03e53, 0x03e65, 0x03e69, 0x03e8b, 0x03ea3, 0x03ebd, 0x03ec5, 0x03ed7, 0x03edd, 0x03ee1, 0x03ef9, 0x03f0d, 0x03f19, 0x03f1f, 0x03f25, 0x03f37, 0x03f3d, 0x03f43, 0x03f45, 0x03f49, 0x03f51, 0x03f57, 0x03f61, 0x03f83, 0x03f89, 0x03f91, 0x03fab, 0x03fb5, 0x03fe3, 0x03ff7, 0x03ffd };
const word _irreducible_polynomials_degree_14[1162]  = { 1161, 0x04021, 0x0402b, 0x04033, 0x04039, 0x0403f, 0x04053, 0x0405f, 0x04065, 0x0407b, 0x04087, 0x040a9, 0x040af, 0x040bb, 0x040bd, 0x040cf, 0x040d7, 0x040eb, 0x040f3, 0x040f9, 0x0410b, 0x0410d, 0x04113, 0x0413b, 0x04143, 0x04149, 0x04157, 0x04167, 0x0416d, 0x04191, 0x0419b, 0x0419d, 0x041a7, 0x041ab, 0x041ad, 0x041b5, 0x041d5, 0x041d9, 0x041f1, 0x04201, 0x0420d, 0x04245, 0x04249, 0x04257, 0x04261, 0x04273, 0x04279, 0x0427f, 0x04283, 0x04285, 0x0429d, 0x042a1, 0x042c7, 0x042cb, 0x042cd, 0x042e3, 0x042e9, 0x042ef, 0x04303, 0x04309, 0x0431b, 0x04321, 0x04327, 0x0433f, 0x04369, 0x0437b, 0x0437d, 0x04387, 0x04395, 0x043af, 0x043c9, 0x043dd, 0x043eb, 0x043ed, 0x043ff, 0x0440b, 0x0440d, 0x04443, 0x0445d, 0x04473, 0x04489, 0x044c1, 0x044d3, 0x044d5, 0x044df, 0x044e3, 0x044f1, 0x044fb, 0x04503, 0x04509, 0x0450f, 0x0452b, 0x04539, 0x04559, 0x0456f, 0x04577, 0x0457d, 0x0458d, 0x04599, 0x0459f, 0x045a5, 0x045b7, 0x045c5, 0x045c9, 0x045d1, 0x045d7, 0x045e7, 0x045f3, 0x045ff, 0x0460f, 0x0461d, 0x04627, 0x04635, 0x04647, 0x04659, 0x04663, 0x04671, 0x0467b, 0x0468d, 0x04693, 0x04695, 0x046a3, 0x046a5, 0x046c5, 0x046cf, 0x046db, 0x046eb, 0x046ff, 0x04725, 0x04731, 0x04743, 0x0474f, 0x0477f, 0x04789, 0x04791, 0x0479b, 0x047a7, 0x047b5, 0x047c1, 0x047d3, 0x047d9, 0x047e5, 0x047e9, 0x047ef, 0x047fd, 0x04813, 0x04819, 0x0481f, 0x0483b, 0x04843, 0x04851, 0x0485b, 0x04861, 0x04867, 0x0487f, 0x04883, 0x04891, 0x048bf, 0x048c7, 0x048e3, 0x048f1, 0x048f7, 0x048fd, 0x0491d, 0x04921, 0x0492d, 0x04939, 0x04941, 0x04953, 0x0495f, 0x04965, 0x04969, 0x04977, 0x04987, 0x0498b, 0x049a9, 0x049cf, 0x049d1, 0x049dd, 0x049e1, 0x049e7, 0x04a17, 0x04a35, 0x04a59, 0x04a65, 0x04a7b, 0x04a81, 0x04a87, 0x04a8b, 0x04a93, 0x04aa3, 0x04aaf, 0x04abd, 0x04ac9, 0x04add, 0x04aed, 0x04af9, 0x04b1f, 0x04b37, 0x04b49, 0x04b5b, 0x04b61, 0x04b6b, 0x04b6d, 0x04b73, 0x04b79, 0x04b85, 0x04b8f, 0x04b97, 0x04ba1, 0x04bab, 0x04bb3, 0x04bb9, 0x04bcb, 0x04bcd, 0x04bd5, 0x04bdf, 0x04bf1, 0x04bfb, 0x04c09, 0x04c1b, 0x04c1d, 0x04c27, 0x04c2b, 0x04c3f, 0x04c47, 0x04c4b, 0x04c53, 0x04c55, 0x04c65, 0x04c7d, 0x04c8d, 0x04ca5, 0x04cbd, 0x04cc5, 0x04cd1, 0x04cd7, 0x04ced, 0x04cff, 0x04d15, 0x04d1f, 0x04d29, 0x04d31, 0x04d45, 0x04d4f, 0x04d51, 0x04d75, 0x04d85, 0x04d8f, 0x04dad, 0x04db3, 0x04dc1, 0x04dd3, 0x04df1, 0x04dfb, 0x04e01, 0x04e15, 0x04e23, 0x04e37, 0x04e43, 0x04e49, 0x04e51, 0x04e6b, 0x04e6d, 0x04e83, 0x04e9b, 0x04eb3, 0x04ed3, 0x04ee5, 0x04ee9, 0x04eef, 0x04f1d, 0x04f21, 0x04f4b, 0x04f55, 0x04f5f, 0x04f69, 0x04f6f, 0x04f81, 0x04f8d, 0x04f9f, 0x04fa5, 0x04fa9, 0x04fb7, 0x04fbb, 0x04fd7, 0x04feb, 0x04fed, 0x04ff9, 0x05007, 0x0501f, 0x05025, 0x0505b, 0x0505d, 0x050a1, 0x050cb, 0x050d5, 0x050d9, 0x050e3, 0x050e9, 0x050ef, 0x050fb, 0x0511b, 0x05121, 0x0512d, 0x0513f, 0x05153, 0x05159, 0x0515f, 0x0516f, 0x0517d, 0x0518b, 0x05199, 0x051a5, 0x051b1, 0x051b7, 0x051cf, 0x051d1, 0x051db, 0x051eb, 0x051ed, 0x051f5, 0x051ff, 0x05205, 0x0520f, 0x0522d, 0x05265, 0x05271, 0x0528b, 0x05299, 0x052a3, 0x052b1, 0x052bd, 0x052c3, 0x052c5, 0x052cf, 0x052d1, 0x052e7, 0x052eb, 0x052f9, 0x05301, 0x05319, 0x05325, 0x05329, 0x05337, 0x05349, 0x0535b, 0x05367, 0x05375, 0x0539b, 0x0539d, 0x053a7, 0x053b9, 0x053bf, 0x053c7, 0x053d3, 0x053e5, 0x053f1, 0x053fd, 0x05403, 0x0540f, 0x05417, 0x05435, 0x05439, 0x05447, 0x05459, 0x0545f, 0x0547b, 0x0548d, 0x05493, 0x05495, 0x0549f, 0x054b1, 0x054b7, 0x054bb, 0x054dd, 0x054e1, 0x054eb, 0x05513, 0x05519, 0x0552f, 0x05537, 0x05567, 0x0556b, 0x05575, 0x05579, 0x0557f, 0x05583, 0x05585, 0x05591, 0x05597, 0x055ad, 0x055c1, 0x055d3, 0x055df, 0x055e9, 0x055f7, 0x05607, 0x0560d, 0x05615, 0x05623, 0x05629, 0x0562f, 0x05631, 0x05657, 0x0565d, 0x0569b, 0x056ab, 0x056ad, 0x056b3, 0x056c1, 0x056c7, 0x056cd, 0x056f1, 0x056f7, 0x056fb, 0x0572b, 0x0572d, 0x0573f, 0x0574b, 0x05755, 0x05759, 0x05763, 0x05765, 0x0577d, 0x05793, 0x057af, 0x057bd, 0x057c3, 0x057c5, 0x057d7, 0x057e7, 0x057f3, 0x05803, 0x05811, 0x05821, 0x0582d, 0x05833, 0x05835, 0x0583f, 0x05841, 0x0587d, 0x05887, 0x0588b, 0x05895, 0x05899, 0x058a3, 0x058af, 0x058b1, 0x058d1, 0x058dd, 0x058e7, 0x058eb, 0x058f9, 0x058ff, 0x05913, 0x05923, 0x0593b, 0x0594f, 0x0597f, 0x0598f, 0x05997, 0x0599d, 0x059a1, 0x059a7, 0x059ab, 0x059b5, 0x059cb, 0x059df, 0x059e9, 0x059fd, 0x05a0d, 0x05a13, 0x05a1f, 0x05a25, 0x05a3b, 0x05a3d, 0x05a43, 0x05a45, 0x05a49, 0x05a75, 0x05a8f, 0x05aa7, 0x05ab5, 0x05ac1, 0x05ad3, 0x05ad5, 0x05ad9, 0x05af7, 0x05b0f, 0x05b17, 0x05b1d, 0x05b2b, 0x05b33, 0x05b39, 0x05b41, 0x05b63, 0x05b69, 0x05b77, 0x05b99, 0x05ba3, 0x05ba9, 0x05bc3, 0x05bc5, 0x05be1, 0x05beb, 0x05bf9, 0x05c07, 0x05c0b, 0x05c19, 0x05c31, 0x05c37, 0x05c49, 0x05c5d, 0x05c6d, 0x05c79, 0x05c83, 0x05ca1, 0x05cbf, 0x05cc1, 0x05ccb, 0x05ccd, 0x05ce5, 0x05d05, 0x05d11, 0x05d1d, 0x05d33, 0x05d35, 0x05d47, 0x05d6f, 0x05d87, 0x05d8d, 0x05d95, 0x05d9f, 0x05da3, 0x05da9, 0x05dbb, 0x05dc9, 0x05dcf, 0x05ddb, 0x05de1, 0x05e03, 0x05e0f, 0x05e1b, 0x05e2d, 0x05e47, 0x05e4b, 0x05e63, 0x05e7d, 0x05e81, 0x05e99, 0x05ea5, 0x05ea9, 0x05ed7, 0x05edb, 0x05ee7, 0x05ef5, 0x05f07, 0x05f0d, 0x05f19, 0x05f1f, 0x05f23, 0x05f3d, 0x05f43, 0x05f45, 0x05f51, 0x05f61, 0x05f6b, 0x05f75, 0x05f89, 0x05f97, 0x05fab, 0x05fcd, 0x05fdf, 0x05fe3, 0x05fe5, 0x05ff1, 0x0600b, 0x0600d, 0x06015, 0x0602f, 0x0603d, 0x0604f, 0x06051, 0x06061, 0x06067, 0x0606b, 0x06089, 0x06097, 0x0609d, 0x060a1, 0x060ab, 0x060b3, 0x060b9, 0x060d3, 0x060d5, 0x060f7, 0x06109, 0x06111, 0x0612d, 0x06139, 0x06141, 0x06153, 0x06171, 0x0617d, 0x0619f, 0x061a5, 0x061af, 0x061bb, 0x061db, 0x061ed, 0x061f5, 0x061ff, 0x0622b, 0x06233, 0x06235, 0x06239, 0x06247, 0x0624d, 0x06263, 0x0626f, 0x0627d, 0x0628d, 0x0629f, 0x062a5, 0x062a9, 0x062b1, 0x062b7, 0x062d7, 0x062db, 0x062dd, 0x062eb, 0x062ed, 0x062f3, 0x062ff, 0x06323, 0x06331, 0x06337, 0x0633d, 0x0635b, 0x0636d, 0x06375, 0x06385, 0x06389, 0x06391, 0x063a1, 0x063bf, 0x063ef, 0x063fd, 0x06409, 0x06417, 0x0642d, 0x06441, 0x06447, 0x0644d, 0x06455, 0x0648b, 0x06495, 0x0649f, 0x064a9, 0x064b1, 0x064e7, 0x064f5, 0x06501, 0x0650b, 0x06519, 0x06537, 0x0653b, 0x06543, 0x06545, 0x06549, 0x0655b, 0x06583, 0x06591, 0x065a7, 0x065ad, 0x065b9, 0x065bf, 0x065d3, 0x065d5, 0x065d9, 0x065e5, 0x065ef, 0x065f1, 0x065f7, 0x06601, 0x06607, 0x0660d, 0x06623, 0x0663b, 0x0664f, 0x0665d, 0x0666b, 0x0666d, 0x06673, 0x06683, 0x0668f, 0x06697, 0x066a7, 0x066ab, 0x066b5, 0x066b9, 0x066d9, 0x066df, 0x066e9, 0x06711, 0x0671b, 0x06721, 0x06733, 0x0674b, 0x0675f, 0x06769, 0x06777, 0x06781, 0x067a3, 0x067af, 0x067b7, 0x067d1, 0x067e7, 0x067eb, 0x067f5, 0x06803, 0x06811, 0x0681b, 0x0681d, 0x0682b, 0x06841, 0x06853, 0x0686f, 0x06877, 0x0688b, 0x0688d, 0x06893, 0x0689f, 0x068a5, 0x068a9, 0x068c5, 0x068c9, 0x068ff, 0x06907, 0x06919, 0x0692f, 0x0693d, 0x06957, 0x0696b, 0x06973, 0x06975, 0x06979, 0x06985, 0x0698f, 0x0699b, 0x0699d, 0x069a1, 0x069ab, 0x069c7, 0x069cd, 0x069e9, 0x06a01, 0x06a07, 0x06a0b, 0x06a19, 0x06a23, 0x06a2f, 0x06a51, 0x06a5b, 0x06a67, 0x06a6b, 0x06a6d, 0x06a75, 0x06a83, 0x06a97, 0x06ab3, 0x06ab5, 0x06abf, 0x06ac1, 0x06acb, 0x06acd, 0x06ae9, 0x06aef, 0x06afb, 0x06afd, 0x06b03, 0x06b17, 0x06b2b, 0x06b33, 0x06b39, 0x06b47, 0x06b4b, 0x06b55, 0x06b69, 0x06b7d, 0x06b81, 0x06b8d, 0x06b95, 0x06b9f, 0x06ba3, 0x06ba5, 0x06bb1, 0x06bb7, 0x06bc5, 0x06bd7, 0x06be1, 0x06bed, 0x06bf3, 0x06bf9, 0x06c0b, 0x06c19, 0x06c2f, 0x06c3d, 0x06c45, 0x06c57, 0x06c61, 0x06c73, 0x06cb5, 0x06cb9, 0x06cbf, 0x06cc1, 0x06cc7, 0x06ccb, 0x06ce5, 0x06cf1, 0x06d05, 0x06d09, 0x06d0f, 0x06d17, 0x06d2b, 0x06d53, 0x06d63, 0x06d65, 0x06d69, 0x06d7b, 0x06da3, 0x06db1, 0x06dbb, 0x06dbd, 0x06dc3, 0x06dc5, 0x06ddd, 0x06e09, 0x06e27, 0x06e2d, 0x06e33, 0x06e41, 0x06e4d, 0x06e53, 0x06e6f, 0x06e7b, 0x06e81, 0x06e87, 0x06e95, 0x06eaf, 0x06eb7, 0x06ec3, 0x06edb, 0x06edd, 0x06ef9, 0x06f01, 0x06f15, 0x06f29, 0x06f31, 0x06f3b, 0x06f4f, 0x06f57, 0x06f5b, 0x06f61, 0x06f85, 0x06f91, 0x06f97, 0x06fab, 0x06fb5, 0x06fc7, 0x06fd9, 0x06fdf, 0x06fe9, 0x07005, 0x0701d, 0x0702b, 0x07033, 0x07035, 0x0704b, 0x0707d, 0x07081, 0x0708d, 0x070a9, 0x070b7, 0x070bb, 0x070c9, 0x070cf, 0x070dd, 0x070e1, 0x07113, 0x07115, 0x07119, 0x07123, 0x07131, 0x07137, 0x0713d, 0x0715d, 0x0716b, 0x07189, 0x07197, 0x0719b, 0x071a1, 0x071b5, 0x071bf, 0x071cb, 0x071df, 0x071e5, 0x071f7, 0x071fb, 0x07219, 0x0721f, 0x0722f, 0x07231, 0x0723b, 0x0724f, 0x07261, 0x07267, 0x0728f, 0x072ad, 0x072b3, 0x072c1, 0x072cd, 0x072d3, 0x072df, 0x072e5, 0x072f1, 0x072f7, 0x07303, 0x07309, 0x07327, 0x0732b, 0x07341, 0x07355, 0x07365, 0x0736f, 0x0738d, 0x07393, 0x073a5, 0x073bd, 0x073c9, 0x073d1, 0x073e7, 0x073f3, 0x073f5, 0x073ff, 0x07413, 0x07415, 0x07429, 0x0744f, 0x07457, 0x0745b, 0x0746b, 0x0746d, 0x07483, 0x074ab, 0x074b3, 0x074c7, 0x074cd, 0x074d5, 0x074df, 0x074e9, 0x074fb, 0x074fd, 0x07517, 0x0751b, 0x07521, 0x07535, 0x07541, 0x0754b, 0x07577, 0x0757b, 0x07581, 0x07599, 0x075a3, 0x075bd, 0x075cf, 0x075d1, 0x075eb, 0x075f5, 0x075f9, 0x075ff, 0x0761d, 0x07639, 0x07647, 0x07653, 0x07655, 0x0765f, 0x07663, 0x07665, 0x07669, 0x07687, 0x07695, 0x0769f, 0x076a3, 0x076af, 0x076bb, 0x076c5, 0x076d1, 0x076eb, 0x076f3, 0x076f9, 0x0770b, 0x0772f, 0x07749, 0x07751, 0x07757, 0x0776d, 0x07773, 0x07783, 0x07789, 0x077a7, 0x077ad, 0x077b5, 0x077c7, 0x077d3, 0x077d5, 0x077ef, 0x07815, 0x07825, 0x07831, 0x0783d, 0x0784f, 0x07851, 0x0785b, 0x0786d, 0x078a7, 0x078ad, 0x078b3, 0x078bf, 0x078cb, 0x078cd, 0x078d9, 0x078df, 0x078e9, 0x078ef, 0x07903, 0x0790f, 0x07917, 0x07927, 0x07933, 0x0794d, 0x07959, 0x0796f, 0x07971, 0x0797b, 0x07981, 0x07987, 0x079a5, 0x079b1, 0x079c5, 0x079c9, 0x079d7, 0x079dd, 0x07a03, 0x07a1b, 0x07a27, 0x07a2b, 0x07a35, 0x07a3f, 0x07a4b, 0x07a55, 0x07a77, 0x07a81, 0x07a8d, 0x07aa9, 0x07ab7, 0x07abb, 0x07ac3, 0x07ae1, 0x07af3, 0x07af5, 0x07b0b, 0x07b23, 0x07b3b, 0x07b45, 0x07b4f, 0x07b51, 0x07b5d, 0x07b67, 0x07b79, 0x07b85, 0x07b8f, 0x07ba1, 0x07bab, 0x07bb9, 0x07bd3, 0x07be3, 0x07bf1, 0x07bf7, 0x07c05, 0x07c09, 0x07c27, 0x07c2d, 0x07c59, 0x07c69, 0x07c7d, 0x07c8b, 0x07c93, 0x07c95, 0x07ca3, 0x07cb7, 0x07cc3, 0x07cd1, 0x07cdd, 0x07ceb, 0x07cf9, 0x07cff, 0x07d01, 0x07d15, 0x07d37, 0x07d45, 0x07d49, 0x07d73, 0x07d79, 0x07d8f, 0x07d91, 0x07d97, 0x07da7, 0x07db3, 0x07dc7, 0x07dcd, 0x07dd5, 0x07de9, 0x07dfb, 0x07dfd, 0x07e01, 0x07e0d, 0x07e19, 0x07e2f, 0x07e45, 0x07e61, 0x07e75, 0x07e7f, 0x07e89, 0x07e8f, 0x07e9b, 0x07e9d, 0x07eab, 0x07ebf, 0x07ec7, 0x07ed3, 0x07ee3, 0x07ee5, 0x07f09, 0x07f21, 0x07f3f, 0x07f4d, 0x07f55, 0x07f71, 0x07f8b, 0x07f8d, 0x07f99, 0x07f9f, 0x07fa3, 0x07fb1, 0x07fc3, 0x07fc5, 0x07fd1, 0x07fd7, 0x07fe1, 0x07fe7 };
const word _irreducible_polynomials_degree_15[2183]  = { 2182, 0x08003, 0x08011, 0x08017, 0x0802d, 0x08035, 0x0805f, 0x0806f, 0x08077, 0x08081, 0x08087, 0x08093, 0x080a5, 0x080b1, 0x080bd, 0x080c3, 0x080cf, 0x080dd, 0x080e7, 0x080f5, 0x080ff, 0x08101, 0x08115, 0x08125, 0x08157, 0x0815d, 0x08161, 0x0816d, 0x08179, 0x08185, 0x081a1, 0x081a7, 0x081b9, 0x081cb, 0x081cd, 0x081d5, 0x081df, 0x081fd, 0x08213, 0x0823b, 0x08245, 0x0827f, 0x08289, 0x0828f, 0x0829b, 0x082cb, 0x082d9, 0x082e3, 0x082f1, 0x08317, 0x0831b, 0x0832b, 0x08333, 0x08347, 0x0834d, 0x0835f, 0x08363, 0x08369, 0x08371, 0x0838b, 0x08395, 0x08399, 0x083af, 0x083bb, 0x083bd, 0x083c5, 0x083d1, 0x083f5, 0x08419, 0x08423, 0x0842f, 0x08431, 0x08437, 0x08449, 0x08467, 0x0846d, 0x08479, 0x08483, 0x08485, 0x08497, 0x084a1, 0x084b5, 0x084df, 0x084ef, 0x084f7, 0x084fb, 0x084fd, 0x0851d, 0x08521, 0x08527, 0x08533, 0x08547, 0x0854b, 0x0855f, 0x08571, 0x0857b, 0x08581, 0x0858d, 0x085a3, 0x085b1, 0x085b7, 0x085c5, 0x085c9, 0x085db, 0x085ed, 0x085f3, 0x08609, 0x08611, 0x0861d, 0x0862b, 0x08655, 0x08659, 0x08665, 0x0867d, 0x08681, 0x08693, 0x086a9, 0x086af, 0x086b7, 0x086c3, 0x086db, 0x086f3, 0x086ff, 0x0870b, 0x0870d, 0x08715, 0x0871f, 0x08729, 0x08731, 0x0876b, 0x08785, 0x08789, 0x0878f, 0x0879d, 0x087ab, 0x087ad, 0x087b3, 0x087bf, 0x087df, 0x087e9, 0x087f1, 0x087f7, 0x08801, 0x08807, 0x08825, 0x08837, 0x0883b, 0x08849, 0x0885d, 0x08861, 0x08873, 0x08879, 0x08883, 0x0888f, 0x08891, 0x088b3, 0x088b5, 0x088c7, 0x088d9, 0x08903, 0x08905, 0x08909, 0x08911, 0x08917, 0x08927, 0x08965, 0x0897d, 0x08995, 0x08999, 0x089af, 0x089cf, 0x089d1, 0x089db, 0x089dd, 0x089ed, 0x089ff, 0x08a17, 0x08a39, 0x08a55, 0x08a63, 0x08a65, 0x08a71, 0x08a7b, 0x08a87, 0x08a8b, 0x08a9f, 0x08aaf, 0x08ab1, 0x08add, 0x08b0b, 0x08b0d, 0x08b3b, 0x08b3d, 0x08b43, 0x08b6d, 0x08b73, 0x08b91, 0x08b9d, 0x08bc1, 0x08bcb, 0x08bd5, 0x08bd9, 0x08be5, 0x08bf1, 0x08c21, 0x08c27, 0x08c47, 0x08c55, 0x08c59, 0x08c63, 0x08c69, 0x08c87, 0x08c93, 0x08ca3, 0x08ca9, 0x08cbb, 0x08cc9, 0x08ce1, 0x08ceb, 0x08d01, 0x08d07, 0x08d0b, 0x08d25, 0x08d2f, 0x08d43, 0x08d51, 0x08d5b, 0x08d6d, 0x08d7f, 0x08d85, 0x08d89, 0x08d9b, 0x08da1, 0x08dbf, 0x08dcb, 0x08dcd, 0x08de5, 0x08de9, 0x08def, 0x08df7, 0x08e0b, 0x08e1f, 0x08e3b, 0x08e51, 0x08e57, 0x08e73, 0x08e75, 0x08e8f, 0x08e97, 0x08e9b, 0x08ea1, 0x08ebf, 0x08ec1, 0x08ed3, 0x08edf, 0x08ee5, 0x08efb, 0x08f0f, 0x08f27, 0x08f2b, 0x08f39, 0x08f41, 0x08f53, 0x08f55, 0x08f65, 0x08f77, 0x08f7d, 0x08fa5, 0x08fb7, 0x08fbd, 0x08fc9, 0x08fd1, 0x08fe1, 0x08fe7, 0x0900b, 0x09019, 0x0901f, 0x09025, 0x09037, 0x0903d, 0x09043, 0x09057, 0x09061, 0x0906d, 0x09075, 0x09089, 0x09091, 0x0909b, 0x0909d, 0x090ad, 0x090e3, 0x09105, 0x09109, 0x0910f, 0x09117, 0x0911b, 0x0911d, 0x09141, 0x0914d, 0x09177, 0x0917b, 0x09199, 0x091a3, 0x091a9, 0x091b1, 0x091bd, 0x091cf, 0x091e1, 0x09203, 0x09211, 0x0921b, 0x09221, 0x09227, 0x09233, 0x0923f, 0x0924d, 0x09253, 0x0925f, 0x0927b, 0x09287, 0x09295, 0x092a3, 0x092b7, 0x092c9, 0x092dd, 0x092f5, 0x092f9, 0x09313, 0x09331, 0x09337, 0x09345, 0x09349, 0x09357, 0x09373, 0x0937f, 0x09383, 0x0939d, 0x093a1, 0x093b5, 0x093cb, 0x093df, 0x093ef, 0x093f1, 0x093fb, 0x0940f, 0x0942b, 0x0942d, 0x09435, 0x0944b, 0x09453, 0x09477, 0x0947d, 0x0948b, 0x09493, 0x09499, 0x094a5, 0x094a9, 0x094bb, 0x094c5, 0x094d7, 0x094db, 0x094e1, 0x094e7, 0x094f5, 0x094ff, 0x09507, 0x09515, 0x09529, 0x09531, 0x0953d, 0x09545, 0x09561, 0x0956b, 0x09589, 0x0959b, 0x095ad, 0x095cd, 0x095df, 0x095e3, 0x095e9, 0x09613, 0x09615, 0x0962f, 0x09631, 0x09637, 0x0963d, 0x09643, 0x0965d, 0x09667, 0x09679, 0x09697, 0x096ad, 0x096c1, 0x096cb, 0x096d3, 0x096df, 0x096ef, 0x096fb, 0x09705, 0x0972b, 0x09735, 0x09739, 0x0974d, 0x09763, 0x09795, 0x0979f, 0x097a3, 0x097a5, 0x097a9, 0x097b1, 0x097db, 0x097dd, 0x097e1, 0x097f3, 0x097ff, 0x09809, 0x09821, 0x0982d, 0x09847, 0x09895, 0x09899, 0x098af, 0x098bb, 0x098c3, 0x098c5, 0x098d7, 0x098dd, 0x098e7, 0x098f3, 0x098f9, 0x0990b, 0x09919, 0x09925, 0x09929, 0x0992f, 0x09937, 0x09943, 0x09945, 0x0995b, 0x0996d, 0x0997f, 0x09989, 0x09991, 0x0999b, 0x099b9, 0x099bf, 0x099c1, 0x099cb, 0x099d5, 0x099e3, 0x099e5, 0x09a25, 0x09a31, 0x09a4f, 0x09a57, 0x09a5b, 0x09a5d, 0x09a61, 0x09a6d, 0x09a79, 0x09a7f, 0x09a85, 0x09a97, 0x09a9d, 0x09aab, 0x09ab9, 0x09ac7, 0x09acb, 0x09ae3, 0x09afb, 0x09b11, 0x09b1b, 0x09b2d, 0x09b33, 0x09b41, 0x09b53, 0x09b5f, 0x09b6f, 0x09b77, 0x09b7d, 0x09b87, 0x09b8b, 0x09b93, 0x09bb7, 0x09bbd, 0x09bd1, 0x09beb, 0x09bf9, 0x09bff, 0x09c15, 0x09c1f, 0x09c2f, 0x09c3b, 0x09c3d, 0x09c51, 0x09c5d, 0x09c67, 0x09c73, 0x09c75, 0x09c79, 0x09c85, 0x09cbf, 0x09cc7, 0x09cd5, 0x09ce5, 0x09ce9, 0x09cf1, 0x09cf7, 0x09d03, 0x09d55, 0x09d59, 0x09d63, 0x09d77, 0x09d81, 0x09d8b, 0x09d99, 0x09d9f, 0x09da5, 0x09dbd, 0x09e03, 0x09e11, 0x09e17, 0x09e21, 0x09e27, 0x09e39, 0x09e47, 0x09e55, 0x09e59, 0x09e5f, 0x09e63, 0x09e69, 0x09e81, 0x09e9f, 0x09ea3, 0x09eed, 0x09ef3, 0x09f0b, 0x09f13, 0x09f19, 0x09f23, 0x09f2f, 0x09f49, 0x09f4f, 0x09f67, 0x09f9b, 0x09fad, 0x09fb3, 0x09fb5, 0x09fcd, 0x09fd9, 0x09ff7, 0x0a00b, 0x0a015, 0x0a023, 0x0a025, 0x0a02f, 0x0a043, 0x0a05b, 0x0a05d, 0x0a067, 0x0a06b, 0x0a06d, 0x0a07f, 0x0a089, 0x0a091, 0x0a0a7, 0x0a0b5, 0x0a0cd, 0x0a0df, 0x0a0e9, 0x0a0fb, 0x0a103, 0x0a10f, 0x0a11d, 0x0a121, 0x0a127, 0x0a133, 0x0a135, 0x0a139, 0x0a147, 0x0a159, 0x0a16f, 0x0a17b, 0x0a181, 0x0a19f, 0x0a1b1, 0x0a1c5, 0x0a1cf, 0x0a1e1, 0x0a20f, 0x0a21d, 0x0a22d, 0x0a235, 0x0a241, 0x0a247, 0x0a255, 0x0a263, 0x0a26f, 0x0a299, 0x0a2a9, 0x0a2bb, 0x0a2bd, 0x0a2c5, 0x0a2c9, 0x0a2d7, 0x0a2db, 0x0a2ed, 0x0a307, 0x0a30b, 0x0a30d, 0x0a319, 0x0a31f, 0x0a323, 0x0a329, 0x0a345, 0x0a36b, 0x0a373, 0x0a385, 0x0a38f, 0x0a397, 0x0a3a1, 0x0a3b3, 0x0a3c1, 0x0a3cd, 0x0a3d5, 0x0a403, 0x0a405, 0x0a409, 0x0a411, 0x0a453, 0x0a459, 0x0a465, 0x0a481, 0x0a487, 0x0a48b, 0x0a495, 0x0a499, 0x0a4a5, 0x0a4af, 0x0a4b1, 0x0a4bd, 0x0a4c3, 0x0a4db, 0x0a4eb, 0x0a4ed, 0x0a501, 0x0a523, 0x0a525, 0x0a529, 0x0a54f, 0x0a573, 0x0a583, 0x0a5a7, 0x0a5b9, 0x0a5c7, 0x0a5cd, 0x0a5e9, 0x0a5f1, 0x0a5fb, 0x0a613, 0x0a625, 0x0a62f, 0x0a637, 0x0a63d, 0x0a651, 0x0a661, 0x0a673, 0x0a691, 0x0a69b, 0x0a69d, 0x0a6c7, 0x0a6f1, 0x0a6fd, 0x0a71b, 0x0a72b, 0x0a739, 0x0a75f, 0x0a771, 0x0a78d, 0x0a799, 0x0a7b1, 0x0a7bb, 0x0a7bd, 0x0a7cf, 0x0a7d1, 0x0a7dd, 0x0a7e7, 0x0a805, 0x0a82b, 0x0a833, 0x0a839, 0x0a84b, 0x0a84d, 0x0a855, 0x0a869, 0x0a86f, 0x0a87d, 0x0a881, 0x0a893, 0x0a89f, 0x0a8a3, 0x0a8a9, 0x0a8b7, 0x0a8cf, 0x0a8dd, 0x0a8e1, 0x0a8eb, 0x0a90b, 0x0a913, 0x0a919, 0x0a925, 0x0a949, 0x0a975, 0x0a991, 0x0a9a7, 0x0a9ad, 0x0a9bf, 0x0a9c1, 0x0a9e9, 0x0a9ef, 0x0a9f7, 0x0aa0d, 0x0aa15, 0x0aa2f, 0x0aa31, 0x0aa43, 0x0aa45, 0x0aa4f, 0x0aa51, 0x0aa61, 0x0aa79, 0x0aa9d, 0x0aaab, 0x0aaad, 0x0aab3, 0x0aab9, 0x0aad3, 0x0aadf, 0x0aaf1, 0x0ab03, 0x0ab0f, 0x0ab1d, 0x0ab35, 0x0ab39, 0x0ab47, 0x0ab63, 0x0ab7d, 0x0ab81, 0x0ab8b, 0x0ab99, 0x0abbb, 0x0abc5, 0x0abd1, 0x0abd7, 0x0abdb, 0x0abe7, 0x0abf3, 0x0ac01, 0x0ac07, 0x0ac0d, 0x0ac13, 0x0ac29, 0x0ac37, 0x0ac3b, 0x0ac45, 0x0ac5b, 0x0ac6d, 0x0ac73, 0x0ac7f, 0x0ac85, 0x0ac8f, 0x0ac97, 0x0acc7, 0x0acd5, 0x0acdf, 0x0ace9, 0x0acfd, 0x0ad05, 0x0ad11, 0x0ad1b, 0x0ad21, 0x0ad4b, 0x0ad4d, 0x0ad6f, 0x0ada3, 0x0adaf, 0x0adc9, 0x0addd, 0x0ade7, 0x0adeb, 0x0adf9, 0x0ae09, 0x0ae1d, 0x0ae2d, 0x0ae39, 0x0ae3f, 0x0ae4d, 0x0ae5f, 0x0ae63, 0x0ae71, 0x0ae77, 0x0ae7b, 0x0ae8d, 0x0ae95, 0x0aea3, 0x0aebb, 0x0aed7, 0x0aedd, 0x0aee7, 0x0aef5, 0x0af01, 0x0af13, 0x0af1f, 0x0af29, 0x0af2f, 0x0af3b, 0x0af43, 0x0af49, 0x0af67, 0x0af75, 0x0af8f, 0x0af9d, 0x0afab, 0x0afc1, 0x0afd3, 0x0afe3, 0x0aff7, 0x0affd, 0x0b00f, 0x0b01b, 0x0b02d, 0x0b035, 0x0b03f, 0x0b055, 0x0b077, 0x0b08d, 0x0b093, 0x0b0af, 0x0b0c3, 0x0b0c5, 0x0b0d1, 0x0b0e1, 0x0b0f3, 0x0b107, 0x0b10b, 0x0b10d, 0x0b13d, 0x0b157, 0x0b15d, 0x0b175, 0x0b18f, 0x0b197, 0x0b19d, 0x0b1a1, 0x0b1c7, 0x0b1df, 0x0b1e5, 0x0b20b, 0x0b213, 0x0b215, 0x0b243, 0x0b249, 0x0b24f, 0x0b275, 0x0b27f, 0x0b289, 0x0b29b, 0x0b2b5, 0x0b2c1, 0x0b2df, 0x0b2e9, 0x0b2ef, 0x0b2f7, 0x0b305, 0x0b317, 0x0b327, 0x0b363, 0x0b37b, 0x0b381, 0x0b393, 0x0b3a5, 0x0b3a9, 0x0b3b1, 0x0b3b7, 0x0b3c5, 0x0b3d7, 0x0b3db, 0x0b3dd, 0x0b3f9, 0x0b401, 0x0b40d, 0x0b419, 0x0b41f, 0x0b429, 0x0b443, 0x0b445, 0x0b457, 0x0b46d, 0x0b475, 0x0b47f, 0x0b4a7, 0x0b4bf, 0x0b4d9, 0x0b4ef, 0x0b4f7, 0x0b509, 0x0b51d, 0x0b533, 0x0b53f, 0x0b547, 0x0b553, 0x0b555, 0x0b569, 0x0b577, 0x0b57b, 0x0b595, 0x0b5a9, 0x0b5bb, 0x0b5c3, 0x0b5d7, 0x0b5e1, 0x0b5e7, 0x0b5eb, 0x0b5f9, 0x0b605, 0x0b609, 0x0b617, 0x0b61b, 0x0b621, 0x0b627, 0x0b62d, 0x0b633, 0x0b635, 0x0b64b, 0x0b659, 0x0b67b, 0x0b681, 0x0b68b, 0x0b699, 0x0b6af, 0x0b6b1, 0x0b6d1, 0x0b6e7, 0x0b6eb, 0x0b707, 0x0b71f, 0x0b725, 0x0b745, 0x0b757, 0x0b75b, 0x0b76b, 0x0b779, 0x0b783, 0x0b791, 0x0b797, 0x0b79d, 0x0b7a1, 0x0b7b3, 0x0b7cb, 0x0b7fd, 0x0b81f, 0x0b845, 0x0b84f, 0x0b861, 0x0b875, 0x0b885, 0x0b889, 0x0b89d, 0x0b8a1, 0x0b8ad, 0x0b8bf, 0x0b8d3, 0x0b8d5, 0x0b8fb, 0x0b903, 0x0b909, 0x0b91d, 0x0b927, 0x0b92b, 0x0b947, 0x0b955, 0x0b959, 0x0b965, 0x0b977, 0x0b98d, 0x0b9af, 0x0b9bb, 0x0b9c3, 0x0b9c9, 0x0b9d1, 0x0b9e1, 0x0b9ed, 0x0b9f3, 0x0b9f5, 0x0ba05, 0x0ba0f, 0x0ba11, 0x0ba2b, 0x0ba39, 0x0ba59, 0x0ba69, 0x0ba6f, 0x0ba81, 0x0ba87, 0x0ba8d, 0x0ba93, 0x0bab7, 0x0bad7, 0x0baf3, 0x0baff, 0x0bb01, 0x0bb07, 0x0bb15, 0x0bb19, 0x0bb23, 0x0bb3d, 0x0bb43, 0x0bb49, 0x0bb4f, 0x0bb51, 0x0bb5b, 0x0bb67, 0x0bb75, 0x0bb91, 0x0bbb5, 0x0bbbf, 0x0bbcb, 0x0bbcd, 0x0bbe5, 0x0bbe9, 0x0bc09, 0x0bc33, 0x0bc39, 0x0bc63, 0x0bc65, 0x0bc69, 0x0bc7b, 0x0bc8b, 0x0bc8d, 0x0bc9f, 0x0bca9, 0x0bcc3, 0x0bcd1, 0x0bcdd, 0x0bd01, 0x0bd13, 0x0bd25, 0x0bd2f, 0x0bd37, 0x0bd45, 0x0bd67, 0x0bd7f, 0x0bd83, 0x0bd89, 0x0bdb9, 0x0bdc1, 0x0bdd3, 0x0bdd9, 0x0bde5, 0x0bdf1, 0x0be07, 0x0be15, 0x0be23, 0x0be29, 0x0be37, 0x0be57, 0x0be5b, 0x0be61, 0x0be6b, 0x0be7f, 0x0be83, 0x0be91, 0x0beab, 0x0bed5, 0x0bed9, 0x0bef1, 0x0bf17, 0x0bf21, 0x0bf27, 0x0bf33, 0x0bf35, 0x0bf53, 0x0bf65, 0x0bf81, 0x0bf8b, 0x0bfa3, 0x0bfb7, 0x0bfc3, 0x0bfdb, 0x0bfed, 0x0bff5, 0x0bfff, 0x0c001, 0x0c007, 0x0c013, 0x0c01f, 0x0c025, 0x0c049, 0x0c073, 0x0c079, 0x0c085, 0x0c091, 0x0c097, 0x0c09d, 0x0c0a7, 0x0c0b9, 0x0c0cb, 0x0c0d5, 0x0c0e3, 0x0c0fb, 0x0c111, 0x0c121, 0x0c133, 0x0c13f, 0x0c14b, 0x0c15f, 0x0c16f, 0x0c177, 0x0c17b, 0x0c17d, 0x0c193, 0x0c1a5, 0x0c1bd, 0x0c1c3, 0x0c1c9, 0x0c1db, 0x0c1e7, 0x0c1ed, 0x0c205, 0x0c209, 0x0c217, 0x0c22d, 0x0c24d, 0x0c253, 0x0c255, 0x0c269, 0x0c299, 0x0c2b1, 0x0c2cf, 0x0c2d1, 0x0c2dd, 0x0c2f5, 0x0c301, 0x0c30d, 0x0c319, 0x0c325, 0x0c32f, 0x0c33d, 0x0c357, 0x0c35b, 0x0c361, 0x0c373, 0x0c383, 0x0c39b, 0x0c39d, 0x0c3ab, 0x0c3ad, 0x0c3bf, 0x0c3c7, 0x0c3e3, 0x0c3ef, 0x0c3fd, 0x0c405, 0x0c421, 0x0c433, 0x0c447, 0x0c44b, 0x0c45f, 0x0c46f, 0x0c47d, 0x0c493, 0x0c49f, 0x0c4a5, 0x0c4b7, 0x0c4c5, 0x0c4d7, 0x0c4dd, 0x0c4eb, 0x0c4f9, 0x0c4ff, 0x0c50b, 0x0c515, 0x0c52f, 0x0c531, 0x0c549, 0x0c55b, 0x0c575, 0x0c579, 0x0c589, 0x0c59b, 0x0c5a1, 0x0c5b3, 0x0c5b5, 0x0c5bf, 0x0c5cb, 0x0c5df, 0x0c5e9, 0x0c5ef, 0x0c5fd, 0x0c62f, 0x0c631, 0x0c63d, 0x0c645, 0x0c651, 0x0c66b, 0x0c675, 0x0c679, 0x0c68f, 0x0c697, 0x0c6ab, 0x0c6b3, 0x0c6b9, 0x0c6c1, 0x0c6c7, 0x0c6cd, 0x0c6d5, 0x0c6e3, 0x0c6e9, 0x0c6fb, 0x0c703, 0x0c709, 0x0c741, 0x0c753, 0x0c759, 0x0c763, 0x0c777, 0x0c799, 0x0c79f, 0x0c7a9, 0x0c7c3, 0x0c7d7, 0x0c7db, 0x0c7f5, 0x0c803, 0x0c81b, 0x0c835, 0x0c841, 0x0c84d, 0x0c85f, 0x0c865, 0x0c869, 0x0c895, 0x0c8bd, 0x0c8c9, 0x0c8f5, 0x0c8f9, 0x0c8ff, 0x0c901, 0x0c90d, 0x0c915, 0x0c91f, 0x0c923, 0x0c929, 0x0c931, 0x0c937, 0x0c93b, 0x0c95b, 0x0c95d, 0x0c961, 0x0c97f, 0x0c983, 0x0c98f, 0x0c9ab, 0x0c9c7, 0x0c9cb, 0x0c9cd, 0x0c9d9, 0x0ca25, 0x0ca29, 0x0ca2f, 0x0ca43, 0x0ca49, 0x0ca67, 0x0ca8f, 0x0caad, 0x0cabf, 0x0cad9, 0x0cae3, 0x0caef, 0x0caf1, 0x0cafd, 0x0cb17, 0x0cb1b, 0x0cb1d, 0x0cb2b, 0x0cb3f, 0x0cb55, 0x0cb69, 0x0cb71, 0x0cb7b, 0x0cb87, 0x0cbbd, 0x0cbd7, 0x0cbe7, 0x0cbf5, 0x0cc07, 0x0cc15, 0x0cc23, 0x0cc3d, 0x0cc49, 0x0cc5b, 0x0cc6d, 0x0cc7f, 0x0cc83, 0x0cc85, 0x0cc9b, 0x0cca1, 0x0cca7, 0x0ccad, 0x0ccb3, 0x0ccc1, 0x0ccd9, 0x0ccf7, 0x0ccfb, 0x0ccfd, 0x0cd11, 0x0cd2b, 0x0cd33, 0x0cd55, 0x0cd63, 0x0cd6f, 0x0cd87, 0x0cd9f, 0x0cda3, 0x0cdb7, 0x0cdc5, 0x0cdcf, 0x0cdd7, 0x0cde1, 0x0cdeb, 0x0cded, 0x0cdf9, 0x0cdff, 0x0ce03, 0x0ce11, 0x0ce27, 0x0ce35, 0x0ce39, 0x0ce65, 0x0ce6f, 0x0ce71, 0x0ce9f, 0x0cea5, 0x0cebb, 0x0cec3, 0x0cec5, 0x0cec9, 0x0ced1, 0x0ced7, 0x0cef3, 0x0ceff, 0x0cf0d, 0x0cf19, 0x0cf1f, 0x0cf2f, 0x0cf5d, 0x0cf61, 0x0cf6b, 0x0cf73, 0x0cf79, 0x0cf8f, 0x0cf9b, 0x0cf9d, 0x0cfa1, 0x0cfab, 0x0cfc7, 0x0cfd5, 0x0cfe9, 0x0cff7, 0x0d005, 0x0d009, 0x0d03f, 0x0d047, 0x0d04d, 0x0d071, 0x0d077, 0x0d087, 0x0d08b, 0x0d08d, 0x0d095, 0x0d099, 0x0d0a3, 0x0d0b1, 0x0d0c5, 0x0d0cf, 0x0d0d1, 0x0d0d7, 0x0d0e1, 0x0d0f9, 0x0d10b, 0x0d125, 0x0d129, 0x0d12f, 0x0d13d, 0x0d151, 0x0d15b, 0x0d16d, 0x0d19b, 0x0d1b9, 0x0d1bf, 0x0d1c1, 0x0d1d5, 0x0d1d9, 0x0d1ef, 0x0d1fd, 0x0d207, 0x0d215, 0x0d21f, 0x0d223, 0x0d229, 0x0d237, 0x0d257, 0x0d26b, 0x0d26d, 0x0d283, 0x0d28f, 0x0d2a1, 0x0d2b5, 0x0d2cb, 0x0d303, 0x0d317, 0x0d327, 0x0d341, 0x0d34b, 0x0d359, 0x0d35f, 0x0d369, 0x0d377, 0x0d37b, 0x0d381, 0x0d393, 0x0d399, 0x0d3a3, 0x0d3b1, 0x0d3c9, 0x0d3d1, 0x0d3dd, 0x0d3e7, 0x0d3ed, 0x0d415, 0x0d429, 0x0d43b, 0x0d45d, 0x0d461, 0x0d46b, 0x0d497, 0x0d49d, 0x0d4ab, 0x0d4b3, 0x0d4bf, 0x0d4c1, 0x0d4d3, 0x0d4e5, 0x0d4e9, 0x0d4f1, 0x0d50f, 0x0d51b, 0x0d527, 0x0d52b, 0x0d547, 0x0d555, 0x0d559, 0x0d563, 0x0d57d, 0x0d593, 0x0d5b7, 0x0d5c3, 0x0d5e1, 0x0d5e7, 0x0d5f3, 0x0d5f5, 0x0d605, 0x0d627, 0x0d62b, 0x0d64b, 0x0d65f, 0x0d663, 0x0d66f, 0x0d67d, 0x0d687, 0x0d6a9, 0x0d6b7, 0x0d6c5, 0x0d6d7, 0x0d6e1, 0x0d6ed, 0x0d6f3, 0x0d715, 0x0d723, 0x0d725, 0x0d731, 0x0d767, 0x0d76d, 0x0d77f, 0x0d7ad, 0x0d7b3, 0x0d7b5, 0x0d7bf, 0x0d7d9, 0x0d7f7, 0x0d7fb, 0x0d80d, 0x0d813, 0x0d849, 0x0d857, 0x0d86d, 0x0d889, 0x0d88f, 0x0d89b, 0x0d8a7, 0x0d8ab, 0x0d8b5, 0x0d8c1, 0x0d8c7, 0x0d8d3, 0x0d8d9, 0x0d8e5, 0x0d909, 0x0d91b, 0x0d933, 0x0d941, 0x0d947, 0x0d94d, 0x0d95f, 0x0d965, 0x0d971, 0x0d98b, 0x0d999, 0x0d99f, 0x0d9a3, 0x0d9a9, 0x0d9b1, 0x0d9c3, 0x0d9d7, 0x0d9f3, 0x0d9f9, 0x0da05, 0x0da17, 0x0da27, 0x0da33, 0x0da35, 0x0da3f, 0x0da59, 0x0da7d, 0x0da8b, 0x0da93, 0x0da99, 0x0daa3, 0x0dab1, 0x0dac3, 0x0dadd, 0x0daed, 0x0db07, 0x0db1f, 0x0db25, 0x0db29, 0x0db3b, 0x0db45, 0x0db4f, 0x0db61, 0x0db83, 0x0db91, 0x0dba1, 0x0dbc7, 0x0dbcd, 0x0dbd5, 0x0dbdf, 0x0dbe3, 0x0dbe9, 0x0dbef, 0x0dbfb, 0x0dbfd, 0x0dc11, 0x0dc2b, 0x0dc35, 0x0dc39, 0x0dc41, 0x0dc6f, 0x0dc71, 0x0dc93, 0x0dc9f, 0x0dcaf, 0x0dcd1, 0x0dcd7, 0x0dcdb, 0x0dcf5, 0x0dd19, 0x0dd29, 0x0dd31, 0x0dd45, 0x0dd57, 0x0dd67, 0x0dd73, 0x0dd75, 0x0dd9d, 0x0ddad, 0x0ddc1, 0x0ddd5, 0x0dde5, 0x0ddf7, 0x0ddfb, 0x0de07, 0x0de2f, 0x0de3d, 0x0de49, 0x0de4f, 0x0de51, 0x0de6d, 0x0de75, 0x0de83, 0x0de85, 0x0de89, 0x0dea1, 0x0dead, 0x0decb, 0x0decd, 0x0ded3, 0x0df03, 0x0df05, 0x0df17, 0x0df1d, 0x0df21, 0x0df33, 0x0df59, 0x0df63, 0x0df69, 0x0df6f, 0x0df71, 0x0dfa5, 0x0dfbb, 0x0dfc9, 0x0dfdb, 0x0dfeb, 0x0e003, 0x0e00f, 0x0e011, 0x0e033, 0x0e035, 0x0e047, 0x0e04b, 0x0e05f, 0x0e07b, 0x0e07d, 0x0e08d, 0x0e09f, 0x0e0a9, 0x0e0b1, 0x0e0b7, 0x0e0c5, 0x0e0d7, 0x0e0db, 0x0e0dd, 0x0e0ed, 0x0e0ff, 0x0e101, 0x0e10b, 0x0e125, 0x0e131, 0x0e149, 0x0e14f, 0x0e151, 0x0e15d, 0x0e167, 0x0e16b, 0x0e17f, 0x0e1b3, 0x0e1d3, 0x0e1d9, 0x0e1ef, 0x0e207, 0x0e20b, 0x0e219, 0x0e223, 0x0e231, 0x0e245, 0x0e24f, 0x0e267, 0x0e279, 0x0e285, 0x0e28f, 0x0e29b, 0x0e29d, 0x0e2a1, 0x0e2ab, 0x0e2ad, 0x0e2bf, 0x0e2c1, 0x0e2d5, 0x0e30f, 0x0e311, 0x0e31b, 0x0e335, 0x0e339, 0x0e359, 0x0e363, 0x0e365, 0x0e377, 0x0e38d, 0x0e393, 0x0e39f, 0x0e3a5, 0x0e3af, 0x0e3b7, 0x0e3c3, 0x0e3db, 0x0e3f3, 0x0e3ff, 0x0e41f, 0x0e431, 0x0e449, 0x0e45b, 0x0e46b, 0x0e46d, 0x0e473, 0x0e479, 0x0e485, 0x0e491, 0x0e497, 0x0e49d, 0x0e4a1, 0x0e4ab, 0x0e4cb, 0x0e4cd, 0x0e4df, 0x0e4f1, 0x0e4fd, 0x0e503, 0x0e505, 0x0e517, 0x0e51b, 0x0e52d, 0x0e533, 0x0e581, 0x0e595, 0x0e5a5, 0x0e5af, 0x0e5e7, 0x0e605, 0x0e621, 0x0e639, 0x0e63f, 0x0e647, 0x0e653, 0x0e669, 0x0e687, 0x0e6bb, 0x0e6bd, 0x0e6d7, 0x0e6dd, 0x0e6eb, 0x0e6f5, 0x0e6f9, 0x0e701, 0x0e719, 0x0e729, 0x0e76d, 0x0e775, 0x0e783, 0x0e78f, 0x0e7a7, 0x0e7ab, 0x0e7ad, 0x0e7b5, 0x0e7cb, 0x0e7d3, 0x0e7d5, 0x0e7e5, 0x0e7f1, 0x0e7f7, 0x0e801, 0x0e82f, 0x0e843, 0x0e84f, 0x0e851, 0x0e85b, 0x0e86d, 0x0e879, 0x0e889, 0x0e891, 0x0e8a7, 0x0e8bf, 0x0e8c1, 0x0e8cb, 0x0e8cd, 0x0e8d3, 0x0e8fb, 0x0e8fd, 0x0e903, 0x0e90f, 0x0e921, 0x0e927, 0x0e92b, 0x0e935, 0x0e959, 0x0e95f, 0x0e963, 0x0e969, 0x0e971, 0x0e98d, 0x0e9b7, 0x0e9c5, 0x0e9cf, 0x0e9d7, 0x0e9ed, 0x0ea09, 0x0ea1b, 0x0ea2d, 0x0ea4b, 0x0ea59, 0x0ea71, 0x0ea7d, 0x0ea81, 0x0ea8d, 0x0eabb, 0x0eac3, 0x0eac9, 0x0ead7, 0x0eaed, 0x0eaff, 0x0eb07, 0x0eb0b, 0x0eb19, 0x0eb23, 0x0eb29, 0x0eb3b, 0x0eb45, 0x0eb57, 0x0eb5d, 0x0eb67, 0x0eb6b, 0x0eb73, 0x0eb75, 0x0eb97, 0x0eb9b, 0x0ebad, 0x0ebb3, 0x0ebcd, 0x0ebd3, 0x0ebd5, 0x0ebe3, 0x0ec09, 0x0ec11, 0x0ec21, 0x0ec35, 0x0ec4b, 0x0ec5f, 0x0ec65, 0x0ec69, 0x0ec7d, 0x0ec93, 0x0ec99, 0x0ecaf, 0x0ecb7, 0x0ecbd, 0x0ecc9, 0x0eccf, 0x0ecff, 0x0ed07, 0x0ed15, 0x0ed23, 0x0ed37, 0x0ed49, 0x0ed5d, 0x0ed61, 0x0ed6b, 0x0ed97, 0x0eda1, 0x0edab, 0x0edb3, 0x0edc7, 0x0edcd, 0x0edd9, 0x0eddf, 0x0edef, 0x0edf1, 0x0edfd, 0x0ee01, 0x0ee0b, 0x0ee0d, 0x0ee1f, 0x0ee29, 0x0ee75, 0x0ee83, 0x0ee89, 0x0ee9d, 0x0eead, 0x0eeb9, 0x0eec7, 0x0eecb, 0x0eed9, 0x0eee3, 0x0eef1, 0x0eef7, 0x0ef0f, 0x0ef21, 0x0ef2d, 0x0ef33, 0x0ef39, 0x0ef4d, 0x0ef77, 0x0ef95, 0x0ef9f, 0x0efb1, 0x0efbb, 0x0efe1, 0x0efe7, 0x0efeb, 0x0eff3, 0x0eff5, 0x0eff9, 0x0f007, 0x0f00d, 0x0f01f, 0x0f029, 0x0f02f, 0x0f045, 0x0f05d, 0x0f085, 0x0f089, 0x0f097, 0x0f0ab, 0x0f0bf, 0x0f0c7, 0x0f0d5, 0x0f0df, 0x0f0f1, 0x0f0f7, 0x0f111, 0x0f11b, 0x0f135, 0x0f141, 0x0f147, 0x0f14b, 0x0f153, 0x0f163, 0x0f171, 0x0f18d, 0x0f193, 0x0f1c5, 0x0f1e1, 0x0f1e7, 0x0f1f3, 0x0f1f5, 0x0f1ff, 0x0f217, 0x0f21d, 0x0f247, 0x0f24d, 0x0f255, 0x0f259, 0x0f26f, 0x0f27b, 0x0f287, 0x0f29f, 0x0f2a5, 0x0f2db, 0x0f2dd, 0x0f2f9, 0x0f2ff, 0x0f301, 0x0f30b, 0x0f315, 0x0f32f, 0x0f337, 0x0f343, 0x0f385, 0x0f389, 0x0f391, 0x0f397, 0x0f3b3, 0x0f3df, 0x0f3e5, 0x0f405, 0x0f40f, 0x0f417, 0x0f421, 0x0f439, 0x0f453, 0x0f455, 0x0f463, 0x0f465, 0x0f469, 0x0f47b, 0x0f48b, 0x0f499, 0x0f4a3, 0x0f4b1, 0x0f4bd, 0x0f4c3, 0x0f4cf, 0x0f4f3, 0x0f4f5, 0x0f4f9, 0x0f50d, 0x0f519, 0x0f525, 0x0f537, 0x0f53b, 0x0f551, 0x0f561, 0x0f56d, 0x0f591, 0x0f59d, 0x0f5a7, 0x0f5b5, 0x0f5bf, 0x0f5c1, 0x0f5c7, 0x0f601, 0x0f615, 0x0f61f, 0x0f623, 0x0f63b, 0x0f645, 0x0f64f, 0x0f65d, 0x0f66b, 0x0f673, 0x0f683, 0x0f685, 0x0f6b3, 0x0f6b5, 0x0f6d9, 0x0f6ef, 0x0f6fb, 0x0f721, 0x0f72d, 0x0f73f, 0x0f74d, 0x0f753, 0x0f769, 0x0f76f, 0x0f787, 0x0f78b, 0x0f795, 0x0f7a3, 0x0f7b1, 0x0f7b7, 0x0f7c3, 0x0f7c9, 0x0f7db, 0x0f7ff, 0x0f803, 0x0f809, 0x0f80f, 0x0f81d, 0x0f827, 0x0f82d, 0x0f839, 0x0f83f, 0x0f84b, 0x0f86f, 0x0f871, 0x0f877, 0x0f893, 0x0f8c5, 0x0f8db, 0x0f8e1, 0x0f8ed, 0x0f8f3, 0x0f8f5, 0x0f907, 0x0f915, 0x0f923, 0x0f93b, 0x0f93d, 0x0f94f, 0x0f951, 0x0f973, 0x0f979, 0x0f985, 0x0f99b, 0x0f9b3, 0x0f9b9, 0x0f9c7, 0x0f9e3, 0x0f9e9, 0x0f9f7, 0x0fa01, 0x0fa07, 0x0fa13, 0x0fa23, 0x0fa37, 0x0fa49, 0x0fa6b, 0x0fa75, 0x0fa79, 0x0fa7f, 0x0fa83, 0x0fa97, 0x0fa9b, 0x0faa1, 0x0fac1, 0x0facb, 0x0fad9, 0x0fadf, 0x0fae5, 0x0fb05, 0x0fb0f, 0x0fb21, 0x0fb27, 0x0fb35, 0x0fb4d, 0x0fb55, 0x0fb5f, 0x0fb69, 0x0fb71, 0x0fb81, 0x0fb8d, 0x0fba3, 0x0fba9, 0x0fbb7, 0x0fbc9, 0x0fbcf, 0x0fbdb, 0x0fbe1, 0x0fc0b, 0x0fc0d, 0x0fc1f, 0x0fc49, 0x0fc5b, 0x0fc67, 0x0fc75, 0x0fc83, 0x0fcad, 0x0fcd3, 0x0fcef, 0x0fd0f, 0x0fd17, 0x0fd1d, 0x0fd2b, 0x0fd2d, 0x0fd39, 0x0fd47, 0x0fd53, 0x0fd71, 0x0fd8b, 0x0fd95, 0x0fd99, 0x0fda3, 0x0fdaf, 0x0fdb1, 0x0fdc3, 0x0fddd, 0x0fde1, 0x0fdeb, 0x0fe05, 0x0fe2d, 0x0fe33, 0x0fe35, 0x0fe41, 0x0fe4d, 0x0fe59, 0x0fe5f, 0x0fe7d, 0x0fe87, 0x0fe93, 0x0fe99, 0x0feb1, 0x0febd, 0x0fec9, 0x0feeb, 0x0feff, 0x0ff01, 0x0ff07, 0x0ff13, 0x0ff23, 0x0ff29, 0x0ff37, 0x0ff4f, 0x0ff57, 0x0ff5d, 0x0ff61, 0x0ff73, 0x0ff7f, 0x0ff8f, 0x0ff91, 0x0ffb3, 0x0ffc7, 0x0ffd9, 0x0ffe9, 0x0ffef, 0x0fffd };
const word _irreducible_polynomials_degree_16[4081]  = { 4080, 0x1002b, 0x1002d, 0x10039, 0x1003f, 0x10047, 0x10053, 0x1008d, 0x100bd, 0x100d7, 0x100f5, 0x10129, 0x1012f, 0x1013b, 0x1013d, 0x1014f, 0x1015d, 0x10173, 0x10175, 0x1018f, 0x10197, 0x101a1, 0x101ad, 0x101bf, 0x101c7, 0x101cd, 0x101d5, 0x101e9, 0x101f7, 0x10215, 0x10219, 0x10225, 0x1022f, 0x1025b, 0x1025d, 0x1026b, 0x1026d, 0x10275, 0x10285, 0x10291, 0x102a1, 0x102b3, 0x102cd, 0x102d3, 0x102e5, 0x102f7, 0x102fb, 0x10317, 0x1031d, 0x10335, 0x10339, 0x1034b, 0x1034d, 0x10369, 0x1036f, 0x10371, 0x10387, 0x1038d, 0x1039f, 0x103a3, 0x103bb, 0x103cf, 0x103dd, 0x103ed, 0x103f9, 0x1040b, 0x10425, 0x10429, 0x10457, 0x10461, 0x10467, 0x1046d, 0x10483, 0x10489, 0x10491, 0x104a7, 0x104ad, 0x104b5, 0x104bf, 0x104c1, 0x104fd, 0x10509, 0x10533, 0x10539, 0x10547, 0x10563, 0x10565, 0x10569, 0x10587, 0x105af, 0x105c3, 0x105cf, 0x105d1, 0x105dd, 0x105eb, 0x105f5, 0x105f9, 0x10641, 0x1064b, 0x10653, 0x1065f, 0x1067b, 0x1067d, 0x10687, 0x1068b, 0x106a3, 0x106af, 0x106c3, 0x106c9, 0x106f5, 0x106ff, 0x1076b, 0x1076d, 0x10779, 0x1077f, 0x10783, 0x1078f, 0x10797, 0x107ef, 0x107f1, 0x107fb, 0x10807, 0x1080b, 0x1080d, 0x10815, 0x1083b, 0x10861, 0x10879, 0x10897, 0x1089d, 0x108ab, 0x108bf, 0x108d5, 0x108df, 0x108e3, 0x108e9, 0x108f1, 0x108fb, 0x10939, 0x10959, 0x10977, 0x1097d, 0x1098b, 0x109a5, 0x109af, 0x109c3, 0x109c5, 0x109e1, 0x109e7, 0x109ed, 0x109f3, 0x10a03, 0x10a11, 0x10a35, 0x10a39, 0x10a5f, 0x10a63, 0x10a71, 0x10a7d, 0x10a81, 0x10a9f, 0x10abb, 0x10ac5, 0x10acf, 0x10add, 0x10ae1, 0x10aff, 0x10b01, 0x10b13, 0x10b15, 0x10b51, 0x10b5d, 0x10b91, 0x10b97, 0x10bb3, 0x10bcd, 0x10bd3, 0x10be5, 0x10bef, 0x10bf7, 0x10bfd, 0x10c0f, 0x10c1d, 0x10c21, 0x10c2b, 0x10c41, 0x10c69, 0x10c71, 0x10c7b, 0x10c8d, 0x10c95, 0x10ca3, 0x10caf, 0x10cbd, 0x10cdd, 0x10ceb, 0x10cf3, 0x10d0d, 0x10d19, 0x10d1f, 0x10d43, 0x10d49, 0x10d83, 0x10d8f, 0x10d91, 0x10d9d, 0x10dad, 0x10db5, 0x10dd3, 0x10de3, 0x10de5, 0x10e23, 0x10e45, 0x10e73, 0x10e7f, 0x10e85, 0x10e91, 0x10e97, 0x10e9d, 0x10ea1, 0x10ea7, 0x10ec7, 0x10ecb, 0x10efd, 0x10f03, 0x10f05, 0x10f09, 0x10f1b, 0x10f21, 0x10f69, 0x10f77, 0x10f87, 0x10f8b, 0x10f95, 0x10f99, 0x10fb1, 0x10fbd, 0x10fcf, 0x10fdb, 0x10fe7, 0x10feb, 0x1100b, 0x11025, 0x1103d, 0x11043, 0x1104f, 0x1105d, 0x1107f, 0x11083, 0x11085, 0x11097, 0x1109b, 0x110a1, 0x110a7, 0x110b5, 0x1111d, 0x11127, 0x11139, 0x11147, 0x1114d, 0x1115f, 0x1116f, 0x111c9, 0x111db, 0x111dd, 0x111eb, 0x111ed, 0x111f3, 0x111f9, 0x11241, 0x11247, 0x11253, 0x11255, 0x11263, 0x11277, 0x11281, 0x1128b, 0x11293, 0x11299, 0x112a5, 0x112af, 0x112bd, 0x112c5, 0x112d7, 0x112db, 0x112e1, 0x112ed, 0x112f3, 0x112ff, 0x1133b, 0x1133d, 0x11351, 0x11357, 0x11361, 0x1136b, 0x11375, 0x11379, 0x1139d, 0x113a1, 0x113df, 0x113f7, 0x11409, 0x1141b, 0x11433, 0x11435, 0x1144b, 0x1144d, 0x11459, 0x1145f, 0x11487, 0x11493, 0x114af, 0x114b1, 0x114c9, 0x114d1, 0x114ed, 0x114f9, 0x11507, 0x11513, 0x1158f, 0x11591, 0x115a1, 0x115a7, 0x115ab, 0x115d5, 0x115d9, 0x115df, 0x115e3, 0x115fb, 0x1161f, 0x1162f, 0x11637, 0x1163d, 0x11643, 0x11645, 0x11649, 0x11651, 0x11675, 0x11683, 0x1169b, 0x116ab, 0x116d5, 0x116d9, 0x116f7, 0x116fd, 0x1170f, 0x1172d, 0x11733, 0x1173f, 0x11741, 0x11747, 0x1174d, 0x11753, 0x11765, 0x11795, 0x11799, 0x117a3, 0x117a9, 0x117d7, 0x117eb, 0x117f5, 0x1182b, 0x11833, 0x11855, 0x11863, 0x1186f, 0x11887, 0x1188d, 0x11893, 0x118a9, 0x118b1, 0x118b7, 0x118bb, 0x118cf, 0x118db, 0x118f9, 0x118ff, 0x1190b, 0x11923, 0x11925, 0x11931, 0x11937, 0x1195b, 0x1196b, 0x1196d, 0x11975, 0x11979, 0x119b3, 0x119cb, 0x119cd, 0x119e5, 0x119fb, 0x119fd, 0x11a07, 0x11a0d, 0x11a23, 0x11a31, 0x11a43, 0x11a51, 0x11a57, 0x11a5d, 0x11a6b, 0x11a73, 0x11a79, 0x11a89, 0x11a9b, 0x11abf, 0x11ae3, 0x11ae5, 0x11b09, 0x11b1d, 0x11b27, 0x11b2b, 0x11b47, 0x11b4b, 0x11b55, 0x11b59, 0x11b65, 0x11b7d, 0x11b8b, 0x11b8d, 0x11b93, 0x11baf, 0x11bbb, 0x11bbd, 0x11bc3, 0x11bd7, 0x11be1, 0x11bff, 0x11c07, 0x11c13, 0x11c23, 0x11c29, 0x11c45, 0x11c4f, 0x11c57, 0x11c5d, 0x11c61, 0x11c7f, 0x11c85, 0x11c9d, 0x11ca1, 0x11cb5, 0x11ccb, 0x11ccd, 0x11cd9, 0x11ce9, 0x11cef, 0x11cf1, 0x11d17, 0x11d1b, 0x11d2d, 0x11d3f, 0x11d4b, 0x11d53, 0x11d59, 0x11d5f, 0x11d65, 0x11d69, 0x11d77, 0x11d81, 0x11d87, 0x11dc9, 0x11ded, 0x11dff, 0x11e21, 0x11e3f, 0x11e5f, 0x11e71, 0x11e7b, 0x11e99, 0x11e9f, 0x11ea9, 0x11ebd, 0x11ec3, 0x11ec5, 0x11ecf, 0x11edd, 0x11ee7, 0x11eeb, 0x11ef3, 0x11f13, 0x11f29, 0x11f2f, 0x11f3b, 0x11f3d, 0x11f57, 0x11f67, 0x11f6d, 0x11f73, 0x11f75, 0x11f83, 0x11f9b, 0x11f9d, 0x11fb9, 0x11fbf, 0x11fc1, 0x1200d, 0x1201f, 0x12025, 0x1203d, 0x12051, 0x12073, 0x1208f, 0x1209b, 0x120ab, 0x120b5, 0x120b9, 0x120c7, 0x120e3, 0x12105, 0x12109, 0x1211b, 0x12141, 0x12153, 0x12163, 0x12169, 0x1217b, 0x12187, 0x12195, 0x121a9, 0x121b1, 0x121c5, 0x121cf, 0x121e1, 0x121e7, 0x121f9, 0x12217, 0x12227, 0x12235, 0x12241, 0x12253, 0x12259, 0x1225f, 0x12269, 0x1227b, 0x1228b, 0x12293, 0x12295, 0x122a5, 0x122af, 0x122b1, 0x122cf, 0x122dd, 0x122e7, 0x122f9, 0x12315, 0x1231f, 0x12325, 0x12329, 0x12345, 0x12357, 0x1236d, 0x12383, 0x12397, 0x123a7, 0x123b9, 0x123e3, 0x123e9, 0x12403, 0x1240f, 0x1242d, 0x12439, 0x1243f, 0x1244b, 0x12453, 0x1248d, 0x12499, 0x1249f, 0x124b7, 0x124c5, 0x124d1, 0x124e7, 0x124ff, 0x12515, 0x12523, 0x12529, 0x1253b, 0x12557, 0x1255d, 0x12561, 0x12573, 0x12579, 0x12585, 0x1259b, 0x125bf, 0x125cd, 0x125e5, 0x125f7, 0x12607, 0x12615, 0x12623, 0x1263d, 0x12651, 0x1265b, 0x1266d, 0x12675, 0x1267f, 0x12683, 0x1269b, 0x126ad, 0x126c1, 0x126e9, 0x126fb, 0x12711, 0x1272b, 0x12733, 0x12739, 0x12759, 0x12763, 0x12771, 0x12787, 0x127a5, 0x127c5, 0x127d7, 0x127ed, 0x127f3, 0x1281b, 0x1281d, 0x1282d, 0x12841, 0x12847, 0x12871, 0x1287b, 0x1289f, 0x128a3, 0x128a9, 0x128af, 0x128c5, 0x128f5, 0x12901, 0x1290b, 0x12919, 0x1292f, 0x1293b, 0x12943, 0x12945, 0x12949, 0x12973, 0x1297f, 0x12983, 0x12989, 0x1299d, 0x129ad, 0x129b9, 0x129c7, 0x129d5, 0x129f1, 0x129f7, 0x12a25, 0x12a29, 0x12a31, 0x12a3b, 0x12a7f, 0x12a85, 0x12a9d, 0x12aa7, 0x12acd, 0x12ad5, 0x12ad9, 0x12ae3, 0x12ae9, 0x12af1, 0x12b05, 0x12b09, 0x12b27, 0x12b33, 0x12b35, 0x12b53, 0x12b69, 0x12b6f, 0x12b7b, 0x12b7d, 0x12b93, 0x12b99, 0x12baf, 0x12bbd, 0x12bd1, 0x12bdb, 0x12bdd, 0x12bf9, 0x12c07, 0x12c0b, 0x12c1f, 0x12c3b, 0x12c3d, 0x12c43, 0x12c4f, 0x12c5d, 0x12c61, 0x12c6b, 0x12c73, 0x12c79, 0x12c89, 0x12cad, 0x12cb9, 0x12ccd, 0x12cd3, 0x12ce3, 0x12cf7, 0x12cfb, 0x12d09, 0x12d27, 0x12d2d, 0x12d35, 0x12d41, 0x12d63, 0x12d65, 0x12d71, 0x12d77, 0x12d81, 0x12d8b, 0x12d8d, 0x12d9f, 0x12da9, 0x12ddb, 0x12de1, 0x12e0f, 0x12e17, 0x12e21, 0x12e33, 0x12e35, 0x12e47, 0x12e4b, 0x12e4d, 0x12e5f, 0x12e6f, 0x12e71, 0x12e8d, 0x12ea5, 0x12ea9, 0x12ec5, 0x12ec9, 0x12eeb, 0x12ef9, 0x12eff, 0x12f01, 0x12f07, 0x12f13, 0x12f3d, 0x12f43, 0x12f5b, 0x12f67, 0x12f7f, 0x12f89, 0x12f97, 0x12f9b, 0x12fa7, 0x12fb5, 0x12fbf, 0x12fcb, 0x12fd3, 0x12fdf, 0x13005, 0x13017, 0x13027, 0x1304b, 0x13059, 0x13065, 0x13077, 0x1307b, 0x1307d, 0x13081, 0x13093, 0x130af, 0x130b7, 0x130c3, 0x130db, 0x130dd, 0x130ed, 0x130f5, 0x130f9, 0x13107, 0x1310d, 0x13125, 0x13129, 0x1315d, 0x13161, 0x13173, 0x131a7, 0x131ab, 0x131cd, 0x131d9, 0x131e3, 0x131fb, 0x1322f, 0x13237, 0x1323d, 0x13249, 0x13257, 0x1325b, 0x13267, 0x13275, 0x13291, 0x1329d, 0x132ab, 0x132ad, 0x132b5, 0x132cb, 0x132d9, 0x132ef, 0x132f1, 0x13305, 0x1331b, 0x1332b, 0x13339, 0x13347, 0x13377, 0x1339f, 0x133a9, 0x133c5, 0x133d1, 0x133e1, 0x133eb, 0x133f3, 0x1340b, 0x13415, 0x13419, 0x13425, 0x13443, 0x13451, 0x1347f, 0x13489, 0x1349b, 0x134a7, 0x134bf, 0x134cd, 0x134fd, 0x13505, 0x1351d, 0x13521, 0x13533, 0x13547, 0x1354b, 0x13559, 0x13571, 0x1357d, 0x1359f, 0x135a3, 0x135b1, 0x135c9, 0x135d7, 0x135f3, 0x1360f, 0x1361b, 0x1361d, 0x13635, 0x1363f, 0x1366f, 0x13671, 0x1368d, 0x13699, 0x136a9, 0x136b7, 0x136c3, 0x136d1, 0x136d7, 0x136ed, 0x13707, 0x1370b, 0x13719, 0x13723, 0x13743, 0x1374f, 0x13751, 0x13775, 0x13779, 0x13797, 0x137ab, 0x137b9, 0x137cd, 0x137d5, 0x137e3, 0x13801, 0x1380d, 0x1381f, 0x13837, 0x13849, 0x1384f, 0x1385b, 0x1385d, 0x1386d, 0x1388f, 0x138a1, 0x138c7, 0x138cb, 0x138cd, 0x138df, 0x138fb, 0x13911, 0x13917, 0x13921, 0x13935, 0x1393f, 0x13941, 0x1394d, 0x13953, 0x1395f, 0x13965, 0x1396f, 0x13977, 0x13981, 0x13999, 0x139a3, 0x139b7, 0x139bd, 0x139c9, 0x139eb, 0x139ff, 0x13a09, 0x13a1b, 0x13a1d, 0x13a2d, 0x13a33, 0x13a4d, 0x13a55, 0x13a63, 0x13a65, 0x13a69, 0x13a77, 0x13a95, 0x13a9f, 0x13aaf, 0x13abb, 0x13abd, 0x13acf, 0x13af3, 0x13af9, 0x13b13, 0x13b29, 0x13b3b, 0x13b3d, 0x13b45, 0x13b5b, 0x13b6d, 0x13b73, 0x13b83, 0x13b89, 0x13b9d, 0x13bcb, 0x13bd3, 0x13bd9, 0x13be3, 0x13bef, 0x13bf1, 0x13c05, 0x13c21, 0x13c27, 0x13c47, 0x13c4b, 0x13c69, 0x13c6f, 0x13c93, 0x13cb1, 0x13cdb, 0x13cf3, 0x13cf9, 0x13cff, 0x13d07, 0x13d13, 0x13d23, 0x13d31, 0x13d3d, 0x13d43, 0x13d49, 0x13d67, 0x13d6b, 0x13d6d, 0x13d83, 0x13d91, 0x13db3, 0x13db5, 0x13dc1, 0x13dcd, 0x13dd9, 0x13def, 0x13dfb, 0x13e0b, 0x13e19, 0x13e1f, 0x13e31, 0x13e3b, 0x13e43, 0x13e45, 0x13e51, 0x13e57, 0x13e79, 0x13e85, 0x13e89, 0x13e8f, 0x13eb9, 0x13ed3, 0x13edf, 0x13ee9, 0x13efb, 0x13f03, 0x13f09, 0x13f11, 0x13f1d, 0x13f41, 0x13f81, 0x13f8b, 0x13f9f, 0x13fa5, 0x13fa9, 0x13fb7, 0x13fc5, 0x13fdb, 0x13ff5, 0x14019, 0x14075, 0x14079, 0x14085, 0x1409b, 0x1409d, 0x140ad, 0x140cd, 0x140d3, 0x140e3, 0x140f7, 0x14103, 0x14109, 0x1413f, 0x1414b, 0x14153, 0x14159, 0x14163, 0x14193, 0x14199, 0x141a9, 0x141af, 0x141bb, 0x141bd, 0x141e1, 0x141e7, 0x141f3, 0x141ff, 0x14203, 0x14205, 0x14211, 0x1424b, 0x14255, 0x14271, 0x1427b, 0x1427d, 0x14281, 0x14287, 0x142a5, 0x142a9, 0x142b7, 0x142bb, 0x142cf, 0x142e1, 0x142ed, 0x142f9, 0x14307, 0x1430b, 0x14323, 0x14343, 0x14349, 0x1437f, 0x14383, 0x143a7, 0x143bf, 0x143ef, 0x143f7, 0x143fb, 0x1440f, 0x14447, 0x1444d, 0x14471, 0x14477, 0x1447b, 0x1448d, 0x14495, 0x1449f, 0x144a3, 0x144bb, 0x144c5, 0x144d1, 0x144e1, 0x144f9, 0x1450b, 0x1451f, 0x14529, 0x14537, 0x14543, 0x14573, 0x14589, 0x1458f, 0x1459b, 0x1459d, 0x145b5, 0x145b9, 0x145e5, 0x145f7, 0x145fd, 0x14623, 0x14629, 0x14637, 0x14645, 0x14649, 0x1465d, 0x14673, 0x1468f, 0x14691, 0x146a1, 0x146cb, 0x146d5, 0x146df, 0x146e9, 0x146f1, 0x146fd, 0x14709, 0x14717, 0x14721, 0x1472d, 0x1474b, 0x1475f, 0x1477d, 0x14793, 0x14795, 0x14799, 0x147af, 0x147b7, 0x147c3, 0x147c9, 0x147eb, 0x147f9, 0x14809, 0x14811, 0x1481d, 0x14827, 0x14835, 0x1483f, 0x14841, 0x14847, 0x14859, 0x14881, 0x14893, 0x148a9, 0x148c3, 0x148d7, 0x148db, 0x148f5, 0x148ff, 0x1490b, 0x14915, 0x14919, 0x1492f, 0x14931, 0x14943, 0x1495d, 0x14967, 0x14989, 0x1498f, 0x149df, 0x149e5, 0x149fb, 0x14a13, 0x14a15, 0x14a23, 0x14a2f, 0x14a67, 0x14a6d, 0x14a75, 0x14a85, 0x14a89, 0x14a91, 0x14aa7, 0x14aad, 0x14ab5, 0x14abf, 0x14acb, 0x14adf, 0x14ae9, 0x14af7, 0x14afb, 0x14b03, 0x14b0f, 0x14b21, 0x14b4d, 0x14b63, 0x14b7d, 0x14b87, 0x14b95, 0x14bb7, 0x14bbb, 0x14bc9, 0x14bcf, 0x14bdb, 0x14bdd, 0x14bf3, 0x14bf9, 0x14bff, 0x14c0d, 0x14c19, 0x14c1f, 0x14c67, 0x14c73, 0x14c83, 0x14c9b, 0x14ca7, 0x14cb9, 0x14cc7, 0x14ccb, 0x14cf7, 0x14cfd, 0x14d0f, 0x14d2d, 0x14d39, 0x14d41, 0x14d4d, 0x14d53, 0x14d5f, 0x14d69, 0x14d71, 0x14d7b, 0x14d8b, 0x14d8d, 0x14db1, 0x14dd1, 0x14de7, 0x14deb, 0x14df3, 0x14e17, 0x14e4b, 0x14e6f, 0x14e77, 0x14e81, 0x14e87, 0x14ea3, 0x14eaf, 0x14eb1, 0x14ebd, 0x14edb, 0x14eeb, 0x14ef5, 0x14eff, 0x14f13, 0x14f25, 0x14f31, 0x14f45, 0x14f49, 0x14f4f, 0x14f57, 0x14f5b, 0x14f61, 0x14f6b, 0x14f6d, 0x14f8f, 0x14fa1, 0x14fc7, 0x14fcd, 0x14fe3, 0x14ffb, 0x14ffd, 0x15003, 0x1500f, 0x15021, 0x15027, 0x1502b, 0x15059, 0x15081, 0x1509f, 0x150a5, 0x150b7, 0x150bb, 0x150c9, 0x150d7, 0x150eb, 0x150ff, 0x15107, 0x15125, 0x15149, 0x1515d, 0x1516d, 0x15175, 0x15183, 0x15189, 0x151a1, 0x151a7, 0x151b3, 0x151b5, 0x151cd, 0x151d3, 0x151df, 0x1520b, 0x1521f, 0x1522f, 0x15245, 0x1525d, 0x15261, 0x1526b, 0x15289, 0x15297, 0x152b3, 0x152b9, 0x152bf, 0x152c7, 0x152d5, 0x152f7, 0x152fb, 0x15303, 0x15309, 0x1531d, 0x15327, 0x15363, 0x1536f, 0x15377, 0x1539f, 0x153a5, 0x153bd, 0x153c5, 0x153d1, 0x153db, 0x153e1, 0x153ed, 0x153f5, 0x1540d, 0x15413, 0x1541f, 0x15431, 0x15457, 0x1545b, 0x1547f, 0x15483, 0x15485, 0x15491, 0x15497, 0x154b5, 0x154b9, 0x154cb, 0x154cd, 0x154d5, 0x154df, 0x15527, 0x1552d, 0x1554b, 0x15593, 0x155af, 0x155b1, 0x155b7, 0x155cf, 0x155db, 0x155ed, 0x155f5, 0x1560f, 0x15617, 0x15621, 0x1562b, 0x1562d, 0x15647, 0x15653, 0x15655, 0x15687, 0x1568d, 0x15695, 0x1569f, 0x156a9, 0x156af, 0x156c5, 0x156cf, 0x156d1, 0x156f5, 0x15701, 0x1570d, 0x15729, 0x15751, 0x1575b, 0x15767, 0x15783, 0x1579d, 0x157ab, 0x157c7, 0x157d3, 0x157d9, 0x15807, 0x15825, 0x1583d, 0x15851, 0x1585b, 0x1586b, 0x15883, 0x15889, 0x158a1, 0x158b3, 0x158d3, 0x158d9, 0x158df, 0x158e9, 0x158ef, 0x158fb, 0x158fd, 0x1592b, 0x15935, 0x15939, 0x1594b, 0x1594d, 0x15963, 0x15969, 0x15981, 0x1598d, 0x15993, 0x1599f, 0x159a3, 0x159a9, 0x159b7, 0x159cf, 0x159d7, 0x159dd, 0x159eb, 0x159ff, 0x15a09, 0x15a0f, 0x15a11, 0x15a41, 0x15a4b, 0x15a55, 0x15a6f, 0x15a71, 0x15a7d, 0x15a99, 0x15aa5, 0x15ab7, 0x15ad7, 0x15adb, 0x15ae7, 0x15aed, 0x15af3, 0x15b0d, 0x15b15, 0x15b3b, 0x15b45, 0x15b4f, 0x15b57, 0x15b61, 0x15b79, 0x15bab, 0x15bad, 0x15bc7, 0x15bdf, 0x15be9, 0x15c05, 0x15c2d, 0x15c33, 0x15c4b, 0x15c81, 0x15c8b, 0x15c93, 0x15c99, 0x15ca3, 0x15ca5, 0x15cc9, 0x15cd1, 0x15cdd, 0x15ceb, 0x15d01, 0x15d07, 0x15d0d, 0x15d15, 0x15d1f, 0x15d31, 0x15d37, 0x15d3b, 0x15d4f, 0x15d6b, 0x15d7f, 0x15d91, 0x15dab, 0x15dc7, 0x15dd9, 0x15ddf, 0x15df1, 0x15dfd, 0x15e01, 0x15e0b, 0x15e13, 0x15e19, 0x15e25, 0x15e29, 0x15e37, 0x15e57, 0x15e67, 0x15e73, 0x15e83, 0x15eab, 0x15ead, 0x15ec1, 0x15ecb, 0x15ed5, 0x15ee5, 0x15f2b, 0x15f3f, 0x15f41, 0x15f47, 0x15f55, 0x15f8d, 0x15f95, 0x15fa3, 0x15fbd, 0x15fd1, 0x15fdd, 0x15ff3, 0x15ff9, 0x16009, 0x16017, 0x16021, 0x1602b, 0x16039, 0x1604d, 0x16055, 0x16063, 0x16065, 0x16077, 0x1609f, 0x160a3, 0x160b1, 0x160dd, 0x1610b, 0x16119, 0x16137, 0x1613b, 0x1613d, 0x16143, 0x1615b, 0x16161, 0x16167, 0x16175, 0x1617f, 0x161ad, 0x161b5, 0x161c7, 0x161d3, 0x161d5, 0x161ef, 0x161fd, 0x16201, 0x16213, 0x16231, 0x1623b, 0x16245, 0x16249, 0x1624f, 0x1625d, 0x16261, 0x1626b, 0x16273, 0x1627f, 0x1628f, 0x16297, 0x1629d, 0x162ab, 0x162b3, 0x162d5, 0x162d9, 0x162e9, 0x162fd, 0x16327, 0x1632d, 0x16333, 0x16335, 0x1634d, 0x16353, 0x16365, 0x16369, 0x1637b, 0x16381, 0x1638b, 0x163a3, 0x163b1, 0x163ed, 0x163f5, 0x16407, 0x1640d, 0x1641f, 0x16423, 0x1643b, 0x16443, 0x16445, 0x16451, 0x1645d, 0x16497, 0x1649b, 0x164ab, 0x164b9, 0x164e9, 0x164f7, 0x16503, 0x1650f, 0x16511, 0x1651b, 0x1652b, 0x16535, 0x16539, 0x1653f, 0x16547, 0x1654b, 0x16565, 0x16577, 0x16581, 0x1658d, 0x16593, 0x165a5, 0x165bb, 0x165d1, 0x165db, 0x165e7, 0x165f3, 0x16603, 0x16605, 0x16617, 0x16627, 0x16639, 0x1664b, 0x16655, 0x16659, 0x16669, 0x16671, 0x16681, 0x166a9, 0x166bb, 0x166d7, 0x16701, 0x16707, 0x16715, 0x16719, 0x1672f, 0x16731, 0x16743, 0x16749, 0x1674f, 0x1676b, 0x16779, 0x167a1, 0x167bf, 0x167cb, 0x167d9, 0x167df, 0x167e5, 0x167f7, 0x16801, 0x1680b, 0x1681f, 0x16829, 0x1683b, 0x16849, 0x16875, 0x1687f, 0x1688f, 0x1689d, 0x168ad, 0x168b9, 0x168c7, 0x168cb, 0x168d3, 0x168d5, 0x16917, 0x1691b, 0x16933, 0x16947, 0x16955, 0x16965, 0x16969, 0x16971, 0x1698d, 0x1699f, 0x169af, 0x169bb, 0x169c5, 0x169cf, 0x169d1, 0x169d7, 0x16a03, 0x16a05, 0x16a1b, 0x16a2d, 0x16a3f, 0x16a41, 0x16a53, 0x16a69, 0x16a6f, 0x16a77, 0x16a8b, 0x16a99, 0x16aa3, 0x16aa5, 0x16abd, 0x16ac3, 0x16ac9, 0x16acf, 0x16ae7, 0x16af5, 0x16b01, 0x16b0d, 0x16b23, 0x16b29, 0x16b57, 0x16b5b, 0x16b61, 0x16b6b, 0x16b8f, 0x16b9d, 0x16bab, 0x16bb3, 0x16bb5, 0x16bc7, 0x16bef, 0x16bfb, 0x16c03, 0x16c1d, 0x16c39, 0x16c41, 0x16c6f, 0x16c77, 0x16c81, 0x16c93, 0x16ca5, 0x16cb7, 0x16cc9, 0x16ce7, 0x16cf3, 0x16d15, 0x16d31, 0x16d3b, 0x16d43, 0x16d4f, 0x16d79, 0x16d89, 0x16d9d, 0x16db3, 0x16db9, 0x16dbf, 0x16dc1, 0x16dc7, 0x16de5, 0x16df1, 0x16e19, 0x16e2f, 0x16e3d, 0x16e51, 0x16e5d, 0x16e67, 0x16e7f, 0x16e85, 0x16e91, 0x16e97, 0x16e9b, 0x16eb5, 0x16ecb, 0x16ed9, 0x16eef, 0x16efd, 0x16f0f, 0x16f11, 0x16f21, 0x16f2b, 0x16f47, 0x16f55, 0x16f71, 0x16f81, 0x16f87, 0x16f8d, 0x16f95, 0x16fb7, 0x16fc9, 0x16fdb, 0x16fed, 0x16fff, 0x17025, 0x17029, 0x17057, 0x1705b, 0x17061, 0x1706b, 0x1706d, 0x17083, 0x1708f, 0x1709d, 0x170ab, 0x170b3, 0x170b9, 0x170d9, 0x170fd, 0x17103, 0x17111, 0x17117, 0x17127, 0x17133, 0x17159, 0x1715f, 0x1716f, 0x17177, 0x1717b, 0x17181, 0x17195, 0x171b1, 0x171bd, 0x171c3, 0x171dd, 0x171f9, 0x17205, 0x1720f, 0x1721b, 0x1721d, 0x17221, 0x1722d, 0x1725f, 0x17263, 0x17271, 0x17287, 0x1728b, 0x1728d, 0x17299, 0x172a3, 0x172a9, 0x172af, 0x172cf, 0x172dd, 0x172e1, 0x172e7, 0x17329, 0x1733d, 0x17343, 0x17345, 0x17361, 0x17367, 0x1736d, 0x17373, 0x1738f, 0x17391, 0x1739b, 0x173ad, 0x173b9, 0x173bf, 0x173d5, 0x173f1, 0x17403, 0x17411, 0x17439, 0x17447, 0x1744d, 0x17469, 0x1746f, 0x17471, 0x1747d, 0x17481, 0x17487, 0x1748d, 0x17495, 0x174b1, 0x174b7, 0x174bb, 0x174c5, 0x174d7, 0x174eb, 0x174ed, 0x17501, 0x17513, 0x17515, 0x17519, 0x17523, 0x17525, 0x17549, 0x1755d, 0x17583, 0x17597, 0x1759b, 0x175a1, 0x175a7, 0x175b3, 0x175cb, 0x175fd, 0x17607, 0x1760d, 0x17619, 0x1763b, 0x1763d, 0x17643, 0x17657, 0x17661, 0x17667, 0x17675, 0x17689, 0x1769d, 0x176a1, 0x176ab, 0x176c7, 0x176df, 0x176f1, 0x17711, 0x1771b, 0x1771d, 0x17735, 0x1773f, 0x17741, 0x1774b, 0x17781, 0x1778b, 0x17793, 0x177a5, 0x177a9, 0x177b7, 0x177c3, 0x177f5, 0x17809, 0x17811, 0x17835, 0x1783f, 0x17853, 0x1785f, 0x17869, 0x17877, 0x17899, 0x178c3, 0x178c9, 0x178cf, 0x178d1, 0x178db, 0x178dd, 0x178e7, 0x178eb, 0x178ed, 0x17901, 0x17907, 0x1790d, 0x17973, 0x17979, 0x1797f, 0x17991, 0x17997, 0x1799d, 0x179ab, 0x179b9, 0x179e9, 0x179ef, 0x179f1, 0x17a01, 0x17a1f, 0x17a57, 0x17a5b, 0x17a61, 0x17a7f, 0x17a91, 0x17aa7, 0x17aab, 0x17aad, 0x17ab9, 0x17ae3, 0x17ae5, 0x17aef, 0x17af1, 0x17af7, 0x17b05, 0x17b1d, 0x17b33, 0x17b39, 0x17b63, 0x17b7b, 0x17b95, 0x17ba9, 0x17bb1, 0x17bbb, 0x17bbd, 0x17be1, 0x17be7, 0x17beb, 0x17bf3, 0x17bf5, 0x17c07, 0x17c19, 0x17c2f, 0x17c37, 0x17c4f, 0x17c5d, 0x17c7f, 0x17c85, 0x17ca1, 0x17cb5, 0x17cbf, 0x17cc1, 0x17ccb, 0x17cd3, 0x17d03, 0x17d21, 0x17d33, 0x17d47, 0x17d59, 0x17d63, 0x17d7b, 0x17d87, 0x17da5, 0x17da9, 0x17db1, 0x17dc5, 0x17dd7, 0x17ddb, 0x17e1d, 0x17e27, 0x17e2b, 0x17e35, 0x17e41, 0x17e4b, 0x17e59, 0x17e65, 0x17e8b, 0x17e8d, 0x17e93, 0x17eaf, 0x17ebb, 0x17ec3, 0x17ec5, 0x17ed1, 0x17ed7, 0x17ee1, 0x17ee7, 0x17eed, 0x17eff, 0x17f0b, 0x17f0d, 0x17f2f, 0x17f31, 0x17f45, 0x17f5d, 0x17f73, 0x17f75, 0x17fa1, 0x17fb3, 0x17fc7, 0x17fd3, 0x17fe3, 0x17fe5, 0x17ffb, 0x18013, 0x18015, 0x18049, 0x1805d, 0x1806d, 0x18085, 0x180a1, 0x180a7, 0x180ad, 0x180cd, 0x180f7, 0x18103, 0x18105, 0x18117, 0x1811d, 0x1812b, 0x1814d, 0x1816f, 0x1817d, 0x18195, 0x181a3, 0x181a5, 0x181b7, 0x181db, 0x181e1, 0x181f9, 0x18211, 0x1821b, 0x1821d, 0x18227, 0x18235, 0x18241, 0x18255, 0x18265, 0x18277, 0x1828b, 0x18293, 0x1829f, 0x182bb, 0x182c3, 0x182c9, 0x182d1, 0x182e7, 0x182f3, 0x182f5, 0x182ff, 0x18307, 0x18315, 0x18329, 0x1833b, 0x18357, 0x1835d, 0x18361, 0x1836b, 0x18379, 0x18385, 0x18389, 0x183ab, 0x183b3, 0x183b9, 0x183bf, 0x183c1, 0x183d5, 0x183df, 0x183f1, 0x183f7, 0x18411, 0x1841b, 0x18433, 0x1844d, 0x18459, 0x18469, 0x1846f, 0x18477, 0x18493, 0x184af, 0x184b1, 0x184b7, 0x184cf, 0x184d1, 0x184dd, 0x184f9, 0x1850d, 0x18513, 0x18523, 0x18525, 0x18529, 0x18537, 0x18545, 0x1854f, 0x18561, 0x1856d, 0x18579, 0x1857f, 0x18585, 0x1859d, 0x185c7, 0x185cd, 0x185d9, 0x185df, 0x185e9, 0x185fb, 0x1860b, 0x18619, 0x18625, 0x18637, 0x1863d, 0x18657, 0x18683, 0x186ab, 0x186ad, 0x186c1, 0x186d9, 0x186f1, 0x186f7, 0x186fd, 0x1870f, 0x1871d, 0x18721, 0x18733, 0x18741, 0x18753, 0x18787, 0x187af, 0x187b1, 0x187c5, 0x187dd, 0x187f3, 0x1880f, 0x18833, 0x1884d, 0x1885f, 0x18863, 0x18871, 0x18887, 0x188a5, 0x188b1, 0x188c5, 0x188c9, 0x188d7, 0x188e1, 0x188eb, 0x18913, 0x1892f, 0x18931, 0x18943, 0x18949, 0x1895d, 0x18979, 0x1897f, 0x18985, 0x1899b, 0x189a7, 0x189ad, 0x189cb, 0x189d9, 0x189ef, 0x189fb, 0x18a0d, 0x18a29, 0x18a45, 0x18a4f, 0x18a61, 0x18a67, 0x18a6b, 0x18a73, 0x18a75, 0x18a9d, 0x18aad, 0x18ab3, 0x18ac1, 0x18ae3, 0x18ae5, 0x18b03, 0x18b35, 0x18b39, 0x18b47, 0x18b4b, 0x18b59, 0x18b63, 0x18b77, 0x18b81, 0x18b8d, 0x18b9f, 0x18bb7, 0x18bd1, 0x18bdb, 0x18bf5, 0x18c0d, 0x18c23, 0x18c31, 0x18c7f, 0x18c91, 0x18c97, 0x18c9d, 0x18ca1, 0x18cb9, 0x18ccb, 0x18ce3, 0x18cef, 0x18cfb, 0x18d05, 0x18d09, 0x18d1b, 0x18d27, 0x18d35, 0x18d41, 0x18d69, 0x18d7b, 0x18d7d, 0x18d95, 0x18da3, 0x18da5, 0x18db7, 0x18dbd, 0x18dc9, 0x18dcf, 0x18de7, 0x18e05, 0x18e09, 0x18e1b, 0x18e21, 0x18e3f, 0x18e47, 0x18e63, 0x18e69, 0x18e77, 0x18e87, 0x18ea3, 0x18ea9, 0x18eb1, 0x18ebb, 0x18ebd, 0x18ecf, 0x18f0b, 0x18f19, 0x18f1f, 0x18f2f, 0x18f51, 0x18f57, 0x18f5b, 0x18f61, 0x18f73, 0x18f89, 0x18f9b, 0x18fb9, 0x18fcb, 0x18fd3, 0x18fd9, 0x18fe5, 0x18fef, 0x18ffd, 0x19003, 0x1900f, 0x1901b, 0x19027, 0x19047, 0x1904b, 0x19055, 0x19071, 0x1908d, 0x190a5, 0x190af, 0x190b7, 0x190d7, 0x190e7, 0x190f5, 0x1910b, 0x1911f, 0x19123, 0x1913b, 0x19143, 0x1914f, 0x19151, 0x19157, 0x1915d, 0x1916b, 0x19179, 0x1919b, 0x191a1, 0x191ab, 0x191b9, 0x191d3, 0x191e5, 0x191e9, 0x191f1, 0x19219, 0x19225, 0x19231, 0x19243, 0x19251, 0x1926d, 0x19275, 0x19279, 0x19283, 0x19289, 0x19291, 0x192a7, 0x192bf, 0x192df, 0x192fd, 0x19305, 0x1930f, 0x19317, 0x19327, 0x19335, 0x1934b, 0x1934d, 0x19355, 0x1937b, 0x1939f, 0x193a9, 0x193b1, 0x193c5, 0x193cf, 0x193db, 0x193dd, 0x193eb, 0x193f3, 0x193ff, 0x19401, 0x19437, 0x1943b, 0x1943d, 0x19449, 0x19457, 0x19473, 0x19489, 0x19491, 0x194a7, 0x194ab, 0x194ad, 0x194c1, 0x194d5, 0x194ef, 0x19505, 0x19509, 0x19517, 0x1951b, 0x1952b, 0x19539, 0x1953f, 0x19547, 0x1955f, 0x19565, 0x19571, 0x1957b, 0x19587, 0x1958b, 0x1958d, 0x195a9, 0x195c3, 0x195d1, 0x195f3, 0x19605, 0x1962d, 0x19635, 0x1965f, 0x19669, 0x19677, 0x1967d, 0x19681, 0x196b7, 0x196db, 0x196e7, 0x196eb, 0x196f9, 0x19707, 0x1970d, 0x19713, 0x19715, 0x1971f, 0x19737, 0x19761, 0x1977f, 0x19797, 0x197a1, 0x197a7, 0x197ab, 0x197b9, 0x197d5, 0x197e3, 0x197e9, 0x197fd, 0x1980b, 0x19823, 0x1982f, 0x19831, 0x19843, 0x19851, 0x19875, 0x1988f, 0x198b9, 0x198c7, 0x198e9, 0x198fb, 0x1990f, 0x1991d, 0x19927, 0x1992b, 0x1992d, 0x19941, 0x19947, 0x19959, 0x1997d, 0x1998b, 0x1998d, 0x199a9, 0x199bd, 0x199c3, 0x199c9, 0x199d1, 0x199e7, 0x199ff, 0x19a0f, 0x19a1d, 0x19a35, 0x19a6f, 0x19a81, 0x19a8d, 0x19a95, 0x19aa3, 0x19ab7, 0x19abb, 0x19ad7, 0x19b07, 0x19b15, 0x19b31, 0x19b37, 0x19b5b, 0x19b5d, 0x19b6d, 0x19b79, 0x19b7f, 0x19b83, 0x19b9b, 0x19ba1, 0x19bad, 0x19bcb, 0x19bdf, 0x19bef, 0x19bfd, 0x19c09, 0x19c17, 0x19c2b, 0x19c3f, 0x19c53, 0x19c65, 0x19c69, 0x19c8d, 0x19ca3, 0x19cb1, 0x19cc5, 0x19ce1, 0x19ceb, 0x19cf3, 0x19cf5, 0x19d01, 0x19d19, 0x19d29, 0x19d2f, 0x19d3d, 0x19d45, 0x19d49, 0x19d7f, 0x19d8f, 0x19d9d, 0x19db9, 0x19dcb, 0x19de3, 0x19def, 0x19df1, 0x19df7, 0x19dfd, 0x19e07, 0x19e4f, 0x19e57, 0x19e5b, 0x19e61, 0x19e6d, 0x19e73, 0x19e79, 0x19e83, 0x19e91, 0x19ea7, 0x19eb5, 0x19eb9, 0x19ec7, 0x19ecb, 0x19edf, 0x19eef, 0x19ef1, 0x19efb, 0x19f05, 0x19f11, 0x19f17, 0x19f21, 0x19f4d, 0x19f53, 0x19f59, 0x19f65, 0x19f6f, 0x19f77, 0x19f7b, 0x19f93, 0x19f99, 0x19fa5, 0x19fb7, 0x19fbd, 0x19fc3, 0x19fc9, 0x19ff5, 0x1a011, 0x1a021, 0x1a02d, 0x1a033, 0x1a041, 0x1a059, 0x1a069, 0x1a06f, 0x1a07b, 0x1a095, 0x1a0c3, 0x1a0cf, 0x1a0db, 0x1a0f5, 0x1a0f9, 0x1a107, 0x1a10d, 0x1a113, 0x1a11f, 0x1a125, 0x1a129, 0x1a131, 0x1a137, 0x1a145, 0x1a185, 0x1a197, 0x1a1ab, 0x1a1d9, 0x1a1e3, 0x1a1fd, 0x1a23b, 0x1a275, 0x1a283, 0x1a289, 0x1a291, 0x1a29d, 0x1a2a7, 0x1a2ad, 0x1a2bf, 0x1a2c1, 0x1a2c7, 0x1a2fd, 0x1a31b, 0x1a321, 0x1a32b, 0x1a333, 0x1a347, 0x1a353, 0x1a35f, 0x1a365, 0x1a369, 0x1a377, 0x1a37b, 0x1a38b, 0x1a38d, 0x1a3b1, 0x1a3cf, 0x1a3dd, 0x1a3e1, 0x1a3f9, 0x1a413, 0x1a419, 0x1a449, 0x1a44f, 0x1a451, 0x1a467, 0x1a475, 0x1a479, 0x1a485, 0x1a4b5, 0x1a4bf, 0x1a4c1, 0x1a4cb, 0x1a4cd, 0x1a4e5, 0x1a4e9, 0x1a4f7, 0x1a4fd, 0x1a505, 0x1a517, 0x1a527, 0x1a52b, 0x1a535, 0x1a54d, 0x1a555, 0x1a559, 0x1a571, 0x1a581, 0x1a593, 0x1a5a3, 0x1a5b1, 0x1a5b7, 0x1a5bb, 0x1a5c5, 0x1a5cf, 0x1a5dd, 0x1a62d, 0x1a639, 0x1a63f, 0x1a64b, 0x1a655, 0x1a663, 0x1a665, 0x1a66f, 0x1a671, 0x1a67d, 0x1a699, 0x1a6a5, 0x1a6b7, 0x1a6c5, 0x1a6e1, 0x1a6ed, 0x1a6f3, 0x1a6f5, 0x1a71f, 0x1a723, 0x1a731, 0x1a757, 0x1a75d, 0x1a767, 0x1a773, 0x1a797, 0x1a7b3, 0x1a7b9, 0x1a7bf, 0x1a7cd, 0x1a7e3, 0x1a7e9, 0x1a801, 0x1a80d, 0x1a815, 0x1a831, 0x1a83b, 0x1a84f, 0x1a857, 0x1a861, 0x1a873, 0x1a8a7, 0x1a8d5, 0x1a8df, 0x1a8ef, 0x1a8fd, 0x1a903, 0x1a917, 0x1a91b, 0x1a933, 0x1a935, 0x1a94b, 0x1a94d, 0x1a953, 0x1a98b, 0x1a999, 0x1a9af, 0x1a9b1, 0x1a9c9, 0x1a9e7, 0x1a9ed, 0x1a9f5, 0x1aa09, 0x1aa0f, 0x1aa1b, 0x1aa1d, 0x1aa21, 0x1aa4d, 0x1aa53, 0x1aa77, 0x1aa7b, 0x1aa8d, 0x1aa99, 0x1aa9f, 0x1aabd, 0x1aac3, 0x1aad1, 0x1aad7, 0x1aadd, 0x1aaf5, 0x1ab0b, 0x1ab13, 0x1ab19, 0x1ab1f, 0x1ab3d, 0x1ab51, 0x1ab57, 0x1ab75, 0x1ab83, 0x1ab8f, 0x1aba7, 0x1abad, 0x1abb5, 0x1abbf, 0x1abd3, 0x1abd5, 0x1abd9, 0x1abef, 0x1ac1d, 0x1ac27, 0x1ac35, 0x1ac47, 0x1ac5f, 0x1ac69, 0x1ac81, 0x1ac8d, 0x1ac95, 0x1aca3, 0x1acb1, 0x1accf, 0x1acd7, 0x1ad13, 0x1ad2f, 0x1ad31, 0x1ad4f, 0x1ad5b, 0x1ad6b, 0x1ad75, 0x1ad79, 0x1ad83, 0x1ad91, 0x1ada7, 0x1adad, 0x1adc1, 0x1adcd, 0x1addf, 0x1ade5, 0x1adfb, 0x1ae15, 0x1ae23, 0x1ae3d, 0x1ae4f, 0x1ae5d, 0x1ae61, 0x1ae73, 0x1ae75, 0x1ae8f, 0x1ae97, 0x1ae9b, 0x1aebf, 0x1aed3, 0x1aedf, 0x1aee5, 0x1aee9, 0x1aef1, 0x1aefb, 0x1af11, 0x1af35, 0x1af39, 0x1af41, 0x1af65, 0x1af93, 0x1af99, 0x1afaf, 0x1afbd, 0x1afc5, 0x1afd1, 0x1afd7, 0x1afe1, 0x1afff, 0x1b013, 0x1b029, 0x1b03b, 0x1b043, 0x1b051, 0x1b05b, 0x1b07f, 0x1b083, 0x1b09b, 0x1b09d, 0x1b0ab, 0x1b0ad, 0x1b0b9, 0x1b0c7, 0x1b0d9, 0x1b0e3, 0x1b0ef, 0x1b109, 0x1b12b, 0x1b12d, 0x1b14d, 0x1b153, 0x1b15f, 0x1b163, 0x1b171, 0x1b177, 0x1b18b, 0x1b199, 0x1b1af, 0x1b1bb, 0x1b1cf, 0x1b1dd, 0x1b1e1, 0x1b1e7, 0x1b205, 0x1b209, 0x1b211, 0x1b21b, 0x1b24d, 0x1b259, 0x1b265, 0x1b277, 0x1b27b, 0x1b287, 0x1b2b1, 0x1b2b7, 0x1b2c9, 0x1b2d1, 0x1b2db, 0x1b2eb, 0x1b2ed, 0x1b313, 0x1b323, 0x1b345, 0x1b349, 0x1b34f, 0x1b35d, 0x1b37f, 0x1b38f, 0x1b397, 0x1b39d, 0x1b3b3, 0x1b3c7, 0x1b3e3, 0x1b3e9, 0x1b3f1, 0x1b40f, 0x1b41b, 0x1b41d, 0x1b435, 0x1b439, 0x1b455, 0x1b46f, 0x1b481, 0x1b499, 0x1b4bd, 0x1b4c9, 0x1b4f3, 0x1b50d, 0x1b531, 0x1b53b, 0x1b56b, 0x1b597, 0x1b5ad, 0x1b5b3, 0x1b5b9, 0x1b5bf, 0x1b5c7, 0x1b5d5, 0x1b5e3, 0x1b5e5, 0x1b5e9, 0x1b607, 0x1b60b, 0x1b619, 0x1b625, 0x1b631, 0x1b63d, 0x1b64f, 0x1b679, 0x1b67f, 0x1b691, 0x1b69b, 0x1b6b5, 0x1b6bf, 0x1b6d3, 0x1b6e5, 0x1b6ef, 0x1b6f7, 0x1b703, 0x1b70f, 0x1b711, 0x1b727, 0x1b74d, 0x1b755, 0x1b769, 0x1b76f, 0x1b77d, 0x1b793, 0x1b795, 0x1b7a3, 0x1b7a5, 0x1b7a9, 0x1b7d7, 0x1b7db, 0x1b7e1, 0x1b7ed, 0x1b7f9, 0x1b81b, 0x1b821, 0x1b82b, 0x1b82d, 0x1b84d, 0x1b853, 0x1b85f, 0x1b869, 0x1b88b, 0x1b88d, 0x1b8a9, 0x1b8d7, 0x1b8dd, 0x1b8e7, 0x1b8f9, 0x1b901, 0x1b90d, 0x1b913, 0x1b91f, 0x1b929, 0x1b92f, 0x1b949, 0x1b95b, 0x1b96d, 0x1b975, 0x1b983, 0x1b991, 0x1b9b5, 0x1b9b9, 0x1b9c7, 0x1b9f1, 0x1b9f7, 0x1ba07, 0x1ba15, 0x1ba31, 0x1ba45, 0x1ba57, 0x1ba5d, 0x1ba83, 0x1ba85, 0x1ba97, 0x1baa1, 0x1bab3, 0x1bab9, 0x1bacd, 0x1bae3, 0x1bafd, 0x1bb05, 0x1bb1b, 0x1bb27, 0x1bb2d, 0x1bb3f, 0x1bb4b, 0x1bb4d, 0x1bb6f, 0x1bb81, 0x1bba5, 0x1bbb1, 0x1bbbd, 0x1bbcf, 0x1bc07, 0x1bc0b, 0x1bc19, 0x1bc29, 0x1bc45, 0x1bc61, 0x1bc67, 0x1bc85, 0x1bc89, 0x1bc9b, 0x1bcab, 0x1bcc1, 0x1bcc7, 0x1bcf1, 0x1bcf7, 0x1bd09, 0x1bd1d, 0x1bd3f, 0x1bd53, 0x1bd63, 0x1bd65, 0x1bd7d, 0x1bd8b, 0x1bd8d, 0x1bd93, 0x1bda9, 0x1bdaf, 0x1bdbd, 0x1bdd7, 0x1bdf3, 0x1be0f, 0x1be17, 0x1be21, 0x1be33, 0x1be35, 0x1be39, 0x1be5f, 0x1be63, 0x1be69, 0x1be6f, 0x1be81, 0x1be95, 0x1bea5, 0x1bec9, 0x1beeb, 0x1bef3, 0x1bef9, 0x1beff, 0x1bf19, 0x1bf23, 0x1bf25, 0x1bf2f, 0x1bf31, 0x1bf37, 0x1bf43, 0x1bf4f, 0x1bf51, 0x1bf67, 0x1bf6b, 0x1bf79, 0x1bf7f, 0x1bf85, 0x1bfa7, 0x1bfad, 0x1bfc1, 0x1bfdf, 0x1bfe5, 0x1bffd, 0x1c017, 0x1c021, 0x1c027, 0x1c035, 0x1c04d, 0x1c069, 0x1c071, 0x1c07b, 0x1c07d, 0x1c087, 0x1c09f, 0x1c0b1, 0x1c0bb, 0x1c0c9, 0x1c0cf, 0x1c0db, 0x1c0dd, 0x1c0f3, 0x1c107, 0x1c10b, 0x1c115, 0x1c119, 0x1c137, 0x1c13d, 0x1c151, 0x1c175, 0x1c179, 0x1c17f, 0x1c183, 0x1c185, 0x1c1b3, 0x1c1cd, 0x1c1d3, 0x1c1d9, 0x1c1df, 0x1c1e9, 0x1c207, 0x1c21f, 0x1c223, 0x1c231, 0x1c24f, 0x1c251, 0x1c25d, 0x1c267, 0x1c285, 0x1c29b, 0x1c29d, 0x1c2c1, 0x1c2d5, 0x1c2df, 0x1c2e3, 0x1c2e5, 0x1c309, 0x1c341, 0x1c353, 0x1c371, 0x1c37d, 0x1c381, 0x1c39f, 0x1c3a5, 0x1c3b7, 0x1c3c3, 0x1c3c9, 0x1c3d7, 0x1c3e1, 0x1c3ed, 0x1c401, 0x1c413, 0x1c425, 0x1c429, 0x1c445, 0x1c45d, 0x1c46b, 0x1c479, 0x1c47f, 0x1c491, 0x1c4a7, 0x1c4d5, 0x1c4e3, 0x1c4e9, 0x1c511, 0x1c527, 0x1c52d, 0x1c533, 0x1c541, 0x1c54d, 0x1c553, 0x1c559, 0x1c577, 0x1c57d, 0x1c58b, 0x1c599, 0x1c5a3, 0x1c5b1, 0x1c5b7, 0x1c5d1, 0x1c5ed, 0x1c5f5, 0x1c609, 0x1c61b, 0x1c62d, 0x1c633, 0x1c639, 0x1c665, 0x1c67b, 0x1c68b, 0x1c695, 0x1c6af, 0x1c6dd, 0x1c6e1, 0x1c6f3, 0x1c701, 0x1c70d, 0x1c729, 0x1c73b, 0x1c743, 0x1c75b, 0x1c767, 0x1c76d, 0x1c775, 0x1c78f, 0x1c79b, 0x1c7ad, 0x1c7b5, 0x1c7bf, 0x1c7d5, 0x1c7e5, 0x1c7fd, 0x1c807, 0x1c813, 0x1c815, 0x1c819, 0x1c825, 0x1c837, 0x1c867, 0x1c86b, 0x1c879, 0x1c883, 0x1c889, 0x1c897, 0x1c8bf, 0x1c8cd, 0x1c8df, 0x1c8ef, 0x1c8f7, 0x1c8fd, 0x1c911, 0x1c91d, 0x1c933, 0x1c947, 0x1c94b, 0x1c955, 0x1c95f, 0x1c963, 0x1c969, 0x1c96f, 0x1c98d, 0x1c993, 0x1c995, 0x1c9a9, 0x1c9b1, 0x1c9bb, 0x1c9d7, 0x1c9db, 0x1ca03, 0x1ca11, 0x1ca2b, 0x1ca41, 0x1ca47, 0x1ca53, 0x1ca59, 0x1ca65, 0x1ca8b, 0x1ca93, 0x1caa5, 0x1caa9, 0x1caaf, 0x1cab7, 0x1cabd, 0x1cae1, 0x1cae7, 0x1caf3, 0x1cb15, 0x1cb19, 0x1cb1f, 0x1cb23, 0x1cb51, 0x1cb5d, 0x1cb6b, 0x1cb85, 0x1cb89, 0x1cbab, 0x1cbbf, 0x1cbd3, 0x1cbe9, 0x1cbfb, 0x1cc27, 0x1cc41, 0x1cc4b, 0x1cc65, 0x1cc6f, 0x1cc7b, 0x1cc87, 0x1cc99, 0x1cca3, 0x1cca5, 0x1ccb7, 0x1ccdd, 0x1cced, 0x1ccf5, 0x1ccff, 0x1cd0d, 0x1cd25, 0x1cd37, 0x1cd4f, 0x1cd57, 0x1cd79, 0x1cd9d, 0x1cdc7, 0x1cdcb, 0x1cdd5, 0x1cddf, 0x1cde9, 0x1cdef, 0x1cdf1, 0x1cdfb, 0x1ce13, 0x1ce3b, 0x1ce3d, 0x1ce49, 0x1ce57, 0x1ce6d, 0x1ce7f, 0x1ce83, 0x1ce89, 0x1ce9d, 0x1cea7, 0x1cead, 0x1ceb5, 0x1cebf, 0x1ced3, 0x1ceef, 0x1cef1, 0x1cefd, 0x1cf05, 0x1cf09, 0x1cf1b, 0x1cf21, 0x1cf2b, 0x1cf33, 0x1cf3f, 0x1cf4d, 0x1cf63, 0x1cf65, 0x1cfbd, 0x1cfe1, 0x1d007, 0x1d00d, 0x1d019, 0x1d02f, 0x1d073, 0x1d089, 0x1d08f, 0x1d0bf, 0x1d0cd, 0x1d0d5, 0x1d0e5, 0x1d0e9, 0x1d0fb, 0x1d103, 0x1d10f, 0x1d11d, 0x1d12b, 0x1d12d, 0x1d139, 0x1d14b, 0x1d153, 0x1d171, 0x1d181, 0x1d193, 0x1d19f, 0x1d1c5, 0x1d1d7, 0x1d1f3, 0x1d211, 0x1d221, 0x1d227, 0x1d24d, 0x1d255, 0x1d263, 0x1d26f, 0x1d28d, 0x1d295, 0x1d29f, 0x1d2bb, 0x1d2cf, 0x1d2e1, 0x1d2eb, 0x1d2ed, 0x1d301, 0x1d30b, 0x1d32f, 0x1d33d, 0x1d35b, 0x1d35d, 0x1d37f, 0x1d389, 0x1d39b, 0x1d3a1, 0x1d3bf, 0x1d3c1, 0x1d3cb, 0x1d3d3, 0x1d3d9, 0x1d3e9, 0x1d3ef, 0x1d41d, 0x1d42b, 0x1d43f, 0x1d441, 0x1d453, 0x1d455, 0x1d45f, 0x1d471, 0x1d499, 0x1d4b1, 0x1d4bb, 0x1d4bd, 0x1d4c3, 0x1d4dd, 0x1d4e7, 0x1d4f3, 0x1d4f5, 0x1d4f9, 0x1d513, 0x1d51f, 0x1d52f, 0x1d549, 0x1d557, 0x1d567, 0x1d57f, 0x1d583, 0x1d589, 0x1d591, 0x1d5ab, 0x1d5ad, 0x1d5b5, 0x1d5bf, 0x1d5cb, 0x1d5e3, 0x1d5e5, 0x1d5ef, 0x1d5f1, 0x1d5f7, 0x1d601, 0x1d613, 0x1d615, 0x1d623, 0x1d625, 0x1d63b, 0x1d65d, 0x1d66b, 0x1d691, 0x1d6ab, 0x1d6b3, 0x1d6b5, 0x1d6cd, 0x1d6d9, 0x1d6fd, 0x1d717, 0x1d727, 0x1d72d, 0x1d735, 0x1d759, 0x1d75f, 0x1d77b, 0x1d77d, 0x1d787, 0x1d7b1, 0x1d7b7, 0x1d7c9, 0x1d7d1, 0x1d7db, 0x1d7eb, 0x1d80f, 0x1d827, 0x1d839, 0x1d853, 0x1d87d, 0x1d899, 0x1d8b7, 0x1d8c3, 0x1d8c5, 0x1d8d1, 0x1d8f5, 0x1d8ff, 0x1d907, 0x1d90b, 0x1d90d, 0x1d91f, 0x1d931, 0x1d943, 0x1d945, 0x1d94f, 0x1d967, 0x1d975, 0x1d9b3, 0x1d9d3, 0x1d9f7, 0x1d9fb, 0x1da13, 0x1da15, 0x1da19, 0x1da31, 0x1da37, 0x1da43, 0x1da49, 0x1da5d, 0x1da67, 0x1da6d, 0x1da85, 0x1da9b, 0x1daa7, 0x1dab3, 0x1dab5, 0x1dacb, 0x1dad3, 0x1dad9, 0x1daef, 0x1db03, 0x1db0f, 0x1db35, 0x1db39, 0x1db47, 0x1db4b, 0x1db55, 0x1db63, 0x1db87, 0x1dba3, 0x1dba5, 0x1dbc5, 0x1dbd7, 0x1dbdd, 0x1dbed, 0x1dbf3, 0x1dbf9, 0x1dc0d, 0x1dc19, 0x1dc2f, 0x1dc3d, 0x1dc43, 0x1dc45, 0x1dc6d, 0x1dc7f, 0x1dc83, 0x1dc91, 0x1dc9b, 0x1dcab, 0x1dcad, 0x1dcb9, 0x1dcd3, 0x1dce3, 0x1dce5, 0x1dd0f, 0x1dd1b, 0x1dd1d, 0x1dd21, 0x1dd39, 0x1dd3f, 0x1dd47, 0x1dd4d, 0x1dd69, 0x1dd71, 0x1dd8b, 0x1dd95, 0x1dd99, 0x1dda3, 0x1ddcf, 0x1dde1, 0x1ddf3, 0x1de03, 0x1de05, 0x1de27, 0x1de4b, 0x1de4d, 0x1de65, 0x1de69, 0x1de7b, 0x1de81, 0x1de95, 0x1dea5, 0x1deaf, 0x1debd, 0x1dec3, 0x1decf, 0x1ded1, 0x1dedb, 0x1df01, 0x1df29, 0x1df37, 0x1df3b, 0x1df45, 0x1df49, 0x1df57, 0x1df73, 0x1df83, 0x1df85, 0x1df8f, 0x1df91, 0x1dfa1, 0x1dfcd, 0x1dfef, 0x1e013, 0x1e015, 0x1e023, 0x1e037, 0x1e045, 0x1e049, 0x1e05b, 0x1e061, 0x1e07f, 0x1e09d, 0x1e0ab, 0x1e0b3, 0x1e0b5, 0x1e0d5, 0x1e0d9, 0x1e0df, 0x1e0e9, 0x1e0fb, 0x1e117, 0x1e133, 0x1e14d, 0x1e165, 0x1e177, 0x1e193, 0x1e1a5, 0x1e1b7, 0x1e1c3, 0x1e1d1, 0x1e1db, 0x1e1ed, 0x1e209, 0x1e217, 0x1e21d, 0x1e22d, 0x1e233, 0x1e239, 0x1e25f, 0x1e28d, 0x1e2c5, 0x1e2eb, 0x1e2f9, 0x1e301, 0x1e31f, 0x1e325, 0x1e32f, 0x1e345, 0x1e351, 0x1e361, 0x1e373, 0x1e38f, 0x1e39b, 0x1e39d, 0x1e3ab, 0x1e3ad, 0x1e3c1, 0x1e3c7, 0x1e3e5, 0x1e3f7, 0x1e411, 0x1e42b, 0x1e439, 0x1e44b, 0x1e469, 0x1e46f, 0x1e471, 0x1e47d, 0x1e487, 0x1e48d, 0x1e4a3, 0x1e4cf, 0x1e4db, 0x1e4eb, 0x1e4f3, 0x1e501, 0x1e513, 0x1e537, 0x1e543, 0x1e567, 0x1e56b, 0x1e56d, 0x1e575, 0x1e57f, 0x1e59b, 0x1e5b5, 0x1e5bf, 0x1e5cd, 0x1e5d9, 0x1e5df, 0x1e5e5, 0x1e5fb, 0x1e607, 0x1e60b, 0x1e62f, 0x1e631, 0x1e63d, 0x1e643, 0x1e64f, 0x1e66b, 0x1e685, 0x1e689, 0x1e697, 0x1e69d, 0x1e6a1, 0x1e6ad, 0x1e6b9, 0x1e6d5, 0x1e6e3, 0x1e6f1, 0x1e6f7, 0x1e709, 0x1e71b, 0x1e72d, 0x1e735, 0x1e73f, 0x1e741, 0x1e74b, 0x1e755, 0x1e763, 0x1e76f, 0x1e777, 0x1e781, 0x1e78b, 0x1e793, 0x1e79f, 0x1e7a5, 0x1e7af, 0x1e7bb, 0x1e7e1, 0x1e817, 0x1e833, 0x1e85f, 0x1e86f, 0x1e877, 0x1e87d, 0x1e881, 0x1e895, 0x1e899, 0x1e8a5, 0x1e8cf, 0x1e8d1, 0x1e8ed, 0x1e901, 0x1e923, 0x1e925, 0x1e929, 0x1e93b, 0x1e957, 0x1e96b, 0x1e973, 0x1e97f, 0x1e98f, 0x1e997, 0x1e9cd, 0x1e9e3, 0x1e9ef, 0x1e9f1, 0x1e9fb, 0x1e9fd, 0x1ea13, 0x1ea19, 0x1ea1f, 0x1ea29, 0x1ea43, 0x1ea51, 0x1ea61, 0x1ea89, 0x1ea91, 0x1ea9d, 0x1eaa7, 0x1eab9, 0x1eac1, 0x1eac7, 0x1ead5, 0x1eae5, 0x1eaef, 0x1eaf7, 0x1eafd, 0x1eb05, 0x1eb1b, 0x1eb21, 0x1eb2b, 0x1eb2d, 0x1eb41, 0x1eb55, 0x1eb7b, 0x1eb9f, 0x1eba9, 0x1ebb1, 0x1ebc3, 0x1ebc5, 0x1ebcf, 0x1ebeb, 0x1ec0b, 0x1ec2f, 0x1ec31, 0x1ec43, 0x1ec4f, 0x1ec5b, 0x1ec5d, 0x1ec67, 0x1ec6d, 0x1ec79, 0x1ec97, 0x1ecad, 0x1ecb3, 0x1ecb5, 0x1eccb, 0x1ecd9, 0x1ece5, 0x1ece9, 0x1ecfb, 0x1ed03, 0x1ed11, 0x1ed1d, 0x1ed27, 0x1ed39, 0x1ed5f, 0x1ed6f, 0x1ed81, 0x1ed95, 0x1ed9f, 0x1eda9, 0x1edbb, 0x1edcf, 0x1eddb, 0x1edf3, 0x1ee1b, 0x1ee27, 0x1ee2b, 0x1ee35, 0x1ee53, 0x1ee63, 0x1ee71, 0x1ee99, 0x1eeaf, 0x1eeb7, 0x1eebd, 0x1eedb, 0x1eee7, 0x1eeed, 0x1eef3, 0x1ef0d, 0x1ef23, 0x1ef2f, 0x1ef3d, 0x1ef57, 0x1ef67, 0x1ef73, 0x1ef79, 0x1ef85, 0x1ef97, 0x1efa1, 0x1efab, 0x1efad, 0x1efb3, 0x1efb9, 0x1efc1, 0x1efdf, 0x1efe3, 0x1eff7, 0x1f009, 0x1f02d, 0x1f039, 0x1f04d, 0x1f055, 0x1f05f, 0x1f065, 0x1f069, 0x1f087, 0x1f095, 0x1f0af, 0x1f0bd, 0x1f0d1, 0x1f0f9, 0x1f10b, 0x1f113, 0x1f137, 0x1f13b, 0x1f145, 0x1f157, 0x1f161, 0x1f175, 0x1f189, 0x1f18f, 0x1f1a7, 0x1f1ab, 0x1f1bf, 0x1f1cb, 0x1f1d3, 0x1f1e3, 0x1f207, 0x1f20d, 0x1f215, 0x1f229, 0x1f245, 0x1f249, 0x1f27f, 0x1f283, 0x1f297, 0x1f2a1, 0x1f2ab, 0x1f2b9, 0x1f2d5, 0x1f2df, 0x1f2f1, 0x1f317, 0x1f32d, 0x1f335, 0x1f359, 0x1f369, 0x1f36f, 0x1f381, 0x1f387, 0x1f393, 0x1f395, 0x1f399, 0x1f3a3, 0x1f3af, 0x1f3cf, 0x1f3f9, 0x1f41f, 0x1f423, 0x1f42f, 0x1f43b, 0x1f43d, 0x1f451, 0x1f457, 0x1f46b, 0x1f489, 0x1f48f, 0x1f49d, 0x1f4a1, 0x1f4c1, 0x1f4d3, 0x1f4e9, 0x1f4f1, 0x1f4fb, 0x1f511, 0x1f51b, 0x1f51d, 0x1f527, 0x1f539, 0x1f553, 0x1f565, 0x1f56f, 0x1f571, 0x1f58b, 0x1f5c5, 0x1f5d7, 0x1f5ff, 0x1f60f, 0x1f621, 0x1f627, 0x1f62b, 0x1f635, 0x1f639, 0x1f655, 0x1f687, 0x1f693, 0x1f69f, 0x1f6a5, 0x1f6c5, 0x1f6dd, 0x1f6eb, 0x1f6f3, 0x1f6f9, 0x1f707, 0x1f715, 0x1f725, 0x1f743, 0x1f74f, 0x1f751, 0x1f767, 0x1f76b, 0x1f775, 0x1f783, 0x1f791, 0x1f7b3, 0x1f7b5, 0x1f7bf, 0x1f7cd, 0x1f7e9, 0x1f7ef, 0x1f7fb, 0x1f801, 0x1f825, 0x1f83d, 0x1f849, 0x1f857, 0x1f873, 0x1f8ad, 0x1f8bf, 0x1f8cb, 0x1f8d9, 0x1f8e3, 0x1f8f1, 0x1f905, 0x1f939, 0x1f93f, 0x1f94d, 0x1f953, 0x1f971, 0x1f977, 0x1f97b, 0x1f9bb, 0x1f9cf, 0x1f9d1, 0x1f9dd, 0x1f9e7, 0x1f9f5, 0x1fa17, 0x1fa21, 0x1fa27, 0x1fa3f, 0x1fa41, 0x1fa4b, 0x1fa59, 0x1fa7d, 0x1fa8b, 0x1fa93, 0x1fa95, 0x1faa5, 0x1fab1, 0x1fadb, 0x1fae7, 0x1faeb, 0x1fb01, 0x1fb1f, 0x1fb49, 0x1fb4f, 0x1fb57, 0x1fb5b, 0x1fb6d, 0x1fb7f, 0x1fb83, 0x1fb85, 0x1fb97, 0x1fb9d, 0x1fba7, 0x1fbab, 0x1fbc7, 0x1fbcb, 0x1fbcd, 0x1fbdf, 0x1fbe9, 0x1fbf1, 0x1fc0f, 0x1fc11, 0x1fc1b, 0x1fc2d, 0x1fc47, 0x1fc55, 0x1fc59, 0x1fc63, 0x1fc71, 0x1fc77, 0x1fc7d, 0x1fc8d, 0x1fc9f, 0x1fca9, 0x1fcbd, 0x1fcc9, 0x1fcdb, 0x1fce1, 0x1fce7, 0x1fced, 0x1fd07, 0x1fd0d, 0x1fd23, 0x1fd29, 0x1fd2f, 0x1fd3d, 0x1fd43, 0x1fd4f, 0x1fd57, 0x1fd73, 0x1fd75, 0x1fd85, 0x1fd97, 0x1fd9b, 0x1fdb3, 0x1fdbf, 0x1fdc1, 0x1fdd3, 0x1fde9, 0x1fdfb, 0x1fe15, 0x1fe25, 0x1fe31, 0x1fe37, 0x1fe49, 0x1fe67, 0x1fe79, 0x1fe83, 0x1fe91, 0x1fea1, 0x1fec1, 0x1fee5, 0x1fee9, 0x1fefb, 0x1fefd, 0x1ff05, 0x1ff33, 0x1ff35, 0x1ff39, 0x1ff5f, 0x1ff71, 0x1ff93, 0x1ffa5, 0x1ffb1, 0x1ffeb, 0x1ffed };

const word* irreducible_polynomials[17] = {NULL,
                                           _irreducible_polynomials_degree_01, _irreducible_polynomials_degree_02, 
                                           _irreducible_polynomials_degree_03, _irreducible_polynomials_degree_04, 
                                           _irreducible_polynomials_degree_05, _irreducible_polynomials_degree_06,
                                           _irreducible_polynomials_degree_07, _irreducible_polynomials_degree_08,
                                           _irreducible_polynomials_degree_09, _irreducible_polynomials_degree_10,
                                           _irreducible_polynomials_degree_11, _irreducible_polynomials_degree_12, 
                                           _irreducible_polynomials_degree_13, _irreducible_polynomials_degree_14,
                                           _irreducible_polynomials_degree_15, _irreducible_polynomials_degree_16};
