/*
 *   Copyright (C) 2003,2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT6MReceive.h"
#include "JT6MLookups.h"
#include "JT6MFoldMessages.h"
#include "JT6MApp.h"

#include "common/Exception.h"
#include "common/SFFT.h"

#include <wx/datetime.h>
#include <wx/debug.h>
#include <wx/log.h>

#include <cmath>
using namespace std;

CJT6MReceive::CJT6MReceive(const wxString& name, EWho who) :
CReceive(name, who),
m_audioSamples(NULL),
m_syncSamples(NULL),
m_binSamples(),
m_correlations(NULL),
m_samplesCount(0),
m_fft(JT6M_FFT_LENGTH)
{
	m_audioSamples = new double[JT6M_MAX_AUDIO_DATA];
	m_syncSamples = new double[JT6M_MAX_AUDIO_DATA];

	for (int i = 0; i < (JT6M_ALPHABET_COUNT + 1); i++)
		m_binSamples[i] = new double[JT6M_MAX_AUDIO_DATA / JT6M_SYMBOL_LENGTH];

        int corrCount = (JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST) * (3 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR;
        m_correlations = new CCorrelation[corrCount];
}

CJT6MReceive::~CJT6MReceive()
{
	delete[] m_audioSamples;
	delete[] m_syncSamples;

	for (int i = 0; i < (JT6M_ALPHABET_COUNT + 1); i++)
		delete[] m_binSamples[i];

	delete[] m_correlations;
}

void CJT6MReceive::reset(bool firstTime)
{
	int corrCount = (JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST) * (3 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR;
	for (int i = 0; i < corrCount; i++)
		m_correlations[i].clear();

	m_samplesCount = 0;
}

void CJT6MReceive::run()
{
	wxString id = createId();

	bool end = false;

	openSoundDevice();

	// The main loop ends if we have been asked to end by the controller
	// isStopped() is true, or we have enough samples, or our time is
	// at an end. At around 29.8 or 59.8 on the clock.
	while (!isStopped() && !end && m_samplesCount < (JT6M_MAX_AUDIO_DATA - JT6M_SOUNDBUF_LENGTH)) {
		int len = JT6M_SOUNDBUF_LENGTH;

		getSoundDevice()->read(m_audioSamples + m_samplesCount, len);

		if (len <= 0) break;

		m_samplesCount += len;

		::wxGetApp().showAudio(m_audioSamples[m_samplesCount - 100], getWho());

		end = getEndTime();
	}

	closeSoundDevice();

	recordAudio(id, m_audioSamples, m_samplesCount);

	double bins[JT6M_FFT_LENGTH];

	for (int samplesCount = 0; samplesCount < m_samplesCount; samplesCount++) {
		if (samplesCount >= JT6M_FFT_LENGTH && (samplesCount % JT6M_SKIP_FACTOR) == 0) {
			m_fft.process(m_audioSamples + samplesCount - JT6M_FFT_LENGTH, bins);
			storeCorrelations(bins, samplesCount);
		}
	}

	// Create the data for the GUI graphs and send it to the GUI
	createReceiveData();

	// Find the approximate correlation time and exact bin number
	int syncBin, timeOffset;
	findCorrelation(syncBin, timeOffset);

	// Find the exact correlation time
	correlate(syncBin, timeOffset);

	// Create the message carrier object with the raw FFT outputs
	// for the FFT bins.
	decode(id, syncBin, timeOffset);
}

/*
 * "You are not expected to understand this"
 *
 * This horrible piece of code is the approximate correlator that
 * slides the reference signal past the incoming data and performs the
 * appropriate correlation calculation and stores it in the appropriate
 * CCorrelation object.
 *
 * "This code was hard to write, therefore it should be hard to understand"
 */
void CJT6MReceive::storeCorrelations(double* data, int samplesCount)
{
	wxASSERT(m_correlations != NULL);
	wxASSERT(data != NULL);
	wxASSERT(samplesCount >= 0);

	int sampleNo = samplesCount / JT6M_SKIP_FACTOR;

	for (int t = 0; t < (3 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR; t++) {
		int pos = (sampleNo + t) % (3 * JT6M_SYMBOL_LENGTH / JT6M_SKIP_FACTOR);

		if ((pos % (JT6M_SYMBOL_LENGTH / JT6M_SKIP_FACTOR)) == 0) {
			int index = pos / (JT6M_SYMBOL_LENGTH / JT6M_SKIP_FACTOR);

			double mult = (index == 2) ? +1.0 : -0.5;

			if (index > 2)
				throw CException(wxT("index > 2 in storeCorrelations"));

			for (int bin = 0; bin < (JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST); bin++) {
				int index = getCorrelationsIndex(bin, t);
				m_correlations[index].addProduct(mult, data[bin + JT6M_SYNCBIN_FIRST]);
			}
		}
	}
}

/*
 * This is simpler than it looks. It simply searches a 2D array (stored as a
 * 1D array) for the best correlation value and returns it. These are
 * only approximate correlations, the exact FFT bin is returned but only
 * the approximate timing.
 */
void CJT6MReceive::findCorrelation(int& syncBin, int& offset) const
{
	wxASSERT(m_correlations != NULL);

	double quality = 0.0;
	int maxI = 0, maxJ = 0;

	for (int i = 0; i < (JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST); i++) {
		for (int j = 0; j < (1 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR; j++) {

			int n = getCorrelationsIndex(i, j);

			double value;
			bool ret = m_correlations[n].getRawValue(value);

			if (ret && value > quality) {
				quality = value;
				maxJ    = j;
				maxI    = i;
			}
		}
	}

	offset  = maxJ * JT6M_SKIP_FACTOR;;
	syncBin = maxI + JT6M_SYNCBIN_FIRST;
}

/*
 * This method finds the exact correlation time within the already
 * detected synchronisation bin. Because of the previous
 * approximate correlation, we only have to search within a relatively
 * narrow time range, thus saving a large amount of time.
 */
void CJT6MReceive::correlate(int syncBin, int& offset) const
{
	wxASSERT(m_audioSamples != NULL);
	wxASSERT(m_samplesCount > 0);
	wxASSERT(syncBin >= JT6M_SYNCBIN_FIRST);

	CSFFT sfft(JT6M_FFT_LENGTH, syncBin, syncBin + 1);

	double bins[JT6M_FFT_LENGTH];

	for (int i = 0; i < m_samplesCount; i++) {
		sfft.process(m_audioSamples[i], bins);
		m_syncSamples[i] = bins[syncBin];
	}

//	int startOffset = offset - JT6M_SKIP_FACTOR;
	int startOffset = offset - (3 * JT6M_SYMBOL_LENGTH);
	while (startOffset < 0) startOffset = 0;
//	int   endOffset = startOffset + 2 * JT6M_SKIP_FACTOR;
	int   endOffset = startOffset + 6 * JT6M_SYMBOL_LENGTH;

	double bestCorrelation = 0.0;
	for (int timeOffset = startOffset; timeOffset < endOffset; timeOffset++) {
		CCorrelation correlation;

		int index = 0;
		for (int pos = timeOffset; pos < m_samplesCount; pos += JT6M_SYMBOL_LENGTH) {

			double mult = (index == 2) ? +1.0 : -0.5;

			if (index > 2)
				throw CException(wxT("index > 2 in correlate"));

			correlation.addProduct(mult, m_syncSamples[pos]);

			index = (index + 1) % 3;
		}

		double value;
		bool ret = correlation.getRawValue(value);

		if (ret && value > bestCorrelation) {
			bestCorrelation = value;
			offset          = timeOffset;
		}
	}

	offset %= (3 * JT6M_SYMBOL_LENGTH);
	offset -= 2 * JT6M_SYMBOL_LENGTH;

	if (offset < 0)
		offset += 3 * JT6M_SYMBOL_LENGTH;
}

/*
 * At this stage we do the decoding of the FFT bins that contain the message
 * data. We have the timing and location of these FFT bins so it is a
 * quick and efficient piece of code. We simply store the message data and
 * make no decision about which character is represented where, this is
 * done later in the CJT6MMessage class.
 */

// XXX
// Must check getMinLength() here
void CJT6MReceive::decode(const wxString& id, int syncBin, int timeOffset)
{
	CJT6MFoldMessages* messages[JT6M_MAX_MESSAGE_LENGTH / 2];
	for (int i = 0; i < (JT6M_MAX_MESSAGE_LENGTH / 2); i++)
		messages[i] = new CJT6MFoldMessages(2 + i * 2);

	double bins[JT6M_FFT_LENGTH];

	CJT6MLookups lookups;

	int pos = 0;
	bool inBurst = false;
	int burstStart = 0;

	for (int index = timeOffset; index < (m_samplesCount - JT6M_FFT_LENGTH); index += JT6M_SYMBOL_LENGTH) {
		m_fft.process(m_audioSamples + index, bins);

		for (int i = 0; i < (JT6M_ALPHABET_COUNT + 1); i++)
			m_binSamples[i][pos] = bins[syncBin + i];

		if ((pos % 3) == 0) {
			CAverage noise;
			noise.addValue(bins[syncBin - 2]);
			noise.addValue(bins[syncBin - 3]);
			noise.addValue(bins[syncBin - 4]);
			noise.addValue(bins[syncBin - 5]);

			if (!inBurst && bins[syncBin] > noise.getAverage()) {
				burstStart = index;
				inBurst    = true;
			} else if (inBurst && bins[syncBin] <= noise.getAverage()) {
				double timStart = double(burstStart - JT6M_FFT_LENGTH) / double(JT6M_SAMPLE_RATE);
				int     timDiff = 1000 * (index - burstStart) / JT6M_SAMPLE_RATE;
				int          DF = ((syncBin - 50) * JT6M_SAMPLE_RATE) / JT6M_SYMBOL_LENGTH;

//				CJT6MMessage* message = new CJT6MMessage(id, timStart, timDiff, 0, DF, 0, text);
//				receiveMessage(message);

				inBurst = false;
			}
		} else {
			if (inBurst) {
				for (int i = 0; i < JT6M_MAX_MESSAGE_LENGTH / 2; i++)
					messages[i]->addBins(pos, bins + syncBin + 1);
			}
		}

		pos++;
	}

	if (inBurst) {
		double timStart = double(burstStart - JT6M_FFT_LENGTH) / double(JT6M_SAMPLE_RATE);
		int     timDiff = 1000 * (m_samplesCount - burstStart) / JT6M_SAMPLE_RATE;
		int          DF = ((syncBin - 50) * JT6M_SAMPLE_RATE) / JT6M_SYMBOL_LENGTH;

//		CJT6MMessage* message = new CJT6MMessage(id, timStart, timDiff, 0, DF, 0, text);
//		::wxGetApp().receiveMessage(message, getWho());
	}

	int       bestI = 0;
	double bestQual = messages[0]->getQuality();

	for (int i = 1; i < JT6M_MAX_MESSAGE_LENGTH / 2; i++) {
		double qual = messages[i]->getQuality();

		if (qual > bestQual) {
			bestQual = qual;
			bestI    = i;
		}
	}

	fprintf(stderr, "%d: %f %s\n", messages[bestI]->getLength(),
                                       messages[bestI]->getQuality(),
                                       messages[bestI]->getMessage().c_str());

	for (int i = 0; i < JT6M_MAX_MESSAGE_LENGTH / 2; i++)
		delete messages[i];
}

/*
 * Prepare the results of the approximate correlation so that they may
 * be displayed on the graph in the GUI.
 */
void CJT6MReceive::createReceiveData() const
{
	CJT6MLevels* levels = new CJT6MLevels();

	levels->setAudioData(m_audioSamples, m_samplesCount);
	levels->setSyncData(m_syncSamples, m_samplesCount);

	for (int i = 0; i < (JT6M_ALPHABET_COUNT + 1); i++)
		levels->setSpectrumData(i, m_binSamples[i], m_samplesCount);

	::wxGetApp().showLevels(levels, getWho());
}

int CJT6MReceive::getCorrelationsIndex(int bin, int offset) const
{
	int nBins   = JT6M_SYNCBIN_LAST - JT6M_SYNCBIN_FIRST;
	int nOffset = (3 * JT6M_SYMBOL_LENGTH) / JT6M_SKIP_FACTOR;

	wxASSERT(bin >= 0 && bin < nBins);
	wxASSERT(offset >= 0 && offset < nOffset);

	return bin * nOffset + offset;
}

