/*

Sound audio library
   
Copyright (c) 1999 Pascal Hofstee & Alain Kalker
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: info.c,v 1.10 1999/12/07 12:10:03 daeron Exp $

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <audiofile.h>
#include "wsound.h"

#include <assert.h>

SAudioFileInfo*
SGetAudioFileInfo(char *file)
{
	SAudioFileInfo	*afInfo;
	AFfilehandle	inFile;
	char		*path;
		
	assert(file != NULL);
	
	/* open audio file */
	inFile = afOpenFile(file, "r", NULL);
	if (!inFile) {
		SErrorCode = SERR_AFOPEN;
		return NULL;
	}
	
	/* duplicate path for internal storage */
	path = strdup(file);
	if (!path) {
		SErrorCode = SERR_NOMEMORY;
		return NULL;
	}
	
	/* aquire memory to store file info */
	afInfo = malloc(sizeof(struct SAudioFileInfo));
	if (!afInfo) {
		afCloseFile(inFile);
		SErrorCode = SERR_NOMEMORY;
		return NULL;
	}
	
	afInfo->FileHandle	= inFile;
	afInfo->Path		= path;
	afInfo->FileFormat	= afGetFileFormat(inFile, &afInfo->Version);
	afGetSampleFormat(inFile, AF_DEFAULT_TRACK, &afInfo->SampleFormat, &afInfo->SampleWidth);
	afInfo->SampleRate	= afGetRate(inFile, AF_DEFAULT_TRACK);
	afInfo->ByteOrder	= afGetByteOrder(inFile, AF_DEFAULT_TRACK);
	afInfo->Channels	= afGetChannels(inFile, AF_DEFAULT_TRACK);
	afInfo->FrameCount	= afGetFrameCount(inFile, AF_DEFAULT_TRACK);
	afInfo->TrackBytes	= afGetTrackBytes(inFile, AF_DEFAULT_TRACK);
	afInfo->DataOffset	= afGetDataOffset(inFile, AF_DEFAULT_TRACK);
	afInfo->Compression	= afGetCompression(inFile, AF_DEFAULT_TRACK);
	
	return afInfo;
}

void
SPrintAudioFileInfo(SAudioFileInfo *afInfo)
{
	assert(afInfo != NULL);

	fprintf(stderr, "AudioFile Information for %s\n", afInfo->Path);
	fprintf(stderr, "File Format    ");
	switch (afInfo->FileFormat) {
		case AF_FILE_AIFFC:
			fprintf(stderr, "Audio Interchange File Format AIFF-C (aifc)");
			break;
		case AF_FILE_AIFF:
			fprintf(stderr, "Audio Interchange File Format (aiff)");
			break;
		case AF_FILE_NEXTSND:
			fprintf(stderr, "NeXT .snd/Sun .au Format (next)");
			break;
		case AF_FILE_WAVE:
			fprintf(stderr, "MS RIFF WAVE Format (wave)");
			break;
		default:
			fprintf(stderr, "Unknown file");
	}
	fprintf(stderr, "\n");
	

	fprintf(stderr, "Compression    ");
	switch (afInfo->Compression) {
		case AF_COMPRESSION_NONE:
			fprintf(stderr, "None\n");
			break;
		case AF_COMPRESSION_G722:
			fprintf(stderr, "G722\n");
			break;
		case AF_COMPRESSION_G711_ULAW:
			fprintf(stderr, "G711 ulaw\n");
			break;
		case AF_COMPRESSION_G711_ALAW:
			fprintf(stderr, "G711 alaw\n");
			break;
		case AF_COMPRESSION_APPLE_ACE2:
			fprintf(stderr, "Apple ACE2\n");
			break;
		case AF_COMPRESSION_APPLE_ACE8:
			fprintf(stderr, "Apple ACE8\n");
			break;
		case AF_COMPRESSION_APPLE_MAC3:
			fprintf(stderr, "Apple MAC3\n");
			break;
		case AF_COMPRESSION_APPLE_MAC6:
			fprintf(stderr, "Apple MAC6\n");
			break;
		case AF_COMPRESSION_MPEG1:
			fprintf(stderr, "MPEG 1\n");
			break;
		case AF_COMPRESSION_AWARE_MULTIRATE:
			fprintf(stderr, "Aware MultiRate\n");
			break;
		case AF_COMPRESSION_G726:
			fprintf(stderr, "G726\n");
			break;
		case AF_COMPRESSION_G728:
			fprintf(stderr, "G728\n");
			break;
		case AF_COMPRESSION_DVI_AUDIO:
			fprintf(stderr, "DVI Audio\n");
			break;
		case AF_COMPRESSION_GSM:
			fprintf(stderr, "GSM\n");
			break;
		case AF_COMPRESSION_FS1016:
			fprintf(stderr, "FS1016\n");
			break;
		case AF_COMPRESSION_DEFAULT_MPEG_I:
			fprintf(stderr, "MPEG 1 Layer 1\n");
			break;
		case AF_COMPRESSION_DEFAULT_MPEG_II:
			fprintf(stderr, "MPEG 1 Layer 2\n");
			break;
		case AF_COMPRESSION_DEFAULT_MULTIRATE:
			fprintf(stderr, "MultiRate\n");
			break;
		case AF_COMPRESSION_DEFAULT_LOSSLESS:
			fprintf(stderr, "Lossless\n");
			break;
		default:
			fprintf(stderr, "Unknown\n");
	}
	
	fprintf(stderr, "Data Format    ");
	switch (afInfo->SampleFormat) {
		case AF_SAMPFMT_TWOSCOMP:
			fprintf(stderr, "%d-bit integer (2's complement, %s)\n", afInfo->SampleWidth,
					afInfo->ByteOrder == AF_BYTEORDER_BIGENDIAN ?
				       	"big endian" : "little endian");
			break;
                case AF_SAMPFMT_UNSIGNED:
			fprintf(stderr, "%d-bit integer (unsigned, %s)\n", afInfo->SampleWidth,
					afInfo->ByteOrder == AF_BYTEORDER_BIGENDIAN ?
				       	"big endian" : "little endian");
			break;
		case AF_SAMPFMT_FLOAT:
		case AF_SAMPFMT_DOUBLE:
		default:
			fprintf(stderr, "Unsupported (%d)\n", afInfo->SampleFormat);
			break;
	}
	
	
	fprintf(stderr, "Audio Data     %ld bytes begins at offset %ld (%lx hex)\n",
			afInfo->TrackBytes,
			afInfo->DataOffset,
			afInfo->DataOffset);
	fprintf(stderr, "               %d channel%s, %ld frames\n",
			afInfo->Channels,
			afInfo->Channels > 1 ? "s" : "",
			afInfo->FrameCount);
	
	fprintf(stderr, "Sampling Rate  %.2f Hz\n", afInfo->SampleRate);
	
	fprintf(stderr, "Duration       %.2f seconds\n",
			afInfo->FrameCount /
			afInfo->SampleRate);
}

void
SDestroyAudioFileInfo(SAudioFileInfo *afInfo)
{
	assert(afInfo != NULL);
	
	afCloseFile(afInfo->FileHandle);
	if (afInfo->Path)
		free(afInfo->Path);
	
	free(afInfo);
}
