/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tika.server;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.Duration;
import java.time.Instant;
import org.apache.tika.server.ServerStatus;
import org.apache.tika.server.ServerTimeouts;
import org.apache.tika.server.TaskStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerStatusWatcher
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ServerStatusWatcher.class);
    private final ServerStatus serverStatus;
    private final DataInputStream fromParent;
    private final DataOutputStream toParent;
    private final long maxFiles;
    private final ServerTimeouts serverTimeouts;
    private volatile Instant lastPing = null;

    public ServerStatusWatcher(ServerStatus serverStatus, InputStream inputStream, OutputStream outputStream, long maxFiles, ServerTimeouts serverTimeouts) {
        this.serverStatus = serverStatus;
        this.maxFiles = maxFiles;
        this.serverTimeouts = serverTimeouts;
        this.fromParent = new DataInputStream(inputStream);
        this.toParent = new DataOutputStream(outputStream);
        Thread statusWatcher = new Thread(new StatusWatcher());
        statusWatcher.setDaemon(true);
        statusWatcher.start();
    }

    @Override
    public void run() {
        try {
            this.toParent.writeByte(ServerStatus.STATUS.OPERATING.getByte());
            this.toParent.flush();
        }
        catch (Exception e) {
            LOG.warn("Exception writing startup ping to parent", e);
            this.serverStatus.setStatus(ServerStatus.STATUS.PARENT_EXCEPTION);
            this.shutdown(ServerStatus.STATUS.PARENT_EXCEPTION);
        }
        byte directive = -1;
        while (true) {
            try {
                directive = this.fromParent.readByte();
                this.lastPing = Instant.now();
            }
            catch (Exception e) {
                LOG.warn("Exception reading from parent", e);
                this.serverStatus.setStatus(ServerStatus.STATUS.PARENT_EXCEPTION);
                this.shutdown(ServerStatus.STATUS.PARENT_EXCEPTION);
            }
            if (directive == ServerStatus.DIRECTIVES.PING.getByte()) {
                if (this.serverStatus.getStatus().equals((Object)ServerStatus.STATUS.OPERATING)) {
                    this.checkForHitMaxFiles();
                    this.checkForTaskTimeouts();
                }
                try {
                    this.toParent.writeByte(this.serverStatus.getStatus().getByte());
                    this.toParent.flush();
                }
                catch (Exception e) {
                    LOG.warn("Exception writing to parent", e);
                    this.serverStatus.setStatus(ServerStatus.STATUS.PARENT_EXCEPTION);
                    this.shutdown(ServerStatus.STATUS.PARENT_EXCEPTION);
                }
                continue;
            }
            if (directive == ServerStatus.DIRECTIVES.SHUTDOWN.getByte()) {
                LOG.info("Parent requested shutdown");
                this.serverStatus.setStatus(ServerStatus.STATUS.PARENT_REQUESTED_SHUTDOWN);
                this.shutdown(ServerStatus.STATUS.PARENT_REQUESTED_SHUTDOWN);
                continue;
            }
            if (directive != ServerStatus.DIRECTIVES.PING_ACTIVE_SERVER_TASKS.getByte()) continue;
            try {
                this.toParent.writeInt(this.serverStatus.getTasks().size());
                this.toParent.flush();
                continue;
            }
            catch (Exception e) {
                LOG.warn("Exception writing to parent", e);
                this.serverStatus.setStatus(ServerStatus.STATUS.PARENT_EXCEPTION);
                this.shutdown(ServerStatus.STATUS.PARENT_EXCEPTION);
                continue;
            }
            break;
        }
    }

    private void checkForHitMaxFiles() {
        if (this.maxFiles < 0L) {
            return;
        }
        long filesProcessed = this.serverStatus.getFilesProcessed();
        if (filesProcessed >= this.maxFiles) {
            this.serverStatus.setStatus(ServerStatus.STATUS.HIT_MAX);
        }
    }

    private void checkForTaskTimeouts() {
        Instant now = Instant.now();
        for (TaskStatus status : this.serverStatus.getTasks().values()) {
            long millisElapsed = Duration.between(status.started, now).toMillis();
            if (millisElapsed <= this.serverTimeouts.getTaskTimeoutMillis()) continue;
            this.serverStatus.setStatus(ServerStatus.STATUS.TIMEOUT);
            if (status.fileName.isPresent()) {
                LOG.error("Timeout task {}, millis elapsed {}, file {}", status.task.toString(), Long.toString(millisElapsed), status.fileName.get());
                continue;
            }
            LOG.error("Timeout task {}, millis elapsed {}", (Object)status.task.toString(), (Object)Long.toString(millisElapsed));
        }
    }

    private void shutdown(ServerStatus.STATUS status) {
        LOG.info("Shutting down child process with status: " + status.name());
        System.exit(status.getShutdownCode());
    }

    private class StatusWatcher
    implements Runnable {
        private StatusWatcher() {
        }

        @Override
        public void run() {
            while (true) {
                long elapsed;
                ServerStatus.STATUS currStatus;
                if ((currStatus = ServerStatusWatcher.this.serverStatus.getStatus()) != ServerStatus.STATUS.OPERATING) {
                    LOG.warn("child process observed " + currStatus.name() + " and is shutting down.");
                    ServerStatusWatcher.this.shutdown(currStatus);
                }
                if (ServerStatusWatcher.this.lastPing != null && (elapsed = Duration.between(ServerStatusWatcher.this.lastPing, Instant.now()).toMillis()) > ServerStatusWatcher.this.serverTimeouts.getPingTimeoutMillis()) {
                    ServerStatusWatcher.this.serverStatus.setStatus(ServerStatus.STATUS.PARENT_EXCEPTION);
                    ServerStatusWatcher.this.shutdown(ServerStatus.STATUS.PARENT_EXCEPTION);
                }
                try {
                    Thread.sleep(ServerStatusWatcher.this.serverTimeouts.getPingPulseMillis());
                }
                catch (InterruptedException e) {
                    return;
                }
            }
        }
    }
}

