######################################################################
#
# make_KEIS83.pl
#
# Copyright (c) 2018, 2021 INABA Hitoshi <ina@cpan.org> in a CPAN
######################################################################

use strict;
use FindBin;
use lib $FindBin::Bin;

# KEIS83
require 'KEIS/KEIS83_by_CP932.pl';
require 'CP932/CP932_by_Unicode.pl';
require 'JIS/JISX0208GR_by_CP932.pl';
require 'KEIS/KEIS83_by_Unicode_CultiCoLtd.pl';

if ($0 eq __FILE__) {
    open(DUMP,">$0.dump") || die;
    binmode(DUMP);
    open(DIFF,">$0.diff") || die;
    binmode(DIFF);
}

# generate array to check for duplicate keys
my @KEIS83_by_Unicode_OVERRIDE = (

    # added and removed mapping
    # Jacode4e 2.13.6.18 or newer
    # Jacode4e::RoundTrip 2.13.81.12 or newer

    '20B9F' => '5BD5', # 𠮟 (JIS2004)
    '21274' => '9FA5', # 𡉴 (国土地理協会住所辞書文字)
    '37E2' => '9FA8', # 㟢 (国土地理協会住所辞書文字)
    '4C17' => '9FD0', # 䰗
    '4F03' => '9FA1', # 伃 (国土地理協会住所辞書文字)
    '4F0B' => '9FA2', # 伋 (国土地理協会住所辞書文字)
    '4FE0' => '6EB0', # 俠 (JIS78 on JIS2000)
    '4FF1' => '59FC', # 俱 (JIS2004)
    '50D9' => '5AC0', # 僙
    '5215' => '9FA3', # 刕 (国土地理協会住所辞書文字)
    '525D' => '6EB8', # 剝 (JIS2004)
    '537D' => '5BBF', # 卽 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '541E' => '',     # 吞 (JIS2004)
    '555E' => '6EBC', # 啞 (JIS78 on JIS2000)
    '5653' => '6EBF', # 噓 (JIS2004)
    '5699' => '6EC3', # 嚙 (JIS78 on JIS2000)
    '56CA' => '6EC6', # 囊 (JIS78 on JIS2000)
    '57C7' => '9FA6', # 埇 (国土地理協会住所辞書文字)
    '57DE' => '9FA7', # 埞 (国土地理協会住所辞書文字)
    '5861' => '6EC9', # 塡 (JIS78 on JIS2000)
    '589E' => '5CD1', # 增 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '59F8' => '',     # 姸 (JIS2004)
    '5BEC' => '5DC7', # 寬 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '5C12' => '59C3', # 尒
    '5C5B' => '6ECD', # 屛 (JIS2004)
    '5C62' => '6ECF', # 屢 (JIS78 on JIS2000)
    '5D6D' => '9FA9', # 嵭 (国土地理協会住所辞書文字)
    '5DE2' => '61CF', # 巢 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '5E77' => '',     # 幷 (JIS2004)
    '5FB5' => '5EE4', # 徵 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '5FB7' => '5EE6', # 德 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '625A' => '9FAA', # 扚 (国土地理協会住所辞書文字)
    '63ED' => '5FDF', # 揭 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6414' => '6EDD', # 搔 (JIS78 on JIS2000)
    '6451' => '6EDF', # 摑 (JIS78 on JIS2000)
    '64CA' => '5FF3', # 擊 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6522' => '6EE2', # 攢 (JIS78 on JIS2000)
    '665A' => '60DB', # 晚 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '66C6' => '60F0', # 曆 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '677B' => '9FAB', # 杻 (国土地理協会住所辞書文字)
    '67C0' => '9FAC', # 柀 (国土地理協会住所辞書文字)
    '6935' => '9FAE', # 椵 (国土地理協会住所辞書文字)
    '6980' => '9FAF', # 榀 (国土地理協会住所辞書文字)
    '6992' => '9FB0', # 榒 (国土地理協会住所辞書文字)
    '6A6B' => '62A9', # 橫 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6A9E' => '9FB2', # 檞 (国土地理協会住所辞書文字)
    '6B65' => '62C5', # 步 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6B77' => '5BC7', # 歷 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6BCF' => '62CF', # 每 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6C97' => '9FB4', # 沗 (国土地理協会住所辞書文字)
    '6D89' => '62F0', # 涉 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6DDA' => '62EF', # 淚 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6E34' => '63AC', # 渴 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6E57' => '9FB5', # 湗 (国土地理協会住所辞書文字)
    '6EAB' => '63C1', # 溫 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6EBF' => '9FB6', # 溿 (国土地理協会住所辞書文字)
    '6F51' => '6EF4', # 潑 (JIS78 on JIS2000)
    '6F8D' => '9FB7', # 澍 (国土地理協会住所辞書文字)
    '7006' => '6EF5', # 瀆 (JIS78 on JIS2000)
    '7028' => '63F1', # 瀨 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '710F' => '9FB9', # 焏 (国土地理協会住所辞書文字)
    '7130' => '6EFA', # 焰 (JIS78 on JIS2000)
    '72BE' => '9FBA', # 犾 (国土地理協会住所辞書文字)
    '72C0' => '64C7', # 狀 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '7501' => '65B9', # 甁 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '757E' => '9FBC', # 畾 (国土地理協会住所辞書文字)
    '7626' => '80A3', # 瘦 (JIS2004) (JIS78)
    '784F' => '65F8', # 硏 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '79B1' => '80A6', # 禱 (JIS78 on JIS2000)
    '7A5D' => '9FC0', # 穝 (国土地理協会住所辞書文字)
    '7C1E' => '80AD', # 簞 (JIS78 on JIS2000)
    '7CA0' => '9FC1', # 粠 (国土地理協会住所辞書文字)
    '7CD9' => '9FC2', # 糙 (国土地理協会住所辞書文字)
    '7DA0' => '67EB', # 綠 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '7DD6' => '67F1', # 緖 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '7DE3' => '67F5', # 緣 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '7E61' => '80B5', # 繡 (JIS78 on JIS2000)
    '7E6B' => '80B4', # 繫 (JIS2004) (JIS78)
    '828E' => '9FC4', # 芎 (国土地理協会住所辞書文字)
    '8330' => '9FC5', # 茰 (国土地理協会住所辞書文字)
    '840A' => '80C4', # 萊 (JIS78 on JIS2000)
    '845A' => '9FC6', # 葚 (国土地理協会住所辞書文字)
    '8523' => '80C6', # 蔣 (JIS78 on JIS2000)
    '85AD' => '9FC7', # 薭 (国土地理協会住所辞書文字)
    '85B0' => '69C9', # 薰 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '865B' => '69D2', # 虛 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '87EC' => '80CE', # 蟬 (JIS78 on JIS2000)
    '880B' => '9FC8', # 蠋 (国土地理協会住所辞書文字)
    '881F' => '80D0', # 蠟 (JIS78 on JIS2000)
    '8CF4' => '6ACF', # 賴 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '8EC0' => '80D6', # 軀 (JIS78 on JIS2000)
    '90DE' => '6BC7', # 郞 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '9115' => '6BCD', # 鄕 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '91AC' => '80E4', # 醬 (JIS78 on JIS2000)
    '91B1' => '80E5', # 醱 (JIS78 on JIS2000)
    '92D3' => '9FCA', # 鋓 (国土地理協会住所辞書文字)
    '9304' => '6CC0', # 錄 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '934A' => '6CC9', # 鍊 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '93BA' => '9FCB', # 鎺 (国土地理協会住所辞書文字)
    '958D' => '9FCC', # 閍 (国土地理協会住所辞書文字)
    '9733' => '9FCD', # 霳 (国土地理協会住所辞書文字)
    '973B' => '9FCE', # 霻 (国土地理協会住所辞書文字)
    '974F' => '9FCF', # 靏 (国土地理協会住所辞書文字)
    '9830' => '80EC', # 頰 (JIS78 on JIS2000)
    '985A' => '80ED', # 顚 (JIS78 on JIS2000)
    '9A52' => '80F0', # 驒 (JIS78 on JIS2000)
    '9B66' => '9FD1', # 魦 (国土地理協会住所辞書文字)
    '9B75' => '9FD2', # 魵 (国土地理協会住所辞書文字)
    '9C00' => '9FD3', # 鰀 (国土地理協会住所辞書文字)
    '9C63' => '9FD4', # 鱣 (国土地理協会住所辞書文字)
    '9D1E' => '9FD6', # 鴞 (国土地理協会住所辞書文字)
    '9D62' => '9FD7', # 鵢 (国土地理協会住所辞書文字)
    '9D70' => '9FD8', # 鵰 (国土地理協会住所辞書文字)
    '9D96' => '9FD5', # 鶖 (国土地理協会住所辞書文字)
    '9DD7' => '80F4', # 鷗 (JIS78 on JIS2000)
    '9E7C' => '80F5', # 鹼 (JIS78 on JIS2000)
    '9EB4' => '80F7', # 麴 (JIS78 on JIS2000)
    '9EB5' => '80F8', # 麵 (JIS78 on JIS2000)
    '9EC3' => '6DFE', # 黃 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '9ED1' => '6EA1', # 黑 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F91D' => '62BE', # 欄 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F928' => '5EC2', # 廊 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F929' => '61A7', # 朗 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F936' => '69D3', # 虜 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F9D0' => '6DC0', # 類 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F9DC' => '6DA2', # 隆 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA10' => '5CC7', # 塚 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA16' => '64CE', # 猪 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA19' => '66BB', # 神 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA1A' => '66C2', # 祥 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA1B' => '66D3', # 福 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA22' => '6AB2', # 諸 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA26' => '6BCA', # 都 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA30' => '59E2', # 侮 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA31' => '5ABD', # 僧 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA32' => '5ACE', # 免 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA33' => '5AF9', # 勉 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA34' => '5BA3', # 勤 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA35' => '5BB4', # 卑 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA36' => '5BEF', # 喝 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA37' => '5BF9', # 嘆 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA38' => '5CA2', # 器 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA3A' => '5CD2', # 墨 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA3B' => '5DD7', # 層 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA3D' => '5EF7', # 悔 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA3F' => '5FC1', # 憎 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA40' => '5FC7', # 懲 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA41' => '5FF7', # 敏 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA43' => '60E4', # 暑 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA44' => '61CD', # 梅 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA45' => '62EA', # 海 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA46' => '63B0', # 渚 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA47' => '63CB', # 漢 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA48' => '64AC', # 煮 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA4A' => '64F4', # 琢 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA4B' => '66A4', # 碑 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA4C' => '66B4', # 社 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA4D' => '66B6', # 祉 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA4E' => '66B5', # 祈 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA4F' => '66BD', # 祐 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA50' => '66BC', # 祖 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA51' => '66BA', # 祝 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA52' => '66CF', # 禍 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA53' => '66D1', # 禎 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA54' => '66EB', # 穀 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA55' => '66F7', # 突 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA56' => '67B6', # 節 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA57' => '67F2', # 練 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA59' => '67F8', # 繁 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA5A' => '68A8', # 署 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA5B' => '68BA', # 者 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA5C' => '68C6', # 臭 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA5F' => '69AC', # 著 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA60' => '69E4', # 褐 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA61' => '66C7', # 視 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA62' => '6AB1', # 謁 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA63' => '6AB9', # 謹 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA64' => '6ACE', # 賓 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA65' => '6AD4', # 贈 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA67' => '6AFD', # 逸 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA68' => '6DAB', # 難 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA69' => '6DBA', # 響 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)

    # Jacode4e original mapping

    '2016' => '',     # ‖
    '2212' => '',     # −
    '2225' => 'A1C2', # ∥
    '301C' => 'A1C1', # 〜
    '51DC' => '',     # 凜
    '582F' => 'F4A1', # 堯
    '5C2D' => 'B6C6', # 尭
    '69C7' => 'F4A2', # 槇
    '69D9' => 'CBEA', # 槙
    '7199' => '64B8', # 熙
    '7464' => 'F4A4', # 瑤
    '7476' => 'E0F6', # 瑶
    '9059' => 'F4A3', # 遙
    '9065' => 'CDDA', # 遥
    'FF0D' => 'A1DD', # －
    'FF5E' => '',     # ～
);

my %KEIS83_by_Unicode_OVERRIDE = @KEIS83_by_Unicode_OVERRIDE;
if (scalar(keys %KEIS83_by_Unicode_OVERRIDE) != (scalar(@KEIS83_by_Unicode_OVERRIDE) / 2)) {
    my %key = ();
    while (my($key) = splice(@KEIS83_by_Unicode_OVERRIDE,0,2)) {
        $key{$key}++;
    }
    for my $key (sort keys %key) {
        if ($key{$key} >= 2) {
            print STDERR "[$key], ";
        }
    }
    die "Duplicate hash key(must unique): %KEIS83_by_Unicode_OVERRIDE\n";
}

my %unicode = map { $_ => 1 } (
    keys_of_CP932_by_Unicode(),
    keys_of_KEIS83_by_Unicode_CultiCoLtd(),
    keys %KEIS83_by_Unicode_OVERRIDE,
);

my %KEIS83_by_Unicode = ();
my %done = ();

for my $unicode (sort { (length($a) <=> length($b)) || ($a cmp $b) } keys %unicode) {
    if (exists($KEIS83_by_Unicode_OVERRIDE{$unicode}) and ($KEIS83_by_Unicode_OVERRIDE{$unicode} eq '')) {
        $done{$KEIS83_by_Unicode_OVERRIDE{$unicode}} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s ", $KEIS83_by_Unicode{$unicode}, $unicode, '----', '----', '----', '----', '----';
printf DUMP "\n";
    }
    elsif (($KEIS83_by_Unicode_OVERRIDE{$unicode} ne '') and not $done{$KEIS83_by_Unicode_OVERRIDE{$unicode}}) {
        $done{$KEIS83_by_Unicode{$unicode} = $KEIS83_by_Unicode_OVERRIDE{$unicode}} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s ", $KEIS83_by_Unicode{$unicode}, $unicode, $KEIS83_by_Unicode{$unicode}, '----', '----', '----', '----';
    }
    elsif ((KEIS83_by_CP932(CP932_by_Unicode($unicode)) ne '') and not $done{KEIS83_by_CP932(CP932_by_Unicode($unicode))}) {
        $done{$KEIS83_by_Unicode{$unicode} = KEIS83_by_CP932(CP932_by_Unicode($unicode))} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s ", $KEIS83_by_Unicode{$unicode}, $unicode, '----', $KEIS83_by_Unicode{$unicode}, '----', '----', '----';
    }
    elsif ((KEIS83_by_Unicode_CultiCoLtd($unicode) ne '') and not $done{KEIS83_by_Unicode_CultiCoLtd($unicode)}) {
        if (0) {
        }
        elsif (
            (JISX0208GR_by_CP932(CP932_by_Unicode($unicode)) ne '') and
            (KEIS83_by_Unicode_CultiCoLtd($unicode)          ne '') and
            (JISX0208GR_by_CP932(CP932_by_Unicode($unicode)) eq KEIS83_by_Unicode_CultiCoLtd($unicode)) and
            not $done{JISX0208GR_by_CP932(CP932_by_Unicode($unicode))}
        ) {
            $done{$KEIS83_by_Unicode{$unicode} = JISX0208GR_by_CP932(CP932_by_Unicode($unicode))} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s ", $KEIS83_by_Unicode{$unicode}, $unicode, '----', '----', $KEIS83_by_Unicode{$unicode}, '----', '----';
        }
        elsif (
            (KEIS83_by_Unicode_CultiCoLtd($unicode)          ne '') and
            (JISX0208GR_by_CP932(CP932_by_Unicode($unicode)) eq '') and
            not $done{KEIS83_by_Unicode_CultiCoLtd($unicode)}
        ) {
            $done{$KEIS83_by_Unicode{$unicode} = KEIS83_by_Unicode_CultiCoLtd($unicode)} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s ", $KEIS83_by_Unicode{$unicode}, $unicode, '----', '----', '----', $KEIS83_by_Unicode{$unicode}, '----';
        }
        elsif (
            (JISX0208GR_by_CP932(CP932_by_Unicode($unicode)) ne '') and
            (KEIS83_by_Unicode_CultiCoLtd($unicode)          eq '') and
            not $done{JISX0208GR_by_CP932(CP932_by_Unicode($unicode))}
        ) {
            $done{$KEIS83_by_Unicode{$unicode} = JISX0208GR_by_CP932(CP932_by_Unicode($unicode))} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s ", $KEIS83_by_Unicode{$unicode}, $unicode, '----', '----', '----', '----', $KEIS83_by_Unicode{$unicode};
        }
        elsif (
            (JISX0208GR_by_CP932(CP932_by_Unicode($unicode)) ne '') and
            (KEIS83_by_Unicode_CultiCoLtd($unicode)          ne '') and
            (JISX0208GR_by_CP932(CP932_by_Unicode($unicode)) ne KEIS83_by_Unicode_CultiCoLtd($unicode)) and
            not $done{JISX0208GR_by_CP932(CP932_by_Unicode($unicode))}
        ) {
die sprintf "Unicode=($unicode), CultiCoLtd=(%s) JISX0208GR=(%s)\n", KEIS83_by_Unicode_CultiCoLtd($unicode), JISX0208GR_by_CP932(CP932_by_Unicode($unicode));
        }
    }

    if ((KEIS83_by_Unicode_CultiCoLtd($unicode) ne '') and (KEIS83_by_CP932(CP932_by_Unicode($unicode)) ne '')) {
        if (KEIS83_by_Unicode_CultiCoLtd($unicode) ne KEIS83_by_CP932(CP932_by_Unicode($unicode))) {
die sprintf "Unicode=($unicode), CultiCoLtd=(%s) Handmade=(%s)\n", KEIS83_by_Unicode_CultiCoLtd($unicode), KEIS83_by_CP932(CP932_by_Unicode($unicode));
        }
    }

printf DUMP "\n" if $KEIS83_by_Unicode{$unicode};

    if (
        ($KEIS83_by_Unicode{$unicode} ne '') and
        (KEIS83_by_Unicode_CultiCoLtd($unicode) ne '') and
        ($KEIS83_by_Unicode{$unicode} ne KEIS83_by_Unicode_CultiCoLtd($unicode)) and
    1) {
        printf DIFF ("[%s] (%s) (%s) (%s)\n",
            (CP932_by_Unicode($unicode) ne '') ? pack('H*',CP932_by_Unicode($unicode)) : (' ' x 2),
            $unicode || (' ' x 4),
            $KEIS83_by_Unicode{$unicode} || (' ' x 4),
            KEIS83_by_Unicode_CultiCoLtd($unicode) || (' ' x 4),
        );
    }
}

close(DUMP);
close(DIFF);

# KEIS83 includes JIS78 and JIS2004 fonts as extended character sets, but this software does not use them
for my $unicode (sort { (length($a) <=> length($b)) || ($a cmp $b) } keys %unicode) {

    # JIS2004
    if (($unicode =~ /^(?:

        20B9F | # => 5BD5 # 𠮟
        4FF1  | # => 59FC # 俱
        525D  | # => 6EB8 # 剝
        541E  | # => '',  # 吞
        541E  | # => '',  # 吞
        5653  | # => 6EBF # 噓
        59F8  | # => '',  # 姸
        5C5B  | # => 6ECD # 屛
        5E77  | # => '',  # 幷
        7626  | # => 80A3 # 瘦
        7E6B    # => 80B4 # 繫

    )$/x) and ($KEIS83_by_Unicode{$unicode} ne '')) {
        next;
    }
}

sub KEIS83_by_Unicode {
    my($unicode) = @_;
    return $KEIS83_by_Unicode{$unicode};
}

sub keys_of_KEIS83_by_Unicode {
    return keys %KEIS83_by_Unicode;
}

sub values_of_KEIS83_by_Unicode {
    return values %KEIS83_by_Unicode;
}

1;

__END__
