package Power::Outlet::Shelly;
use strict;
use warnings;
use base qw{Power::Outlet::Common::IP::HTTP::JSON};

our $VERSION='0.37';

=head1 NAME

Power::Outlet::Shelly - Control and query a Shelly GIPO Relay with HTTP REST API

=head1 SYNOPSIS

  my $outlet = Power::Outlet::Shelly->new(host=>"shelly", index=>0);
  print $outlet->query, "\n";
  print $outlet->on, "\n";
  print $outlet->off, "\n";
  print $outlet->switch, "\n";
  print $outlet->cycle, "\n";

=head1 DESCRIPTION

Power::Outlet::Shelly is a package for controlling and querying a relay index on Shelly hardware.

From: L<https://shelly-api-docs.shelly.cloud/>

Commands can be executed via web (HTTP) requests, for example:

  http://<ip>/relay/0?turn=on
  http://<ip>/relay/0?turn=off
  http://<ip>/relay/0?turn=toggle
  http://<ip>/relay/0?timer=5

=head1 USAGE

  use Power::Outlet::Shelly;
  my $relay = Power::Outlet::Shelly->new(host=>"shelly", relay=>"0");
  print $relay->on, "\n";

=head1 CONSTRUCTOR

=head2 new

  my $outlet = Power::Outlet->new(type=>"Shelly", host=>"shelly", relay=>"0");
  my $outlet = Power::Outlet::Shelly->new(host=>"shelly", relay=>"0");

=head1 PROPERTIES

=head2 index

Shelly hardware supports zero or more relay indexes starting at 0.

Default: 0

=cut

sub index {
  my $self         = shift;
  $self->{'index'} = shift if @_;
  $self->{'index'} = $self->_index_default unless defined $self->{'index'};
  return $self->{'index'};
}

sub _index_default {0};

=head2 host

Sets and returns the hostname or IP address.

Default: shelly

=cut

sub _host_default {'shelly'};

=head2 port

Sets and returns the port number.

Default: 80

=cut

sub _port_default {'80'};

=head2 http_path

Sets and returns the http_path.

Default: /relay/{index}

=cut

sub http_path {
  my $self = shift;
  $self->{'http_path'} = shift if @_;
  die("Error: http_path not defined") unless defined $self->{'http_path'};
  return $self->{'http_path'};
}

=head1 METHODS

=head2 name

=cut

#GET /settings/relay/{index} -> $return->{'name'}

sub _name_default {
  my $self = shift;
  return $self->_call('/settings/relay/')->{'name'};
}

=head2 query

Sends an HTTP message to the device to query the current state

=cut

sub query {
  my $self = shift;
  return $self->_call('/relay/')->{'ison'} ? 'ON' : 'OFF';
}

=head2 on

Sends a message to the device to Turn Power ON

=cut

sub on {
  my $self = shift;
  return $self->_call('/relay/', turn => 'on')->{'ison'} ? 'ON' : 'OFF';
}

=head2 off

Sends a message to the device to Turn Power OFF

=cut

sub off {
  my $self = shift;
  return $self->_call('/relay/', turn => 'off')->{'ison'} ? 'ON' : 'OFF';
}

=head2 switch

Sends a message to the device to toggle the power

=cut

sub switch {
  my $self = shift;
  return $self->_call('/relay/', turn => 'toggle')->{'ison'} ? 'ON' : 'OFF';
}

=head2 cycle

Sends messages to the device to Cycle Power (ON-OFF-ON or OFF-ON-OFF).

=cut

sub cycle {
  my $self = shift;
  my $cycle_duration = $self->cycle_duration; #from Power::Outlet::Common
  my $hash = $self->_call('/relay/', timer => $cycle_duration);
  return $hash->{'timer_started'} > 0 ? 'CYCLE' #is this the correct logic?
       : $hash->{'ison'}              ? 'ON'
       : 'OFF';
}

=head2 cycle_duration

Default; 10 seconds (floating point number)

=cut

sub _call {
  my $self  = shift;
  my $path  = shift;
  my %param = @_;
  #http://<ip>/settings/relay/0
  #http://<ip>/relay/0?turn=on
  #http://<ip>/relay/0?timer=10
  $self->http_path($path . $self->index);
  my $url   = $self->url(undef); #isa URI from Power::Outlet::Common::IP::HTTP
  $url->query_form(%param);
  my $hash  = $self->json_request(GET => $url); #isa HASH

  die("Error: Method _call failed to return expected JSON format") unless ref($hash) eq "HASH";
  return $hash;
}

=head1 BUGS

Please log on RT and send an email to the author.

=head1 SUPPORT

DavisNetworks.com supports all Perl applications including this package.

=head1 AUTHOR

  Michael R. Davis
  CPAN ID: MRDVT
  DavisNetworks.com

=head1 COPYRIGHT

Copyright (c) 2020 Michael R. Davis

This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

The full text of the license can be found in the LICENSE file included with this module.

=head1 SEE ALSO

L<https://shelly-api-docs.shelly.cloud/>

=cut

1;
