package Net::Cisco::ACS::DeviceGroup;
use strict;
use Moose;
use Data::Dumper;

BEGIN {
    use Exporter ();
    use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS %actions);
    $VERSION     = '0.02';
    @ISA         = qw(Exporter);
    @EXPORT      = qw();
    @EXPORT_OK   = qw();
    %EXPORT_TAGS = ();
};

    %actions = (	"query" => "/Rest/NetworkDevice/DeviceGroup",
					"create" => "/Rest/NetworkDevice/DeviceGroup",
               		"update" => "/Rest/NetworkDevice/DeviceGroup",
                	"getByName" => "/Rest/NetworkDevice/DeviceGroup/name/",
                	"getById" => "/Rest/NetworkDevice/DeviceGroup/id/",
           ); 

# MOOSE!		   
	   
has 'description' => (
      is  => 'rw',
      isa => 'Any',
  );

has 'id' => (
      is  => 'rw',
      isa => 'Str',
  );

has 'name' => (
	is => 'rw',
	isa => 'Str',
	);

has 'groupType' => (
	is => 'rw',
	isa => 'Str',
	);

# No Moose	

sub toXML
{ my $self = shift;
  my $result = "";
  my $id = $self->id;
  my $description = $self->description || "";
  my $name = $self->name || "";
  my $grouptype = $self->groupType || "Location";
  if ($id) { $result = "   <id>$id</id>\n"; }
  
  $result = <<XML;
<description>$description</description>
<name>$name</name>
<groupType>$grouptype</groupType>
XML

  return $result;
}

sub header
{ my $self = shift;
  my $devices = shift;
  return qq(<?xml version="1.0" encoding="UTF-8" standalone="yes"?><ns1:deviceGroup xmlns:ns1="networkdevice.rest.mgmt.acs.nm.cisco.com">$devices</ns1:deviceGroup>);
}
	
=head1 NAME

Net::Cisco::ACS::DeviceGroup - Access Cisco ACS functionality through REST API - DeviceGroup fields

=head1 SYNOPSIS

	use Net::Cisco::ACS;
	use Net::Cisco::ACS::DeviceGroup;
	
	my $acs = Net::Cisco::ACS->new(hostname => '10.0.0.1', username => 'acsadmin', password => 'testPassword');
	
	my %devicegroups = $acs->devicegroups;
	# Retrieve all device groups from ACS
	# Returns hash with device name / Net::Cisco::ACS::DeviceGroup pairs

	print $acs->devicegroups->{"All Locations"}->toXML;
	# Dump in XML format (used by ACS for API calls)
	
	my $device = $acs->devicegroups("name","All Locations");
	# Faster call to request specific device group information by name

	my $devicegroup = $acs->devicegroups("id","250");
	# Faster call to request specific device group information by ID (assigned by ACS, present in Net::Cisco::ACS::DeviceGroup)

	$devicegroup->id(0); # Required for new device group!
	my $id = $acs->create($devicegroup);
	# Create new device group based on Net::Cisco::ACS::DeviceGroup instance
	# Return value is ID generated by ACS
	print "Record ID is $id" if $id;
	print $Net::Cisco::ACS::ERROR unless $id;
	# $Net::Cisco::ACS::ERROR contains details about failure

	my $id = $acs->update($devicegroup);
	# Update existing device based on Net::Cisco::ACS::DeviceGroup instance
	# Return value is ID generated by ACS
	print "Record ID is $id" if $id;
	print $Net::Cisco::ACS::ERROR unless $id;
	# $Net::Cisco::ACS::ERROR contains details about failure

	$acs->delete($devicegroup);
	# Delete existing device based on Net::Cisco::ACS::DeviceGroup instance
	
=head1 DESCRIPTION

The Net::Cisco::ACS::DeviceGroup class holds all the device group relevant information from Cisco ACS 5.x

=head1 USAGE

All calls are typically handled through an instance of the L<Net::Cisco::ACS> class. L<Net::Cisco::ACS::DeviceGroup> acts as a container for device group related information.

=over 3

=item new

Class constructor. Returns object of Net::Cisco::ACS::DeviceGroup on succes. The following fields can be set / retrieved:

=over 5

=item description 
=item name 
=item id
=item groupType

Formatting rules may be in place & enforced by Cisco ACS.

=back

=over 3

=item description 

The device group account description, typically used for full device group name.

=item name 

The device group name. This is a required value in the constructor but can be redefined afterwards.

=item groupType

This points to the type of Device Group, typically Location or Device Type but can be customized. See also L<Net::Cisco::ACS::Device> C<deviceType>.

=item id

Cisco ACS generates a unique ID for each Device Group record. This field cannot be updated within ACS but is used for reference. Set to 0 when creating a new record or when duplicating an existing device group.

=item toXML

Dump the record in ACS accept XML formatting (without header).

=item header

Generate the correct XML header. Takes output of C<toXML> as argument.

=back

=head1 BUGS

None yet

=head1 SUPPORT

None yet :)

=head1 AUTHOR

    Hendrik Van Belleghem
    CPAN ID: BEATNIK
    hendrik.vanbelleghem@gmail.com

=head1 COPYRIGHT

This program is free software licensed under the...

	The General Public License (GPL)
	Version 2, June 1991

The full text of the license can be found in the
LICENSE file included with this module.


=head1 SEE ALSO

perl(1).

=cut

#################### main pod documentation end ###################
__PACKAGE__->meta->make_immutable();

1;
# The preceding line will help the module return a true value

