#!/usr/bin/env perl

# You must configure bk_connect.pl to connect with
# your database.
#
# You must also set up the bk_code table schema
# and load at least the core bk_code records before
# this will run.  The native language must be English.

use Test;
BEGIN { plan tests => 25 };

use strict;
use warnings;
use lib 'blib/lib';
use lib '../blib/lib';
use lib '..';
use DBI;
use DBIx::BabelKit;
require("bk_connect.pl");
ok(1);

my $dbh = bk_connect();
ok(1);

my $bkh = DBIx::BabelKit->new($dbh);
ok(1);

# Clean up old data.
$bkh->remove('regression',  1);
$bkh->slave('regression', '2', undef);
$bkh->put('regression', 'en', 3, undef);
ok(1);

# Test basic puts.
$bkh->put('regression', 'en',  1, 'Monday');
$bkh->put('regression', 'en', '2', 'Tuesday', '');
$bkh->put('regression', 'fr', '2', 'mardi', 3, 'd');
ok(1);

# And gets.
ok( join(',', $bkh->get('regression', 'en', '1')), 'Monday,1,' );
ok( join(',', $bkh->get('regression', 'en',  2 )), 'Tuesday,2,' );
ok( join(',', $bkh->get('regression', 'fr',  2 )), 'mardi,0,' );

# Simple select.
my $expect;
$expect = '<select name="regression">
<option value="" selected>Coffee Date?
<option value="1">Monday
<option value="2">Mardi
</select>
';
ok( $bkh->select('regression', 'fr',
                 select_prompt => 'Coffee Date?'
                 ), $expect );

# Slave & desc methods.
$bkh->slave('regression', '3', 'wednes day');
ok( join(',', $bkh->get('regression', 'en', 3)), 'wednes day,3,' );
ok( $bkh->desc('regression', 'fr', 3),           'wednes day' );
ok( $bkh->ucfirst('regression', 'fr', 3),        'Wednes day' );
ok( $bkh->ucwords('regression', 'fr', 3),        'Wednes Day' );
ok( $bkh->desc('regression', 'fr', 2),           'mardi' );
ok( $bkh->render('regression', 'fr', 3),         'wednes day' );
ok( $bkh->data('regression', 'en', 3),           'wednes day' );
ok( $bkh->data('regression', 'fr', 3),           '' );
ok( $bkh->param('regression', 3),                'wednes day' );
$bkh->slave('regression', '3', 'Wednesday');
ok( join(',', $bkh->get('regression', 'en', 3)), 'Wednesday,3,' );

# Select options.
$expect = '<select name="regression_test" onchange="submit()">
<option value="">(None)
<option value="1">Monday
<option value="2" selected>Mardi
</select>
';
ok( $bkh->select('regression', 'fr',
                var_name     => 'regression_test',
                value        => '2',
                subset       => [ 1, '2' ],
                options      => 'onchange="submit()"',
                blank_prompt => '(None)'
                ), $expect );

# Radiobox options.
$expect = '<input type="radio" name="rt" onchange="submit()" value="">(None)<br>
<input type="radio" name="rt" onchange="submit()" value="1">Monday<br>
<input type="radio" name="rt" onchange="submit()" value="2" checked>Mardi';
ok( $bkh->radio('regression', 'fr',
                var_name     => 'rt',
                default      => '2',
                subset       => [ 1, '2' ],
                options      => 'onchange="submit()"',
                blank_prompt => '(None)'
                ), $expect);

# Select multiple options.
$expect= '<select multiple name="reg_test[]" onchange="submit()" size="10">
<option value="1">Monday
<option value="2" selected>Mardi
<option value="3" selected>Wednesday
</select>
';
ok( $bkh->multiple('regression', 'fr',
                var_name => 'reg_test',
                value    => [ '2', 3 ],
                subset   => [ 1, '2', 3 ],
                options  => 'onchange="submit()"',
                size     => 10
                ), $expect);

# Checkbox options.
$expect = '<input type="checkbox" name="checkbox_test[]" onchange="submit()" value="1" checked>Monday<br>
<input type="checkbox" name="checkbox_test[]" onchange="submit()" value="2">Mardi<br>
<input type="checkbox" name="checkbox_test[]" onchange="submit()" value="3" checked>Wednesday';
ok( $bkh->checkbox('regression', 'fr',
                var_name => 'checkbox_test',
                value    => [ '1', 3 ],
                subset   => [ 1, '2', 3 ],
                options  => 'onchange="submit()"'
                ), $expect);

# Clean up the test data.
$bkh->remove('regression',  1);
$bkh->slave('regression', '2', '');
$bkh->put('regression', 'en', 3, '');
ok(1);

my $rows = $bkh->full_set('regression', 'fr');
ok(@$rows, 0);

